# Copyright (c) 2013, 2016, Oracle and/or its affiliates. All rights reserved.
#
#   NAME
#      oraons.pm
#
#   DESCRIPTION
#      This module contains creation of default wallets for ONS.
#
#   NOTES
#      <other useful comments, qualifications, etc.>
#
#   MODIFIED   (MM/DD/YY)
#      akhaladk 03/18/16 - bug 22929413
#      akhaladk 09/17/15 - Ons wallet config

package oraons;

use strict;
#use warnings;
use English;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use Env qw(NLS_LANG);


# root script modules
use crsutils;
use Exporter;

use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);


my @exp_func = qw( verify_ons_dirs create_ons_wallets );

push @EXPORT, @exp_func;


###---------------------------------------------------------
#### Define and verify Ons local/cluster-wide ons directories 
#### This sub MUST be called before any ons setup handling takes place.
#
# @returns SUCCESS or FAILED
#
sub verify_ons_dirs
{
    my $orauser    = $CFG->params('ORACLE_OWNER');
    my $oragroup   = $CFG->params('ORA_DBA_GROUP');
    my $orahost    = tolower_host(); # HOST is/may not set in params object ?!

    trace("user = " . $orauser);
    trace("oragroup = " . $oragroup);

    $CFG->params('ONSWALLETDIR', catdir($CFG->params('ORACLE_BASE'), "crsdata",
                                         $orahost, "onswallet"));
    trace("onswall dir = " . $CFG->params('ONSWALLETDIR'));

    trace("Verifying ONS wallet directories ... ");
    trace("Checking if " . $CFG->params('ONSWALLETDIR') . " exists.");

    check_dir($CFG->params('ONSWALLETDIR')) or return FAILED;

    trace("ONS wallet directories verified. ");

    return SUCCESS;
}

####---------------------------------------------------------
#### Create Ons wallet(s)
# 
# @returns SUCCESS or $FAILURE
#
sub create_ons_wallets
{
    my $orauser      = $CFG->params('ORACLE_OWNER');
    my $oragroup     = $CFG->params('ORA_DBA_GROUP');
    my $crshome      = $CFG->params('ORACLE_HOME');
    my $host         = $CFG->HOST;
    my $status       = SUCCESS;
    my $rc           = 0;
    my @capout       = ();
    my $cmd;
    
    my @program;

    my $E_ORAPKI    = catfile( $crshome, 'bin', 'orapki' );
    my $WALLET_NAME = "ONS";
    my $CN          = '"CN=ons_ssl,C=US"';
    my $KEYSIZE     = '2048';
    my $VALIDITY    = '3650';

    my $onsWalletDir = $CFG->params('ONSWALLETDIR');

    trace("Wallet dir is = " . $onsWalletDir);
    
    my $onsWallet = catdir($onsWalletDir, $WALLET_NAME);
    # Remove existing wallet 
    if (SUCCESS == check_dir($onsWallet))
    {
        $status &= rmtree($onsWallet);
        trace("Removed existing wallets. status = " . $status);
    }
    else
    {
        $status = SUCCESS;
    }
    if (SUCCESS == $status) 
    {
        trace("Creating ons wallet: " . $onsWallet);
        @program = ( $E_ORAPKI, 'wallet', 'create', 
                     '-wallet', "\"$onsWallet\"",
                     '-auto_login_only' );
        trace( join(' ', @program) );
        $rc = run_as_user2($orauser, \@capout, @program );
        if (0 != $rc) {
           trace("Failed to create a wallet for Oracle Cluster ONS. ".
                 "orapki rc=$rc" );
           print_error(50, $rc);

           return FAILED;
        }
    }
    if (SUCCESS == $status) 
    {
        trace("Adding certificate to ONS wallet: ");
        @program = ( $E_ORAPKI, 'wallet', 'add', 
                     '-wallet', "\"$onsWallet\"",
                     '-auto_login_only',
                     '-dn ', $CN,
                     '-keysize', $KEYSIZE,
                     '-self_signed',
                     '-validity', $VALIDITY);

        trace( join(' ', @program) );
        $rc = run_as_user2( $orauser, \@capout, @program );
        if (0 != $rc) {
           trace("Failed to add cert to a wallet for Oracle Cluster ONS. ".
                 "orapki rc=$rc" );
           print_error(51, $rc);
           return FAILED;
        }
    }

    # Stuff the actual/final wallet path so clsCred can import it    
    $CFG->params('ONSWALLETDIR', catfile($onsWallet, "cwallet.sso"));
    trace ("ONS wallet set " . $CFG->params('ONSWALLETDIR'));

    return SUCCESS;
}

1;
