# 
# $Header: has/install/crsconfig/oraolr.pm /main/9 2015/11/27 18:36:27 sbezawad Exp $
#
# oraolr.pm
#
# Copyright (c) 2014, 2015, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      oraolr.pm - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      Required variables from the config objecj $CFG:
#      $CFG->OLR_LOCATION
#      $CFG->SIHA
#      $CFG->DEBUG
#      $CFG->ORA_CRS_HOME
#
#    EXPORT METHODS
#      validate_olrconfig - Creates local olr config file if it does not exists
#      checkOLR           - Check if OLR file exists and is configured
#
#      Note: We should keep the export methods minimal in each component class.
#            To many methods exported from the component class would jeopardize
#            the root script framework.
#            We should try best to implement the interfaces defined in the base
#            class oraClusterwareComp, instead of randomly adding new methods
#            exported to public.
#
#    MODIFIED   (MM/DD/YY)
#    sbezawad   11/17/15 - Bug 21919798: Add cluster properties to clscfg
#    luoli      07/28/15 - Add -y -z -h to CLSCFG command
#    muhe       04/14/15 - Add interface functions body for upgrade and patch
#    xyuan      04/09/15 - Address noncompliance with the base class
#    muhe       03/24/15 - Fix bug 20725601
#    luoli      12/22/14 - rsc modeling for downgrade/deconfig
#    sbezawad   11/26/14 - Bug 20019354: Migrate OCR and OLR to new framework
#    xyuan      09/17/14 - Incorporate review comments
#    muhe       07/17/14 - Creation
# 

package oraClusterwareComp::oraolr;

use parent 'oraClusterwareComp';

use strict;
use English;
use Carp;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;

use crsutils;
use s_crsutils;
use oraocr;
use s_oraolr;

# Steps to configure OLR
use constant UPDATE_OLRLOC  => 'olr_ConfigCurrentNode_step_1';
use constant CONFIGURE_OLR  => 'olr_ConfigCurrentNode_step_2';
use constant UPGRADE_OLR    => 'olr_UpgradeCurrentNode_step_2';

sub new
{
  my $class = shift;
  # Pass the component name into the constructor 
  my $componentName = @_;

  my $self = $class->SUPER::new(@_);
  $self->_initialize();

  return $self;
}

sub _initialize
{
  my $self = shift;

  my $compName = $self->compName;
  trace("Perform initialization tasks before configuring $compName");  
}

# Interface to be implemented
#
# Is the component supported based on platform and user input
sub isSupported
{ 
  # Supported on all platforms
  return TRUE;
}

# Which component does this depend on
sub dependsOn
{
  # No dependency
  return undef;
}

# Has the component already been configured
sub isConfigured
{
  #TODO
  return FALSE;
}

#
# Methods for install
#

# Configure action on first node
sub configureFirstNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to configure $compName ".
        "on the first node");

  configSteps($self, $stepIndicator);
  return SUCCESS;
}

# Configure action on other nodes than the first node
sub configureNonFirstNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to configure $compName ".
        "on the non-first node");

  configSteps($self, $stepIndicator);
  return SUCCESS;
}

# Configure action for SIHA
sub configureSIHA
{
  my $self                = shift;
  my $oraocr              = $CFG->compOCR;
  my $local_config_exists = FALSE;

  $local_config_exists = $oraocr->localonlyOCR_exists();

   # delete olr file if exists
   if (! isCkptSuccess("ROOTCRS_OLR") && (-e $CFG->OLR_LOCATION) )
   {
     if ($CFG->DEBUG) { trace ("unlink " . $CFG->OLR_LOCATION); }
     unlink ($CFG->OLR_LOCATION) || print_error(102, $CFG->OLR_LOCATION, $!);
   }

  $self->validate_olrconfig($CFG->OLR_LOCATION,
                     $CFG->ORA_CRS_HOME) || die(dieformat(292));

  if (! $local_config_exists)
  {
    $oraocr->createLocalOnlyOCR();
  }

  configureOLR_runocrconfig();
  configureOLR_runclscfg();

  return SUCCESS;
}

# Configure action on first node after the configured stack
# has been started
sub postConfigFirstNode
{
  backupOLR();
}

# Configure action on other nodes than the first node after
# the configured stack has been started
sub postConfigNonFirstNode
{
  backupOLR();
}

# Methods for patch
#

# The method for pre checks related to OLR component for patching 
sub prePatchCheck
{
  return SUCCESS;
}

# Methods for upgrade
#

# The method for global checks related to OLR component before upgrade
# This method is called only on the first node when the stack is up.
sub preUpgradeCheck
{
  return SUCCESS;
}

# The method for local checks related to OLR component before upgrading the first node
sub upgradeCheckFirstNode
{
  return SUCCESS;
}

# The method for local checks related to OLR component before upgrading the middle node
sub upgradeCheckMiddleNode
{
  return SUCCESS;
}

# The method for local checks related to OLR component before upgrading the last node
sub upgradeCheckLastNode
{
  return SUCCESS;
}

# Upgrade action on first node
sub upgradeFirstNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to upgrade $compName ".
        "on the first node");

  upgradeSteps($self, $stepIndicator);
  return SUCCESS;
}

# Upgrade action on every node other than first and last node
sub upgradeMiddleNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to upgrade $compName ".
        "on the middle node");

  upgradeSteps($self, $stepIndicator);
  return SUCCESS;
}

# Upgrade action on last node
sub upgradeLastNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to upgrade $compName ".
        "on the last node");

  upgradeSteps($self, $stepIndicator);
  return SUCCESS;
}

# Upgrade action for SIHA
sub upgradeSIHA
{
  my $self                = shift;
  my $oraocr              = $CFG->compOCR;

  $self->validate_olrconfig($CFG->OLR_LOCATION,
                     $CFG->ORA_CRS_HOME) || die(dieformat(292));

  $oraocr->createLocalOnlyOCR();

  chdir(catdir($CFG->ORA_CRS_HOME, 'log'));
  configureOLR_runocrconfig();
  upgradeOLR_runclscfg();

  return SUCCESS;
}

# Upgrade action on the first node after the higher version stack
# has been started
sub postUpgradeFirstNode
{
  backupOLR();
}

# Upgrade action on every node other than first and last node
# after the higher version stack has been started 
sub postUpgradeMiddleNode
{
  backupOLR();
}

# Upgrade action on the last node after the higher version stack
# has been started
sub postUpgradeLastNode
{
  backupOLR();
}

# Downgrade action on nodes other than the last node
sub downgradeNonLastNode
{
  # create the old OLR
  trace("Create olr.loc for the downgraded home");
  s_createOldOLR(); 
}

# Downgrade action on the last node
sub downgradeLastNode
{
  # same operations as nodes than the last node
  my $self = shift;
  $self->downgradeNonLastNode();
}

# Whether or not the system reboot is needed after configuration
sub rebootRequired { return FALSE; }

#
# Export functions specific to oraocr class.
#

sub validate_olrconfig
#------------------------------------------------------------------------------
# Function: Creates local olr config file if it does not exists.
#           It also validates/sets up OLR config if does not exist
# Args    : [0] Complete path of OLR location
#           [1] CRS Home
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $self         = shift;
  my $olrlocation  = shift;
  my $crshome      = shift;

  if (!$olrlocation)
  {
    print_error(9);
    return FALSE;
  }

  if (!(-f $olrlocation))
  {
    create_empty_olrconfig($olrlocation) or return FALSE;
  }
  trace ("OLR location = " . $olrlocation);

  if (!$crshome)
  {
    print_error(4);
    return FAILED;
  }

  if (!(-d $crshome))
  {
    print_error(5, $crshome);
  }
  trace ("Oracle CRS Home = " . $crshome);

  # OSD to validate OLR config
  my $status = s_validate_olrconfig ($olrlocation, $crshome);

  return $status;
}

sub checkOLR
#------------------------------------------------------------------------------
# Function: Check if OLR file exists and is configured.
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $self    = shift;
  my $homeDir = shift;

  my $crshome = ($homeDir) ? $homeDir : ($CFG->params('ORACLE_HOME'));
  my $cmd = catfile($crshome, 'bin', 'ocrcheck');
  my @out = system_cmd_capture($cmd, "-local", "-debug");
  my $rc  = shift @out;
  my $check_passed = FALSE;

  trace("checkOLR rc=$rc");
  my @cmdout1 = grep(/(^PROTL-708)/, @out);
  my @cmdout2 = grep(/(^PROTL-715)/, @out);
  my @cmdout3 = grep(/(^PROTL-721)/, @out);

  if ($rc == 0 && scalar(@cmdout1) == 0 &&
      scalar(@cmdout2) == 0 && scalar(@cmdout3) == 0)
  {
    $check_passed = TRUE;
    trace ("OLR check: passed");
  }
  else
  {
    trace ("OLR check: failed");
  }

  return $check_passed;
}

#
# Private methods
#

sub configSteps
{
  my $self          = $_[0];
  my $stepIndicator = $_[1];

  if (UPDATE_OLRLOC eq $stepIndicator)
  {
    # Delete olr file if exists
    if (-e $CFG->OLR_LOCATION)
    {
      trace("unlink " . $CFG->OLR_LOCATION);
      unlink ($CFG->OLR_LOCATION) || print_error(102, $CFG->OLR_LOCATION, $!);
    }

    $self->validate_olrconfig($CFG->OLR_LOCATION,
                       $CFG->ORA_CRS_HOME) || die(dieformat(292))
  }
  elsif (CONFIGURE_OLR eq $stepIndicator)
  {
    configureOLR_runocrconfig();
    configureOLR_runclscfg();
  } 
  else
  {
    croak "Step indicator out of bounds";
  }
}   

sub upgradeSteps 
{
  my $self          = $_[0];
  my $stepIndicator = $_[1];

  if (UPDATE_OLRLOC eq $stepIndicator)
  {
    $self->validate_olrconfig($CFG->OLR_LOCATION,
                       $CFG->ORA_CRS_HOME) || die(dieformat(292))
  }
  elsif (UPGRADE_OLR eq $stepIndicator)
  {
    configureOLR_runocrconfig();
    upgradeOLR_runclscfg();
  } 
  else
  {
    croak "Step indicator out of bounds";
  }
}   

sub configureOLR_runocrconfig
#------------------------------------------------------------------------------
# Function: Executes 'ocrconfig -local -upgrade' command to create or upgrade
#           OLR
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $ocrconfig = catfile($CFG->ORA_CRS_HOME, "bin", "ocrconfig");
  my $ckptName  = "ROOTCRS_OLR";
  my $status    = FALSE;

  trace ("Creating or upgrading Oracle Local Registry (OLR)");
  if ($CFG->SIHA)
  {
    $status = run_as_user($CFG->params('ORACLE_OWNER'),
                          "$ocrconfig -local -upgrade");
  }
  else
  {
    my @cmd = ($ocrconfig, '-local', '-upgrade',
               $CFG->params('ORACLE_OWNER'),
               $CFG->params('ORA_DBA_GROUP'));
    $status = system_cmd(@cmd);
  }

  if (0 == $status)
  {
    trace ("OLR successfully created or upgraded");
  }
  else
  {
    trace("$ocrconfig -local -upgrade failed with error: $status");
    writeCkpt($ckptName, CKPTFAIL);
    die(dieformat(169));
  }
  return TRUE;
}

sub configureOLR_runclscfg
#------------------------------------------------------------------------------
# Function: Executes 'clscfg -localadd' command to create required keys in OLR
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $clscfg   = catfile($CFG->ORA_CRS_HOME, 'bin', 'clscfg');
  my $ckptName = "ROOTCRS_OLR";
  my $status   = FALSE;
  my @out      = ();

  # create keys in OLR

  my $cmd = "$clscfg -localadd";
  if (defined($CFG->clscfg_guid_arg_z))
  {
    $cmd = $cmd . $CFG->clscfg_guid_arg_z;
  }
 
  if (defined($CFG->clscfg_site2guid_arg_y))
  { 
    $cmd = $cmd . " " . $CFG->clscfg_site2guid_arg_y;
  }

  if (defined($CFG->clscfg_clsprop_olr_arg_p))
  {
    $cmd = $cmd . " " . $CFG->clscfg_clsprop_olr_arg_p;
  }

  if ($CFG->SIHA)
  {
    $status = run_as_user($CFG->params('ORACLE_OWNER'), $cmd);
  }
  else
  {
    @out = system_cmd_capture($cmd);
    $status = shift @out;
  }

  if (0 == $status)
  {
    trace ("Keys created in the OLR successfully");
    writeCkpt($ckptName, CKPTSUC);
    $CFG->wipCkptName("ROOTCRS_STACK");
  }
  else
  {
    error ("Failed to create keys in the OLR, rc = $status, Message:\n ",
           "@out \n");
    writeCkpt($ckptName, CKPTFAIL);
    die(dieformat(188));
  }
  return TRUE;
}

sub upgradeOLR_runclscfg
#------------------------------------------------------------------------------
# Function: Executes 'clscfg -localupgrade' command to create required keys in
#           OLR
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $CLSCFGBIN    = catfile ($CFG->ORA_CRS_HOME, "bin", "clscfg");
  my $status       = FALSE;
  my @out          = ();

  # upgrade keys in OLR
  
  my $cmd = "$CLSCFGBIN -localupgrade";
  if (defined($CFG->clscfg_guid_arg_z))
  {
    $cmd = $cmd . $CFG->clscfg_guid_arg_z;
  }

  if (defined($CFG->clscfg_site2guid_arg_y))
  {
    $cmd = $cmd . " " . $CFG->clscfg_site2guid_arg_y;
  }

  if (defined($CFG->clscfg_clsprop_olr_arg_p))
  {
    $cmd = $cmd . " " . $CFG->clscfg_clsprop_olr_arg_p;
  }

  if ($CFG->SIHA)
  {
    $status = run_as_user($CFG->params('ORACLE_OWNER'), $cmd);
  }
  else
  {
    @out = system_cmd_capture($cmd);
    $status = shift @out;
  }

  if (0 == $status)
  {
    trace ("Keys created in the OLR successfully");
  }
  else
  {
    trace("Failed to create keys in the OLR, rc = $status, Message:\n ".
          "@out \n");
    die(dieformat(188));
  }

  return TRUE;
}

sub create_empty_olrconfig
#------------------------------------------------------------------------------
# Function: Create an empty olrconfig file and reset permission
# Args    : [0] Complete path of OLR location
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $olrlocation  = shift;

  if ($CFG->DEBUG)
  {
    trace ("create $olrlocation");
  }

  open (FILEHDL, ">$olrlocation") or return FALSE;
  close (FILEHDL);

  if ($CFG->SIHA)
  {
    s_set_ownergroup ($CFG->params('ORACLE_OWNER'),
                      $CFG->params('ORA_DBA_GROUP'), $olrlocation)
                     or die(dieformat(152, $olrlocation));
    s_set_perms ("0600", $olrlocation)
                or die(dieformat(153, $olrlocation));
  }
  else
  {
    s_set_ownergroup ($CFG->SUPERUSER, $CFG->params('ORA_DBA_GROUP'),
                      $olrlocation)
                     or die(dieformat(152, $olrlocation));
    s_set_perms ("0600", $olrlocation) 
                or die(dieformat(153, $olrlocation));
  }

  return TRUE;
}

sub backupOLR
#------------------------------------------------------------------------------
# Function: Backup OLR
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $ocrconfig = catfile($CFG->params('ORACLE_HOME'), 'bin', 'ocrconfig');
  my $cmd       = "$ocrconfig -local -manualbackup";
  my @out       = system_cmd_capture($cmd);
  my $status    = shift @out;

  if ($status == 0)
  {
    trace ("Successfully generated OLR backup");
  }
  else
  {
    trace ("Failed to generate OLR backup");
    return FALSE;
  }
  return TRUE;
}

1;
