# 
# $Header: rdbms/admin/catcon.pm /st_rdbms_12.2.0.1.0/17 2017/01/23 09:26:06 akruglik Exp $
#
# catcon.pm
# 
# Copyright (c) 2011, 2017, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      catcon.pm - CONtainer-aware Perl Module for creating/upgrading CATalogs
#
#    DESCRIPTION
#      This module defines subroutines which can be used to execute one or 
#      more SQL statements or a SQL*Plus script in 
#      - a non-Consolidated Database, 
#      - all Containers of a Consolidated Database, or 
#      - a specified Container of a Consolidated Database
#
#    NOTES
#      Subroutines which handle invocation of SQL*Plus scripts may be invoked 
#      from other scripts (e.g. catctl.pl) or by invoking catcon.pl directly.  
#      Subroutines that execute one or more SQL statements may only be 
#      invoked from other Perl scripts.
#
#    MODIFIED   (MM/DD/YY)
#    akruglik    01/18/17 - Backport akruglik_bug-25404458 from main
#    akruglik    01/17/17 - Bug 25404458: avoid references to $CDBorFedRoot
#                           unless it is defined
#    akruglik    01/10/17 - Backport akruglik_bug25363697_main from main
#    akruglik    01/09/17 - Bug 25366291: do not report an error if
#                           gen_inst_conn_strings fetches no rows; force all 
#                           processing to occur on the default instance
#    akruglik    01/09/17 - Bug 25363697: avoid sending diagnostics statements
#                           to sqlplus process immediately after executing
#                           catshutdown.sql even if it was executed by a
#                           separate call to catconExec
#    akruglik    01/06/17 - Backport akruglik_bug25315864_main from main
#    akruglik    01/04/17 - Bug 25315864: Instead of trying to set ORACLE_SID,
#                           use connect strings
#    akruglik    12/21/16 - Backport akruglik_bug-25259127 from main
#    akruglik    12/20/16 - Bug 25259127: do not quote name of the errorlogging
#                           table since it may contain owner name
#    akruglik    12/19/16 - Backport akruglik_bug-25243199 from main
#    akruglik    12/16/16 - Bug 25243199: temporarily disable --all_instances
#    akruglik    11/29/16 - Bug 20193612: if running on a RAC database, make
#                           use of multiple instances when processing PDBs
#    akruglik    11/29/16 - Bug 25132308: 12.1.0.1 does not support specifying #                           FORCE in ALTER PDB CLOSE, so we will avoid using 
#                           FORCE (or INSTANCES clause) when closing PDBs in 
#                           a 12.1 CDB
#    akruglik    11/29/16 - Backport akruglik_bug-25089301 from main
#    akruglik    11/28/16 - Bug 25117295: do not specify FORCE if connected to
#                           12.1 CDB and closing a PDB on multiple instances
#    akruglik    11/28/16 - Backport akruglik_bug-25086870 from main
#    akruglik    11/10/16 - Backport akruglik_bug-25061922 from main
#    akruglik    11/09/16 - Backport akruglik_cloud_fixup_main from main
#    akruglik    11/23/16 - Bug 25086870: account for maximum number of
#                           sessions and the number of existing sessions when
#                           computing number of sqlplus processes to spawn
#    akruglik    11/16/16 - Bug 25089301: use gv$pdbs rather than x$con in
#                           curr_pdb_mode_info
#    akruglik    11/09/16 - Bug 25061922: due to an apparent bug in 12.1 code
#                           processing ALTER PDB OPEN/CLOSE
#                           INSTANCES=(inst-list), we cannot use this syntax if
#                           working on a 12.1 DBMS
#    akruglik    11/04/16 - XbranchMerge akruglik_cloud_fixup from
#                           st_rdbms_12.2.0.1.0cloud
#    akruglik    10/27/16 - Backport akruglik_force_pdb_open_mode from main
#    akruglik    09/12/16 - Add support for --force_pdb_open
#    akruglik    09/06/16 - LRG 19650060: Remove invalid Signal handler + 
#                           ignore SIGPIPE if catcon needs to exit because 
#                           some SQL*Plus process died.
#    hohung      08/08/16 - Bug24385625: Look for catcon_kill_sess_gen.sql in
#                           script directory
#    akruglik    07/29/16 - Modify caller of oenSpoolFile to deal with cases
#                           where the spool file was not available
#    akruglik    07/18/16 - LRG 19633516: allow caller to determine whether
#                           death of a spawned sqlplus process should cause
#                           catcon to die
#    akruglik    07/14/16 - Bug 23711335: modify code constructing kill session
#                           script to avoid delays due to concurrent file
#                           access
#    akruglik    06/23/16 - Bug 23614674: verify that a file exists before
#                           attempting to remove it
#    akruglik    06/21/16 - Bug 23292787: modify catconExec to handle cases
#                           where -c or -C and -r was specified
#    akruglik    06/06/16 - Bug 22887047: if a SQLPlus process dies, start a
#                           new one in its place
#    akruglik    05/25/16 - Bug 23326538: use syntax acceptable to Perl 5.6.1
#                           to declare constants
#    akruglik    05/09/16 - Bug 23106360: do not try to open PDB$SEED in READ
#                           WRITE mode if the CDB is open READ ONLY
#    sursridh    04/13/16 - Bug 23020062: Add option to disable pdb lockdown.
#    akruglik    02/10/16 - (22132084): modify get_log_file_base_path to use
#                           cwd if the caller has not supplied LogDir
#    jerrede     01/26/16 - Add Parsing Comment for Andre
#    akruglik    01/08/16 - Bug 22330680: modify exec_DB_script to not assume 
#                           that lines containing a marker start with it
#    akruglik    11/13/15 - (22203085): do not display values of hidden
#                           arguments in diagnostic output
#    akruglik    11/09/15 - Define catconSqlplus
#    jerrede     10/10/15 - Bug 21446778 when catconForce is used don't 
#                           validatate pdb when all pdbs are not opened. 
#                           Functionality needed for upgrade when opening 
#                           PDBs in parallel.
#    juilin      09/23/15 - bug-21485248: replace federation with application
#    frealvar    08/11/15 - bug21274752: message which warns when a component
#                           is not in the database was moved to catctl.pl
#    akruglik    07/16/15 - Bug 21447540: in validate_script_path, if the file
#                           could not be found and we were told to ignore it,
#                           skip other file checks
#    akruglik    06/12/15 - Bug 21202842: avoid displaying passwords
#    akruglik    05/06/15 - Bug 20782683: in validate_con_names, distinguish
#                           case where some specified PDBs were not open from
#                           the case where all PDBs were excluded
#    akruglik    04/29/15 - Bug 20969508: add a flag to tell catcon to not
#                           report error if certain types of violations are
#                           encountered
#    akruglik    04/17/15 - Bug 20905400: modify exec_DB_script to return log
#                           of the output
#    raeburns    03/06/15 - Add query to catconExec to skip running scripts
#    jerrede     02/23/15 - Add no directory prepend for full Windows paths
#    jerrede     02/23/15 - Fix Rae's change to not append the src directory
#                           to a file.  This fails on Windows.
#    akruglik    02/02/15 - Bug 20307059: ensure that container name and its
#                           mode do not get split across multiple lines
#    raeburns    10/22/14 - allow full path names even with -d directory
#    jerrede     10/15/14 - Read Only Oracle Homes
#    akruglik    09/04/14 - Bug 19525627: change API of catconQuery to allow
#                           specification of the PDB in which to run the query
#    akruglik    09/03/14 - Bug 19525987: use upgrade_priority to order
#                           Containers
#    akruglik    07/23/14 - Project 47234: add support for -F flag which will 
#                           cause catcon to run over all Containers in a 
#                           Federation name of whose Root is passed with
#                           the flag
#    jerrede     06/13/14 - Bug 18969473.
#                           Workaround Windows Perl problem:
#                           open2: Can not call method "close" on an undefined
#                           value at /perl/lib/IPC/Open3.pm line 415. This
#                           happens when specifing a container list of 1 item.
#                           catcon.pm dies in Open2 call via catconInit,
#                           get_instance_status and exec_DB_script.
#    akruglik    06/12/14 - Bug 18960833: get rid of excessive messages which
#                           were introduced by a fix for bug 18745291
#    akruglik    05/19/14 - add support for -z flag used to supply EZConnect
#                           strings corresponding to RAC instances which should
#                           be used to run scripts
#    akruglik    05/01/14 - Bug 18672126: having unlinked a file in 
#                           sureunlink, give it some time to disappear
#    akruglik    05/12/14 - Bug 18745291: produce progress messages
#    akruglik    04/17/14 - Bug 18606911: when deciding whether PDB$SEED is 
#                           already open in correct mode, treat READ WRITE as 
#                           a special case of UPGRADE
#    akruglik    04/11/14 - Bug 18548396: set $catcon_RevertSeedPdbMode before 
#                           calling reset_seed_pdb_mode
#    akruglik    04/09/14 - Bug 18545783: change extension of catcon log file
#                           from out to lst
#    akruglik    04/02/14 - Bug 18488530: make sure pdb$seed is OPEN in READ
#                           ONLY mode if the catcon process gets killed
#    akruglik    03/26/14 - Bug 18011217: address unlink issues on Windows by
#                           embedding process id in names of various "done"
#                           files
#    dkeswani    03/14/14 - Bug 18011217: unlink issues on windows
#    akruglik    01/21/14 - Bug 17898118: rename catconInit.MigrateMode to 
#                           SeedMode and change semantics
#    talliu      01/09/13 - 14223369: add new parameter in catconInit to 
#                           indicated whether it is called by cdb_sqlexec.pl 
#    akruglik    01/16/14 - Bug 18085409: add support for running scripts
#                           against ALL PDBs and then the Root
#    akruglik    01/13/14 - Bug 18070841: modify get_pdb_names to ensure that
#                           PDB name and open_mode appear on the same line
#    dkeswani    01/13/14 - Diagnostic txn for intermittent issue in sureunlink
#                           PDB name and open_mode appear on the same line
#    akruglik    01/09/14 - Bug 18029946: add support for ignoring non-existent
#                           or closed PDBs
#    akruglik    01/07/14 - Bug 18020158: provide for a better default
#                           mechanism for internal connect string
#    akruglik    01/06/14 - Bug 17976046: get rid of Term::ReadKey workaround
#    akruglik    01/03/14 - Bug 18003416: add support for determining whether a
#                           script ended with an uncommitted transaction
#    jerrede     12/25/13 - Stripe Comma's from container name
#    akruglik    12/13/13 - Bug 17898041: use user connect string when starting
#                           SQLPlus processes
#    akruglik    12/13/13 - Bug 17898041: use "/ AS SYSDBA" as the default 
#                           connect string
#    akruglik    12/12/13 - Bug 17898118: once a process finishes running a
#                           script in a CDB, force it to switch into the Root
#    akruglik    12/09/13 - Bug 17810688: detect the case where sqlplus is not
#                           in the PATH
#    jerrede     12/05/13 - Don't override set timing values
#    akruglik    11/26/13 - modify get_connect_string to fetch password from
#                           the environment variable, if it is set
#    akruglik    11/20/13 - Bug 17637320: remove workaround added to
#                           additionalInitStmts eons ago
#    jerrede     11/18/13 - Comment out workaround additionalInitStmts
#                           Causing problems when writing to the
#                           seed database. When running post upgrade
#                           in read write mode we were unable to
#                           write to the database when calling
#                           the post upgrade procedure.
#    jerrede     11/06/13 - Move Dangling Flush Statement
#    akruglik    10/28/13 - Add support for migrate mode indicator; add
#                           subroutine to return password supplied by the user
#    akruglik    10/20/13 - Back out ReadKey-related changes
#    akruglik    10/07/13 - Bug 17550069: accept an indicator that we are being
#                           called from a GUI tool which on Windows means that
#                           the passwords and hidden parameters need not be
#                           hidden
#    jerrede     10/02/13 - Performance Improvements
#    akruglik    09/06/13 - in get_num_procs, don't impose a fixed maximum on a
#                           number of processes which may be started
#    akruglik    08/20/13 - ensure that subroutines expected to return a value
#                           do so under all cercumstances
#    jerrede     07/29/13 - Add End Process to Avoid Windows communication
#                           errors
#    akruglik    07/19/13 - add 3 new parameters to the interface of 
#                           catconExec + define catconIsCDB, catconGetConNames 
#                           and catconQuery
#    jerrede     06/18/13 - Fix shutdown checks.
#    akruglik    05/15/13 - Bug 16603368: add support for -I flag
#    akruglik    03/19/13 - do not quote the password if the user has already
#                           quoted it
#    akruglik    03/07/13 - use v$database.cdb to determine whether a DB is a
#                           CDB
#    akruglik    02/08/13 - Bug 16177906: quote user-supplied password in case
#                           it contains any special characters
#    akruglik    12/03/12 - (LRG 8526376): replace calls to
#                           DBMS_APPLICATION_INFO.SET_MODULE/ACTION with
#                           ALTER SESSION SET APPLICATION MODULE/ACTION
#    akruglik    11/15/12 - (LRG 8522365) temporarily comment out calls to
#                           DBMS_APPLICATION_INFO
#    surman      11/09/12 - 15857388: Resolve Perl typo warning
#    akruglik    11/09/12 - (LRG 7357087): PDB$SEED needs to be reopened READ
#                           WRITE unless it is already open READ WRITE or
#                           READ WRITE MIGRATE
#    akruglik    11/08/12 - use DBMS_APPLICATION_INFO to store info about
#                           processes used to run SQL scripts
#    akruglik    11/08/12 - if debugging is turned on, make STDERR hot
#    akruglik    11/08/12 - (15830396): read and ignore output in
#                           exec_DB_script if no marker was passed, to avoid
#                           hangs on Windows
#    surman      10/29/12 - 14787047: Save and restore stdout
#    mjungerm    09/10/12 - don't assume unlink will succeed - lrg 7184718
#    dkeswani    08/13/12 - Bug 14380261 : delete LOCAL variable on WINDOWS
#    akruglik    08/02/12 - (13704981): report an error if database is not open
#    sankejai    07/23/12 - 14248297: close/open PDB$SEED on all RAC instances
#    akruglik    06/26/12 - modify exec_DB_script to check whether @Output
#                           contains at least 1 element before attempting to
#                           test last character of its first element
#    akruglik    05/23/12 - rather than setting SQLTERMINATOR ON, use /
#                           instead of ; to terminate SQL statements
#    akruglik    05/18/12 - add SET SQLTERMINATOR ON after every CONNECT to
#                           ensure that SQLPLus processes do not hang if the
#                           caller sets SQLTERMINATOR to something other than
#                           ON in glogin.sql or login.sql
#    akruglik    04/18/12 - (LRG 6933132) ignore row representing a non-CDB
#                           when fetching rows from CONTAINER$
#    gravipat    03/20/12 - Rename x$pdb to x$con
#    akruglik    02/22/12 - (13745315): chop @Output in exec_DB_script to get 
#                           rid of trailing \r which get added on Windows
#    akruglik    12/09/11 - (13404337): in additionalInitStmts, set SQLPLus
#                           vars to their default values
#    akruglik    10/31/11 - modify additionalInitStmts to set more of SQLPlus
#                           system vars to prevent values set in a script run
#                           against one Container from affecting output of
#                           another script or a script run against another
#                           Container
#    prateeks    10/21/11 - MPMT changes : connect string
#    akruglik    10/14/11 - Bug 13072385: if PDB$SEED was reopened READ WRITE
#                           in concatExec, it will stay that way until
#                           catconWrapUp
#    akruglik    10/11/11 - Allow user to optionally specify internal connect
#                           string
#    akruglik    10/03/11 - address 'Use of uninitialized value in
#                           concatenation' error
#    akruglik    09/20/11 - Add support for specifying multiple containers in
#                           which to run - or not run - scripts
#    akruglik    09/20/11 - make --p and --P the default tags for regular and
#                           secret arguments
#    akruglik    09/20/11 - make default number of processes a function of
#                           cpu_count
#    akruglik    08/24/11 - exec_DB_script was hanging on Windows because Perl
#                           does not handle CHLD signal on Windows
#    pyam        08/08/11 - don't run scripts AS SYSDBA unless running as sys
#    akruglik    08/03/11 - unset TWO_TASK to ensure that connect without
#                           specifyin a service results in a connection to the
#                           root
#    akruglik    07/01/11 - encapsulate common code into subroutines
#    akruglik    06/30/11 - temporarily suspend use of Term::ReadKey
#    akruglik    06/10/11 - rename CatCon.pm to catcon.pm because oratst get
#                           macro is case-insensitive
#    akruglik    06/10/11 - Add support for spooling output of individual
#                           scripts into separate files
#    akruglik    05/26/11 - Creation
#

package catcon;

use 5.006;
use strict;
use warnings;
use English;
use IO::Handle;       # to flush buffers
use Term::ReadKey;    # to not echo password
use IPC::Open2;       # to perform 2-way communication with SQL*Plus
use File::Spec ();    # for fix of bug 17810688
use File::Basename;
use Cwd;
use Fcntl;

require Exporter;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration       use catcon ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
        
) ] );

our @EXPORT_OK = qw ( catconInit catconExec catconRunSqlInEveryProcess 
                      catconShutdown catconBounceProcesses catconWrapUp 
                      catconIsCDB catconGetConNames catconQuery catconUpgForce 
                      catconUpgEndSessions catconUpgStartSessions 
                      catconUserPass catconUpgSetPdbOpen catconSetDbClosed 
                      catconGetUsrPasswdEnvTag catconGetIntPasswdEnvTag 
                      catconXact catconForce catconReverse 
                      catconRevertPdbModes catconEZConnect 
                      catconFedRoot catconIgnoreErr catconVerbose 
                      catconSqlplus catconDisableLockdown 
                      catconRecoverFromChildDeath catconPdbMode catconAllInst);

our @EXPORT = qw(
        
);
our $VERSION = '0.01';

# forward declaration since this subroutine gets declared later than it 
# gets used
sub log_msg($);

# Preloaded methods go here.

# forward declarations
sub catconBounceDeadProcess($);

#
# get_connect_string
#
# Description:
#   If user supplied a user[/password] string
#     If user-supplied string contained a password, 
#       get user name and password from that string
#     else if constructing a user connect string and the appropriate 
#       environment variable was set, set user name to the user-supplied 
#       string and get the password from that variable
#     else 
#       set user name to the user-supplied string and prompt user for a 
#       password
#     end if
#     construct the connect string as a concatenation of 
#     - caller-supplied user,
#     - '/', 
#     - password, 
#     - "@%s" (placeholder which can be relpalced with an EZConnect string 
#         to generate a connect string for connecting to a specific instance) 
#         if $Instances is true, and
#     - AS SYSDBA if user name was SYS.
#   Otherwise,
#     set connect string to '/ AS SYSDBA'
#
# Parameters:
#   - user name, optionally with password; may be undefined
#   - a flag indicating whether to NOT echo a password if a user has to enter 
#     it
#   - password value obtained from the appropriate environment variable, if any
#   - an indicator of whether this string may need to accommodate 
#     specification of EZConnect strings for connecting to various instances
#
# Returns:
#   A Connect string as described above, the same string with redacted 
#   password, and the password.
#
sub get_connect_string ($$$$) {
  my ($user, $hidePasswd, $envPasswd, $Instances) = @_;

  my $password;
  my $connect;                                       # assembled connect string
    # connect string with redacted password (for diagnostic output)
  my $connectDiag; 
  my $sysdba = 0;

  if ($user) {
    # user name specified, possibly followed by /password
    #
    # if the password was not supplied, try to get it from the environment 
    # variable, or, failing that, by prompting the user

    if ($user =~ /(.*)\/(.*)/) {
      # user/password
      $user = $1;
      $password = $2;
    } elsif ($envPasswd) {
      $password = $envPasswd;
      $password =~ tr/"//d;  # strip double quotes
    } else {
      # prompt for password
      print "Enter Password: ";

      # do not enter noecho mode if told to not hide password
      if ($hidePasswd) {
        ReadMode 'noecho';
      }

      $password = ReadLine 0;
      chomp $password;

      # do not restore ReadMode if told to not hide password
      if ($hidePasswd) {
        ReadMode 'normal';
      }

      print "\n";
    }

    if (uc($user) eq "SYS") {
      $sysdba = 1;
    }
   
    # quote the password unless the user has done it for us or unless he hit 
    # return without entering any characters.  We assume that 
    # passwords cannot contain double quotes and trust the user to not play 
    # games with us and only half-quote the password
    if (length($password) > 0 && substr($password,1,1) ne '"') {
      $password = '"'.$password.'"'
    }
    
    if ($Instances) {
      # add a placeholder for an EZConnect string so we can use it 
      # to connect to a specific instance
      $connect = $user."/"."$password"."@%s";
      $connectDiag = $user."/"."########"."@%s";
    } else {
      $connect = $user."/"."$password";
      $connectDiag = $user."/"."########";
    }
  } else {
    # default to OS authentication

    # OS authentication cannot be used if the user wants to run scripts using
    # specified instances
    if ($Instances) {
      log_msg("get_connect_string: OS authentication cannot be used with specified instances\n");
      $connectDiag = $connect = undef;
    } else {
      $connectDiag = $connect = "/"; 
    }

    $password = undef;
    $sysdba = 1;
  } 

  # $connect may be undefined if the caller has not supplied user[/password] 
  # while the user specified instances on which to run scripts
  if ($sysdba && $connect) {
    $connect = $connect . " AS SYSDBA";
    $connectDiag = $connectDiag . " AS SYSDBA";
  }

  return ($connect, $connectDiag, $password);
}

# a file produced by spooling output from SQL*Plus commands may not have 
# been created even though a script that spooled its output into it has 
# completed, so if we need to open such file, we may need to wait a tad 
# until it exists and is non-empty
sub openSpoolFile ($$) {
  my ($spoolFile, $debugOn) = @_;

  my $iters = 0;
  my $MAX_ITERS = 1000;
  my $fh;
  my $exists;
  my $fsize = 0;

  while ((!($exists = -e $spoolFile) || !($fsize = -s $spoolFile)) && 
         $iters < $MAX_ITERS) { 
    $iters++;
    if ($debugOn && ($iters % 1) == 100) {
      if (!$exists) {
        log_msg("openSpoolFile: file ($spoolFile) was not found after ".
                $iters." attempts\n\twill try again\n");
      } else {
        log_msg("openSpoolFile: file ($spoolFile) was found but it was ".
                "empty ($fsize bytes) after $iters attempts\n".
                "\twill try again\n");
      }
    }
 
    select (undef, undef, undef, 0.01);
  }

  if ($iters < $MAX_ITERS) {
    # file exists and is not empty; try to open it and return the file handle
    if (open ($fh, "<", "$spoolFile")) {
      return $fh;
    } else {
      log_msg("openSpoolFile: file ($spoolFile) exists and has non-zero (".
              $fsize." bytes) size, but it could not be opened\n");

      return undef;
    }
  } else {
    if (!$exists) {
      log_msg("openSpoolFile: file ($spoolFile) was not found after ".
              $iters." attempts\n\tGiving up\n");
    } else {
      log_msg("openSpoolFile: file ($spoolFile) was found but it was ".
              "empty ($fsize bytes) after $iters attempts\n\tGiving up\n");
    }

    return undef;
  }
}

# A wrapper over unlink to retry if it fails, as can happen on Windows
# apparently if unlink is attempted on a .done file while the script is
# is still writing to it.  Silent failure to unlink a .done file can
# cause incorrect sequencing leading to problems like lrg 7184718
sub sureunlink ($$) {
  my ($DoneFile, $debugOn) = @_;

  my $iters = 0;
  my $MAX_ITERS = 120;

  while (!unlink($DoneFile) && $iters < $MAX_ITERS) { 
    # Bug 23614674: if unlink failed because the file no longer exists, there 
    # is nothing left to do
    if (! -e $DoneFile) {
      if ($debugOn) {
        log_msg("sureunlink: $DoneFile does not exist\n");
      }

      return;
    }

    $iters++;
    if ($debugOn) {
      log_msg("sureunlink: unlink($DoneFile) failed after $iters attempts due to $!\n");
    }

    sleep(1);
  }

  # Bug 18672126: it looks like sometimes on Windows unlink returns 1 
  # indicating that the file has been deleted, but a subsequent -e test seems 
  # to indicate that it still exists, causing us to die.  
  # 
  # One alternative would be to trust unlink and not perform the -e test at 
  # all, but I am concerned that if things really get out of whack, we may 
  # mistakenly conclude that the next script sent to a given SQL*Plus process 
  # has completed simply because the "done" file created to indicate 
  # completion of previous script took too long to go away
  #
  # To alleviate this conern, I will, instead, give the file a bit of time to 
  # really go away by running a loop while -e returns TRUE for the same 
  # number of iterations as we used above to allow unlink to succeed

  if ($iters < $MAX_ITERS) {
    if ($debugOn) {
      log_msg("sureunlink: unlink($DoneFile) succeeded after ".($iters + 1)." attempt(s)\n");
      log_msg("sureunlink: verify that the file really no longer exists\n");
    }

    $iters = 0;
    while ((-e $DoneFile) && $iters < $MAX_ITERS) {
      $iters++;
      if ($debugOn) {
        log_msg("sureunlink: unlinked file ($DoneFile) appears to still exist after $iters checks\n");
      }
      
      sleep(1);
    }

    if ($debugOn) {
      if ($iters < $MAX_ITERS) {
        log_msg("sureunlink: confirmed that $DoneFile no longer exists after ".($iters + 1)." attempts\n");
      } else {
        log_msg("sureunlink: $DoneFile refused to go away, despite unlink apparently succeeding\n");
      }
    }
  }  else {
    log_msg("sureunlink: could not unlink $DoneFile - $!\n");
  }

  die "could not unlink $DoneFile - $!" if (-e $DoneFile);
}

#
# exec_DB_script
#
# Description:
#   Connect to a database using connect string supplied by the caller, run 
#   statement(s) supplied by the caller, and if caller indicated interest in 
#   some of the output produced by the statements, return a list, possibly 
#   empty, consisting of results returned by that query which contain 
#   specified marker
#
# Parameters:
#   - a reference to a list of statements to execute
#   - a string (marker) which will mark values of interest to the caller; 
#     if no value is supplied, an uninitialized list will be returned
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed
#   - base for a name of a "done" file (see above)
#   - an indicator of whether to produce debugging info
#
# Returns
#   - If marker was supplied, a list consisting of values returned by the 
#     query, stripped of the marker; uninitialized otherwise
#   - List of lines of output produced by running the specified list of 
#     statements; this list is expected to be used by the caller if the 
#     list of values in the first output argument has something unexpected 
#     about it (like consisting of fewer or more rows than expected)
#
sub exec_DB_script (\@$$$$) {
  my ($statements, $marker, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  # file whose existence will indicate that all statements in caller's 
  # script have executed
  my $DoneFile = $DoneFilePathBase."_exec_DB_script.done";

  my @Output;
  my @Spool;

  local (*Reader, *Writer);

  # if the "done" file exists, delete it
  if (-e $DoneFile) {
    sureunlink($DoneFile, $debugOn);

    if ($debugOn) {
      log_msg("exec_DB_script: deleted $DoneFile before running a script\n");
    }
  } elsif ($debugOn) {
    log_msg("exec_DB_script: $DoneFile did not need to be deleted before running a script\n");
  }

  my $pid = open2(\*Reader, \*Writer, "sqlplus /nolog");

  if ($debugOn) {
    log_msg("exec_DB_script: opened Reader and Writer\n");
  }

  # execute sqlplus statements supplied by the caller
  foreach (@$statements) {
    print Writer $_;
    if ($debugOn) {
      if ($_ =~ /^conn/ && $_ ne 'connect / as sysdba') {
        log_msg("exec_DB_script: connected\n");
      } else {
        log_msg("exec_DB_script: executed $_\n");
      }
    }
  }

  # send a statement to generate a "done" file
  print Writer qq/$DoneCmd $DoneFile\n/;

  if ($debugOn) {
    log_msg("exec_DB_script: sent $DoneCmd $DoneFile to Writer\n");
  }

  # send EXIT
  print Writer qq/exit\n/;

  if ($debugOn) {
    log_msg("exec_DB_script: sent -exit- to Writer\n");
  }

  close Writer;       #have to close Writer before read

  if ($debugOn) {
    log_msg("exec_DB_script: closed Writer\n");
  }

  if ($marker) {
    if ($debugOn) {
      log_msg("exec_DB_script: marker = $marker - examine output\n");
    }

    # have to read one line at a time
    while (<Reader>) { 
      # bug 22330680: can't assume that a line containing a marker starts 
      # with a marker, so we look for a marker anywhere in the line and 
      # then remove anything that precedes the end of the marker, i.e. if 
      # the marker is C:A:T:C:O:N and the line looks like this:
      #   x233068-SQL>        C:A:T:C:O:NOPEN	   x233068
      # we will strip off 
      #   x233068-SQL>        C:A:T:C:O:N
      if ($_ =~ /$marker/) {
        if ($debugOn) {
          log_msg("exec_DB_script: line contains marker: $_\n");
        }
        
        # remove characters up to and including the end of the marker
        s/$marker(.*)//;
        # and add it to the list
        push @Output, $1;
        # and to the Spool
        push @Spool, $1;
      } else {
        if ($debugOn) {
          log_msg("exec_DB_script: line does not match marker: $_\n");
        }

        # add output line to the Spool
        push @Spool, $_;
      }
    }

    # (13745315) on Windows, values fetched from SQL*Plus contain trailing 
    # \r, but the rest of the code does not expect these \r's, so we will 
    # chop them here
    if (@Output && substr($Output[0], -1) eq "\r") {
      chop @Output;
      chop @Spool;
    }
  } else {
    if ($debugOn) {
      log_msg("exec_DB_script: marker was undefined; read and ignore output, if any\n");
    }

    # (15830396) read anyway
    while (<Reader>) {
      # add output line to the Spool
      push @Spool, $_;
     }

    if ($debugOn) {
      log_msg("exec_DB_script: finished reading and ignoring output\n");
    }
  }

  if ($debugOn) {
    log_msg("exec_DB_script: waiting for child process to exit\n");
  }

  # wait until the process running SQL statements terminates
  # 
  # NOTE: Instead of waiting for CHLD signal which gets issued on Linux, 
  #       we wait for a "done" file to be generated because this should 
  #       work on Windows as well as Linux (and other Operating Systems, 
  #       one hopes)
  select (undef, undef, undef, 0.01)    until (-e $DoneFile);

  if ($debugOn) {
    log_msg("exec_DB_script: child process exited\n");
  }

  sureunlink($DoneFile, $debugOn);

  if ($debugOn) {
    log_msg("exec_DB_script: deleted $DoneFile after running a script\n");
  }

  close Reader;

  if ($debugOn) {
    log_msg("exec_DB_script: closed Reader\n");
  }

  waitpid($pid, 0);   #makes your program cleaner

  if ($debugOn) {
    log_msg("exec_DB_script: waitpid returned\n");
  }

  return (\@Output, \@Spool);
}

# 
# print_exec_DB_script_output
#
# Description:
#   Print script output returned to the caller of this function by 
#   exec_DB_script
#
# Parameters:
#   - name of a function that called exec_DB_script
#   - reference to a script output buffer
#   - an indicator whether this subroutine is being called because calling 
#     function detected an error during invocation of exec_DB_script
#
# Returns:
#   none
#
sub print_exec_DB_script_output ($$$) {

  my ($callerName, $Spool_ref, $err) = @_;

  if ($err) {
    log_msg("$callerName: unexpected error in exec_DB_script\n  ");
  } else {
    log_msg("$callerName: ");
  }

  log_msg("output produced in exec_DB_script [\n");
  for my $SpoolLine ( @$Spool_ref ) {
    # lines in the Spool buffer are already \n terminated, so don't add 
    # another \n here
    log_msg("    $SpoolLine");
  }
  log_msg("  ] end of output produced in exec_DB_script\n");
}

#
# get_instance_status
#   
# Description
#   Obtain instance status.
#
# Parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   String found in V$INSTANCE.STATUS
#
sub get_instance_status (\@$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetInstStatusStatements = (
    "connect ".$connectString->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || status from v\$instance\n/\n",
  );

  my ($InstStatus_ref, $Spool_ref) = 
    exec_DB_script(@GetInstStatusStatements, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$InstStatus_ref || $#$InstStatus_ref != 0) {
    # instance status could not be obtained; if this was due to the 
    # fact that the instance was idle (ORA-01034 reported), return Idle as 
    # Instance Status; otherwise, report an error
    if (!@$InstStatus_ref) {
      for my $SpoolLine ( @$Spool_ref ) {
        if ($SpoolLine =~ /ORA-01034/) {
          return ("Idle");
        }
      }
    }

    print_exec_DB_script_output("get_instance_status", $Spool_ref, 1);
    undef @$InstStatus_ref;
  }

  return $$InstStatus_ref[0];
}

#
# get_instance_status_and_name
#   
# Description
#   Obtain instance status and name.
#
# Parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   Strings found in V$INSTANCE.STATUS and V$INSTANCE.INSTANCE_NAME
#
sub get_instance_status_and_name (\@$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetInstStatusStatements = (
    "connect ".$connectString->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || status, instance_name from v\$instance\n/\n",
  );

  # should return exactly 1 row
  my ($out_ref, $Spool_ref) = 
    exec_DB_script(@GetInstStatusStatements, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  my $InstStatus;
  my $InstName;

  if (!@$out_ref || $#$out_ref != 0) {
    # instance status and name could not be obtained; if this was due to the 
    # fact that the instance was idle (ORA-01034 reported), return Idle as 
    # InstStatus; otherwise, report an error
    if (!@$out_ref) {
      for my $SpoolLine ( @$Spool_ref ) {
        if ($SpoolLine =~ /ORA-01034/) {
          return ("Idle", "N/A");
        }
      }
    }
    
    print_exec_DB_script_output("get_instance_status_and_name", $Spool_ref, 1);
  } else {
    # split the row into instance status and instance name
    ($InstStatus, $InstName) = split /\s+/, $$out_ref[0];
  }

  return ($InstStatus, $InstName);
}

#
# get_dbms_version
#   
# Description
#   Obtain DBMS version
#
# Parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   String found in V$INSTANCE.VERSION
#
sub get_dbms_version(\@$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetInstStatusStatements = (
    "connect ".$connectString->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || version from v\$instance\n/\n",
  );

  # should return exactly 1 row
  my ($out_ref, $Spool_ref) = 
    exec_DB_script(@GetInstStatusStatements, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$out_ref || $#$out_ref != 0) {
    # DBMS version could not be obtained
    print_exec_DB_script_output("get_dbms_version", $Spool_ref, 1);
    undef @$out_ref;
  }

  return $$out_ref[0];
}

#
# get_CDB_indicator
#   
# Description
#   Obtain an indicator of whether a DB is a CDB
#
# Parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   String found in V$DATABASE.CDB
#
sub get_CDB_indicator (\@$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  # We used to rely on CONTAINER$ not returning any rows, but in 
  # a non-CDB from a shiphome, CONTAINER$ will have a row representing 
  # CDB$ROOT (because we ship a single Seed which is is CDB and unlink 
  # PDB$SEED if a customer wants a non-CDB; you could argue that we 
  # could have purged CDB$ROOT from CONTAINER$ the same way we purged 
  # SEED$PDB, but things are the way they are, and it is more robust 
  # to query V$DATABASE.CDB)

  my @GetIsCdbStatements = (
    "connect $connectString->[0]\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || cdb from v\$database\n/\n",
  );

  my ($IsCDB_ref, $Spool_ref) = 
    exec_DB_script(@GetIsCdbStatements, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$IsCDB_ref || $#$IsCDB_ref != 0) {
    # CDB Indicator could not be obtained
    print_exec_DB_script_output("get_CDB_indicator", $Spool_ref, 1);
    undef @$IsCDB_ref;
  }
  
  return $$IsCDB_ref[0];
}

#
# get_CDB_open_mode
#   
# Description
#   Obtain CDB's open mode
#
# Parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   String found in V$DATABASE.OPEN_MODE
#
sub get_CDB_open_mode (\@$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetCdbOpenModeStatements = (
    "connect $connectString->[0]\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || open_mode from v\$database\n/\n",
  );

  my ($openMode_ref, $Spool_ref) = 
    exec_DB_script(@GetCdbOpenModeStatements, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$openMode_ref || $#$openMode_ref != 0) {
    # CDB open mode could not be obtained
    print_exec_DB_script_output("get_CDB_open_mode", $Spool_ref, 1);
    undef @$openMode_ref;
  }
  
  return $$openMode_ref[0];
}

#
# get_dbid
#   
# Description
#   Obtain database' DBID
#
# Parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   Database' DBID (V$DATABASE.DBID)
#
sub get_dbid (\@$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetDbIdStatements = (
    "connect ".$connectString->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || dbid from v\$database\n/\n",
  );

  my ($DBID_ref, $Spool_ref) = 
    exec_DB_script(@GetDbIdStatements, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$DBID_ref || $#$DBID_ref != 0) {
    # instance status could not be obtained
    print_exec_DB_script_output("get_dbid", $Spool_ref, 1);
    undef @$DBID_ref;
  }

  return $$DBID_ref[0];
}

#
# get_con_id
#   
# Description
#   Obtain id of a CDB Container to which we will connect using a specified 
#   EZConnect string
#
# Parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   CON_ID of the container to which we are connected.
#
sub get_con_id (\@$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetConIdStatements = (
    "connect ".$connectString->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || sys_context(\'USERENV\', \'CON_ID\') as con_id from dual\n/\n",
  );

  my ($CON_ID_ref, $Spool_ref) = 
    exec_DB_script(@GetConIdStatements, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$CON_ID_ref || $#$CON_ID_ref != 0) {
    # instance status could not be obtained
    print_exec_DB_script_output("get_con_id", $Spool_ref, 1);
    undef @$CON_ID_ref;
  }

  return $$CON_ID_ref[0];
}

#
# build_connect_string
#   
# Description
#    Build a Connect String, possibly including an EZConnect String
#
# Parameters:
#   - connect string template - a string that looks like 
#     - user/password [AS SYSDBA] or "/ AS SYSDBA" if EZConnect strings were 
#       not supplied by the caller or
#     - user/password@%s [AS SYSDBA] otherwise
#   - an EZConnect string corresponding to an instance or an empty string if 
#       the default instance is to be used
#   - an indicator of whether to produce debugging messages
# Returns
#   A connect string which can be used to connect to the instance with 
#   specified EZConnect string or to the default instance
#
sub build_connect_string ($$$) {
  my ($connStrTemplate, $EZConnect, $debugOn) = @_;
  
  if ($debugOn) {
    my $msg = <<build_connect_string_DEBUG;
running build_connect_string(
  connStrTemplate = $connStrTemplate, 
  EZConnect       = $EZConnect)
build_connect_string_DEBUG
    log_msg($msg);
  }

  if (!$EZConnect || ($EZConnect eq "")) {
    # caller has not supplied an EZConnect string. Make sure that 
    # $connStrTemplate does not contain a placeholder for one
    if (index($connStrTemplate, "@%s") != -1) {
      my $msg = <<msg;
build_connect_string: connect string template includes an EZConnect 
    placeholder but the caller has supplied an empty EZConnect String
msg
      log_msg($msg);
      return undef;
    }

    # EZConnect string was not supplied, nor was it expected, so just use 
    # the default instance
    if ($debugOn) {
      my $msg = <<msg;
build_connect_string: return caller-supplied string ($connStrTemplate)
    as the connect string
msg
      log_msg($msg);
    }

    return $connStrTemplate;
  } 
    
  # caller has supplied an EZConnect string. Make sure that $connStrTemplate 
  # contains a placeholder for one
  if (index($connStrTemplate, "@%s") == -1) {
    my $msg = <<msg;
build_connect_string: connect string template does not include an EZConnect 
    placeholder but the caller has supplied an EZConnect String
msg
    log_msg($msg);
    return undef;
  }
  
  # construct a connect string by replacing the placeholder with the 
  # supplied EZConnect string
  
  my $outStr = sprintf($connStrTemplate, $EZConnect);

  if ($debugOn) {
    log_msg("build_connect_string: return ".$outStr." as the connect string\n");
  }

  return $outStr;
}

#
# get_num_procs_int
#   
# Description
#   Compute default number of SQL*Plus sessions to be started by picking the 
#   lesser of (2*<cpu_count parameter>) and <sessions parameter> and 
#   subtracting from it the number of active sessions currently connected to 
#   the instance
#
# Parameters:
#   - reference to an array of connect strings:
#       [0] used to connect to a DB
#       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - an indicator of whether we should determine number of processes that 
#     can be opened on all of CDB's instances
#   - reference to a hash mapping instance names to, among other things, a 
#     reference to a hash representing the number of SQL*Plus processes that 
#     will be allocated for that instance if $getProcsOnAllInstances is 
#     set (OUT)
#   - number of processes specified by the caller of get_num_procs; will be 
#     consulted ONLY IF get_num_procs told us to  determine number of 
#     processes that can be opened on all of CDB's instances (recall that in 
#     that case get_num_procs does not blindly satisfy the caller's request 
#     to open specified number of processes but instead calls this function 
#     to determine how many sqlplus processes can be started on every instance)
#   - indicator of whether to produce debugging info
#
# Returns
#   number of processes which will be used to run script(s)
#
sub get_num_procs_int ($$$$$$$) {
  my ($connectString, $DoneCmd, $DoneFilePathBase, 
      $getProcsOnAllInstances, $InstProcMap_ref, $NumProcs, $debugOn) = @_;

  # Bug 20193612: if asked to determine how many processes can be started on 
  #   all of CDB's instances, determine how many processes can be started 
  #   on every instance and return a sum of these numbers; 
  #   otherwise, compute number of sessions that can be created on the current 
  #   instance
  if ($getProcsOnAllInstances) {
    if ($debugOn) {
      log_msg("get_num_procs_int: using all available instances;\n");
      log_msg("\tinvoke get_inst_procs to determine how\n".
              "\tmany processes can be allocated to every instance\n");
    }

    my $instProcs_ref = 
      get_inst_procs($connectString, $DoneCmd, $DoneFilePathBase, $debugOn);

    if ((! defined $instProcs_ref) || !%$instProcs_ref || 
        (keys %$instProcs_ref < 1)) {
      if (! defined $instProcs_ref) {
        log_msg("get_num_procs_int: undefined reference returned by  ".
                "get_inst_procs\n");
      } elsif (!%$instProcs_ref) {
        log_msg("get_num_procs_int: hash reference to which was returned by ".
                "get_inst_procs was undefined\n");
      } else {
        log_msg("get_num_procs_int: hash reference to which was returned by ".
                "get_inst_procs had no entries\n");
      }

      return undef;
    }

    # Hash contains at least one entry.  Traverse the hash adding up 
    # positive number of processes that can be allocated to an instance
    my $retNumProcs = 0;
    
    if ($debugOn) {
      log_msg("get_num_procs_int: examine hash mapping instance names to the ".
              "number of\n".
              "SQL*Plus processes that can be allocated to that instance:\n");
    }

    foreach my $inst (keys %$instProcs_ref) {
      my $procs = $instProcs_ref->{$inst}->{NUM_PROCS};
      if ($debugOn) {
        log_msg("\t$inst => $procs\n");
      }

      if ($procs > 0) {
        $retNumProcs += $procs;
      } else {
        if ($debugOn) {
          log_msg("\tno processes can be allocated to this instance\n");
        }

        # in the unlikely case that this entry contains a negative number, 
        # zero it out
        $instProcs_ref->{$inst}->{NUM_PROCS} = 0 
          if ($instProcs_ref->{$inst}->{NUM_PROCS} < 0);
      }
    }

    # if
    #   - some processes can be started on at least one instance but
    #   - that number is less than the caller specified number of sqlplus 
    # processes to start
    # we will increase the number of processes on every instance on which 
    # some processes can be started until we reach the number specified by 
    # the caller
    if ($retNumProcs > 0 && $NumProcs > $retNumProcs) {
      if ($debugOn) {
        log_msg("get_num_procs_int: total number of processes determined by ".
                "get_inst_procs ($retNumProcs)\n".
                "  is below the numberof processes specified by the caller ".
                "($NumProcs) - increase number\n".
                "  of processes allocated for instances to which some ".
                "processes have been allocated to\n".
                "  meet the number of processes specified by the caller:\n");
      }

      while ($NumProcs > $retNumProcs) {
        foreach my $inst (keys %$instProcs_ref) {
          if ($instProcs_ref->{$inst}->{NUM_PROCS}) {
            $instProcs_ref->{$inst}->{NUM_PROCS}++;
            $retNumProcs++;
            
            if ($debugOn) {
              log_msg("\tincrease number of processes for instance $inst to ".
                      $instProcs_ref->{$inst}->{NUM_PROCS}."\n");
            }
          }
        }
      }
    }

    %$InstProcMap_ref = %$instProcs_ref;

    if ($debugOn) {
      log_msg("get_num_procs_int: handing the resulting hash to the ".
              "caller:\n");

      foreach my $inst (sort keys %$InstProcMap_ref) {
        foreach my $k (keys %{$InstProcMap_ref->{$inst}}) {
          log_msg("\t$inst => ($k => $InstProcMap_ref->{$inst}->{$k})\n");
        }
      }
    }

    # NOTE: if it looks like there are no instances for which we can allocate 
    #       processes (a highly unlikely scenario), we will return 0 to the 
    #       caller and let him deal with it
    return $retNumProcs;
  } else {
    my @GetNumProcsStatements = (
      "connect ".$connectString->[0]."\n",
      "set echo off\n",
      "set heading off\n",
      "select \'C:A:T:C:O:N\' || ".
              "least(cpu_param.value*2, sess_param.value-sess.num_sessions) ".
        "from v\$parameter cpu_param, v\$parameter sess_param, ".
             "(select count(*) as num_sessions ".
                "from v\$session ".
                "where status=\'ACTIVE\') sess ".
        "where cpu_param.name=\'cpu_count\' ".
          "and sess_param.name=\'sessions\'\n/\n",  
    );

    my ($NumProcs_ref, $Spool_ref) = 
      exec_DB_script(@GetNumProcsStatements, "C:A:T:C:O:N", 
                     $DoneCmd, $DoneFilePathBase, $debugOn);

    if (!@$NumProcs_ref || $#$NumProcs_ref != 0) {
      # number of processes could not be obtained
      if (!@$NumProcs_ref) {
        log_msg("get_num_procs_int: array a reference to which was ".
                "returned by exec_DB_script was not defined\n");
      } else {
        log_msg("get_num_procs_int: array a reference to which was ".
                "returned by exec_DB_script did not contain exactly 1 ".
                "row\n");
      }

      print_exec_DB_script_output("get_num_procs_int", $Spool_ref, 1);
      undef @$NumProcs_ref;
    }

    return $$NumProcs_ref[0];
  }
}

#
# get_num_procs - determine number of processes which should be created
#
# parameters:
#   - number of processes, if any, supplied by the user
#   - number of concurrent script invocations, as supplied by the user 
#     (external degree of parallelism)
#   - reference to an array of connect strings:
#       [0] used to connect to a DB, 
#       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - an indicator of whether we should determine number of processes that 
#     can be opened on all of CDB's instances
#   - reference to a hash mapping instance names to, among other things, a 
#     reference to a hash representing the number of SQL*Plus processes that 
#     will be allocated for that instance if $getProcsOnAllInstances is 
#     set (OUT)
#   - an indicator of whether to produce debugging info
#
sub get_num_procs ($$$$$$$$) {
  my ($NumProcs, $ExtParaDegree, $ConnectString, $DoneCmd, $DoneFilePathBase, 
      $getProcsOnAllInstances, $InstProcMap_ref, $DebugOn) = @_;

  my $MinNumProcs  = 1;   # minimum number of processes
  my $ProcsToStart;       # will be returned to the caller

  # compute the number of processes to be started
  if (!$getProcsOnAllInstances && $NumProcs > 0) {

    # caller supplied a number of processes; 
    $ProcsToStart = $NumProcs;

    if ($DebugOn) {
      log_msg("get_num_procs: caller-supplied number of processes ".
              "(not final) = $NumProcs\n");
    }
  } else {    
    # first obtain a default value for number of processes based on 
    # hardware characteristics
    $ProcsToStart = get_num_procs_int($ConnectString, $DoneCmd, 
                                      $DoneFilePathBase, 
                                      $getProcsOnAllInstances, 
                                      $InstProcMap_ref, $NumProcs, $DebugOn);
      
    if (!(defined $ProcsToStart)) {
      log_msg("get_num_procs: unexpected error in get_num_procs_int\n");
      return -1;
    }
    
    if ($DebugOn) {
      log_msg("get_num_procs: computed number of processes (not final) = ".
              "$ProcsToStart\n");
    }

    # if called interactively and the caller has provided the number of 
    # concurrent script invocations on this host, compute number of 
    # processes which will be started in this invocation
    #
    # This option is not used much (if at all), and we will skip this 
    # recalculation if using all available instances

    if ($ExtParaDegree && !$getProcsOnAllInstances) {
      if ($DebugOn) {
        log_msg("get_num_procs: number of concurrent script invocations = ".
                "$ExtParaDegree\n");
      }

      $ProcsToStart = int ($ProcsToStart / $ExtParaDegree);

      if ($DebugOn) {
        my $msg = <<msg;
get_num_procs: adjusted for external parallelism, number of processes (still 
    not final) for this invocation of the script = $ProcsToStart
msg
        log_msg($msg);
      }
    }
  }

  # use number of processes which was either supplied by the user or 
  # computed by get_num_procs_int() it unless it is too small
  #
  # if using all available instances and get_num_procs_int 
  # determined how many processes can be allocated to various instances, we 
  # will not secondguess it, unless it determined that no processes can be 
  # created on any intance, in which case we will tell the caller that a 
  # minimum number of processes can be allocated on the current instance
  if ($getProcsOnAllInstances) {
    if (!$ProcsToStart) {
      # obtain name of the current instance
      my $currInst;
      
      (undef, $currInst) = 
        get_instance_status_and_name(@$ConnectString, $DoneCmd, 
                                     $DoneFilePathBase, $DebugOn);

      if (!$currInst) {
        log_msg("get_num_procs: unexpected error in ".
                "get_instance_status_and_name\n");
        return -1;
      }
      
      if ($DebugOn) {
        log_msg("get_num_procs: standard computation determined that no ".
                "sqlplus processes can be allocated\n".
                "\tto any instance; will record that a minimum number of ".
                "processes should be\n".
                "\tstarted and allocated to the current instance ".
                "($currInst)\n");
      }

      # remember how many sqlplus processes can be allocated to the current 
      # instance
      my %numProcsHash = (NUM_PROCS => $MinNumProcs);

      $InstProcMap_ref->{$currInst} = \%numProcsHash;

      $ProcsToStart = $MinNumProcs;

      if ($DebugOn) {
        log_msg("num_procs: added $currInst => (NUM_PROCS => ".
                $MinNumProcs.") entry to the hash\n");
      }
    }
  } elsif ($ProcsToStart < $MinNumProcs) {
    $ProcsToStart = $MinNumProcs;
  }

  if ($DebugOn) {
    # $InstProcMap_ref will be undefined for a non-CDB case so we need to check
    # whether $InstProcMap_ref is defined before attempting to dereference it
    if ($InstProcMap_ref && %$InstProcMap_ref) {
      log_msg("get_num_procs: handing instance-to-process hash to the ".
              "caller:\n");

      foreach my $inst (sort keys %$InstProcMap_ref) {
        foreach my $k (keys %{$InstProcMap_ref->{$inst}}) {
          log_msg("\t$inst => ($k => $InstProcMap_ref->{$inst}->{$k})\n");
        }
      }
    }

    log_msg("get_num_procs: will start $ProcsToStart processes (final)\n");
  }

  return $ProcsToStart;
}

#
# gen_inst_conn_strings - construct connect strings for instances on which 
#                         a DB is running
#
# parameters:
#   - reference to an array of connect strings:
#       [0] used to connect to a DB, 
#       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - reference to a hash mapping instance names to connect strings (OUT)
#   - an indicator of whether to produce debugging info
#
sub gen_inst_conn_strings ($$$$$) {
  my ($ConnectString, $DoneCmd, $DoneFilePathBase, $InstConnStrMap_ref, 
      $DebugOn) = @_;

  # NOTE: in the below query, we use regexp_substr to extract (ADDRESS=...) 
  #       from local_listener.value since it may contain other info.  
  #       Similarly, regexp_substr is used to extract the first service name 
  #       from service_name.value which may contain a comma-separated list of 
  #       service names
  my @GetInstConnStrStmts = (
      "connect ".$ConnectString->[0]."\n",
      "set echo off\n",
      "set heading off\n",
      "set lines 1000\n",
      "select \'C:A:T:C:O:N\' || ".
             "instance.instance_name || \'@\' || ".
             "\'(DESCRIPTION=\' || ".
             "regexp_substr(local_listener.value, ".
               "\'\\(ADDRESS=(\\([^()]+\\))+\\)\', 1, 1, \'i\') || ".
             "\'(CONNECT_DATA=(SERVICE_NAME=\' || ".
             "regexp_substr(service_name.value, \'[^,]+\', 1, 1) || ".
             "\')))\' ".
        "from gv\$instance instance, gv\$listener_network local_listener, ".
             "gv\$parameter service_name ".
       "where instance.inst_id = local_listener.inst_id ".
         "and instance.inst_id = service_name.inst_id ".
         "and local_listener.type=\'LOCAL LISTENER\' ".
         "and service_name.name=\'service_names\' ".
         "and local_listener.con_id in (0,1) ".
         "and service_name.con_id in (0,1)\n/\n",  
    );

  my ($retValues_ref, $Spool_ref) = 
    exec_DB_script(@GetInstConnStrStmts, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $DebugOn);

  # Bug 25366291: in some cases the above query may return no rows (causing 
  #   @$retValues_ref to be undefined.)  Rather than treating it as an error, 
  #   we will force all processing to occur on the default instance
  if (!$retValues_ref) {
    log_msg("gen_inst_conn_strings: no data was obtained\n");
    print_exec_DB_script_output("gen_inst_conn_strings", $Spool_ref, 1);
    return undef;
  }

  foreach my $row (@$retValues_ref) {
    # $row should consist of an instance name and a connect string, 
    # separated by @
    my @vals = split(/@/, $row);

    if ($#vals != 1) {
      log_msg("gen_inst_conn_strings: data contained ".
              "an element ($row) with unexpected format\n");
      print_exec_DB_script_output("gen_inst_conn_strings", $Spool_ref, 1);
      return undef;
    }

    # gv$listener_network may contain multiple rows for a given instance (one 
    # per local listener) so we want to add a representation of this row to 
    # the hash only if it does not already contain an entry for this instance
    if (! exists $InstConnStrMap_ref->{$vals[0]}) {
      $InstConnStrMap_ref->{$vals[0]} = $vals[1];

      if ($DebugOn) {
        log_msg("gen_inst_conn_strings: mapping instance $vals[0] to ".
                $InstConnStrMap_ref->{$vals[0]}."\n");
      }
    } elsif ($DebugOn) {
      log_msg("gen_inst_conn_strings: InstConnStrMap_ref already contains an ".
              "entry for instance\n".
              "\t".$vals[0]." described by row (".$row.");\n".
              "keeping the existing mapping\n");
    }
  }

  # names of instances represented by %$InstConnStrMap_ref
  my @instances = keys %$InstConnStrMap_ref;

  if ($#instances == 0) {
    # database runs on a single instance so the connect string is unnecessary:
    # (SQL*Plus processes to run scripts will be started using the default 
    # instance and force_pdb_modes() will connect to the default instance 
    # before issuing ALTER PDB CLOSE statements if running against a 12.1 CDB)
    $InstConnStrMap_ref->{$instances[0]} = ' ';

    if ($DebugOn) {
      log_msg("gen_inst_conn_strings: DB running on a single instance,\n".
              "\tconnect string for instance $instances[0] will be ".
              "discarded\n");
    }
  } elsif ($DebugOn && $#instances  == -1) {
      log_msg("gen_inst_conn_strings: no instance connect strings were ".
              "fetched.\n".
              "\tAll processing will occur on the default instance\n");
  }

  # return number of instances on which a DB is running
  return $#instances+1;
}

#
# valid_src_dir make sure source directory exists and is readable
#
# Parameters:
#   - source directory name; may be undefined
#
# Returns
#   1 if valid; 0 otherwise
#
sub valid_src_dir ($) {
  my ($SrcDir) = @_;

  if (!$SrcDir) {
    # no source directory specified - can't complain of it being invalid
    return 1;
  }

  # if a directory for sqlplus script(s) has been specified, verify that it 
  # exists and is readable
  stat($SrcDir);

  if (! -e _ || ! -d _) {
    log_msg("valid_src_dir: Specified source file directory ($SrcDir) does not exist or is not a directory\n");
    return 0;
  }

  if (! -r _) {
    log_msg("valid_src_dir: Specified source file directory ($SrcDir) is unreadable\n");
    return 0;
  }

  return 1;
}

#
# valid_log_dir make sure log directory exists and is writable
#
# Parameters:
#   - log directory name; may be undefined
#
# Returns
#   1 if valid; 0 otherwise
#
sub valid_log_dir ($) {
  my ($LogDir) = @_;

  if (!$LogDir) {
    # no log directory specified - can't complain of it being invalid
    return 1;
  }

  stat($LogDir);

  if (! -e _ || ! -d _) {
      log_msg <<msg;
valid_log_dir: Specified log file directory ($LogDir) does not exist or 
    is not a directory
msg
    return 0;
  }

  if (! -w _) {
      log_msg <<msg;
valid_log_dir: Specified log file directory ($LogDir) is unwritable
msg
    return 0;
  }

  return 1;
}

#
# validate_con_names - determine whether Container name(s) supplied
#     by the caller are valid (i.e. that the DB is Consolidated and contains a 
#     Container with specified names) and modify a list of Containers against 
#     which scripts/statements will be run as directed by the caller:  
#     - if the caller indicated that a list of Container names to be validated
#       refers to Containers against which scripts/statements should not be 
#       run, remove them from $Containers
#     - otherwise, remove names of Containers which did not appear on the 
#       list of Container names to be validated from $Containers
#
# Parameters:
#   - string containing space-delimited Container name(s)
#   - an indicator of whether the above list refers to Containers against 
#     which scripts/statements should not be run
#   - an array of Container names, if any
#   - reference to a hash of indicators of whether a given 
#     Container is open (Y/N); may be undefined if the caller specified that 
#     all relevant PDBs need to be open before being operated upon
#   - an indicator of whether non-existent and closed PDBs should be ignored
#   - an indicator of whether all PDBs should be ignored 
#   - an indicator of whether debugging information should be produced
#
# Returns:
#   An empty list if an error was encountered or if the list of names of 
#   Containers to be excluded consisted of names of all Containers of a CDB.
#   A list of names of Containers which were either explicitly included or 
#   were NOT explicitly excluded by the caller
#
sub validate_con_names (\$$\@$$$$) {
  my ($ConNameStr, $Exclude, $Containers, $IsOpen, $Force, $ForceUpg, 
      $DebugOn) = @_;

  if ($DebugOn) {
    my $msg = <<validate_con_names_DEBUG;
running validate_con_names(
  ConNameStr   = $$ConNameStr, 
  Exclude      = $Exclude, 
  Containers   = @$Containers,
  Force        = $Force,
  ForceUpg     = $ForceUpg)
validate_con_names_DEBUG
    log_msg($msg);
  }

  if (!${$ConNameStr}) {
    # this subroutine should not be called unless there are Container names to 
    # validate
    log_msg("validate_con_names: missing Container name string\n");

    return ();
  }

  my $SkippedClosedPDBs = 0; # were any PDBs skipped because they were not open

  my $LocConNameStr = $$ConNameStr;
  $LocConNameStr =~ s/^'(.*)'$/$1/;  # strip single quotes
  # extract Container names into an array
  my @ConNameArr = split(/  */, $LocConNameStr);

  if (!@ConNameArr) {
    # string supplied by the caller contained no Container names
    log_msg("validate_con_names: no Container names to validate\n");

    return ();
  }

  # string supplied by the caller contained 1 or more Container names

  if (!@$Containers) {
    # report error and quit since the database is not Consolidated
    log_msg("validate_con_names: Container name(s) supplied but the DB is ".
            "not a CDB\n");

    return ();
  } 
    
  if ($DebugOn) {
    log_msg("validate_con_names: Container name string consisted of the ".
            "following names {\n");
    foreach (@ConNameArr) {
      log_msg("validate_con_names: \t$_\n");
    }
    log_msg("validate_con_names: }\n");
  }

  #
  # Trust that user knows what they are doing and don't validate
  # the pdbs just get out.  Needed to open when PDB's are closed
  # during upgrade.
  #
  if ($ForceUpg){
      if ($DebugOn) {
          log_msg("validate_con_names: No Validation, since ForceUpg is ".
                  "specified, just return given array\n");
      }
      return @ConNameArr;
  }

  # so here's a dilemma: 
  #   we need to 
  #   (1) verify that all names in @ConNameArr are also in @$Containers AND
  #   (2) compute either 
  #       @$Containers - @ConNameArr (if $Exclude != 0) or 
  #       @$Containers <intersect> @ConNameArr (if $Exclude == 0)
  #
  #       In either case, I would like to ensure that CDB$ROOT, if it ends up 
  #       in the resulting set, remains the first element of the array (makes 
  #       it easier when we nnee to decide at what point we can parallelize 
  #       execution across all remaining PDBs) and that PDBs are ordered by 
  #       their Container ID (I am just used to processing them in that order)
  #
  #   (1) would be easiest to accomplish if I were to store contents of 
  #   @$Containers in a hash and iterate over @ConNameArr, checking against 
  #`  that hash.  However, (2) requires that I traverse @$Containers and 
  #   decide whether to keep or remove an element based on whether it occurs 
  #   in @ConNameArr (which would require that I construct a hash using 
  #   contents of @ConNameArr) and the value of $Exclude
  #
  # I intend to implement the following algorithm:
  # - store contents of @ConNameArr in a hash (%ConNameHash)
  # - create a new array which will store result of (2) (@ConOut)
  # - for every element C in @$Containers
  #   - if %ConNameHash contains an entry for C
  #     - if !$Exclude
  #       - append C to @ConOut
  #   - else
  #     - if $Exclude
  #       - append C to @ConOut
  # - if, after we traverse @$Containers, %ConNameHash still contains some 
  #   entries which were not matched, report an error since it indicates that 
  #   $$ConNameStr contained some names which do not refer to existing 
  #   Container names
  # - if @ConOut is empty, report an error since there will be no Containers 
  #   in which to run scripts/statements
  # - return @ConOut which will by now contain names of all Containers in which
  #   scripts/statements will need to be run
  #

  my %ConNameHash;   # hash of elements of @ConNameArr

  foreach (@ConNameArr) {
    undef $ConNameHash{uc $_};
  }

  # array of Container names against which scripts/statements should be run
  my @ConOut = ();
  my $matched = 0;   # number of elements of @$Containers found in %ConNameHash

  for (my $CurCon = 0; $CurCon <= $#$Containers; $CurCon++) {

    my $Con = uc $$Containers[$CurCon];
    my $Open = ($IsOpen && defined $IsOpen->{$Con}) ? $IsOpen->{$Con} : "Y";
    
    # if we have matched every element of %ConNameHash, there is no reason to 
    # check whether it contains $Con
    if (($matched < @ConNameArr) && exists($ConNameHash{$Con})) {
      $matched++; # remember that one more element of @$Containers was found

      # remove $Con from %ConNameHash so that if we end up not matching some 
      # specified Container names, we can list them as a part of error message
      delete $ConNameHash{$Con};

      if ($DebugOn) {
        log_msg("validate_con_names: $$Containers[$CurCon] was matched\n");
      }

      # add matched Container name to @ConOut if caller indicated that 
      # Containers whose names were specified are to be included in the set 
      # of Containers against which scripts/statements will be run
      if (!$Exclude) {
        # Bug 18029946: will run scripts against this Container only if it is 
        # open
        if ($Open eq "Y") {
          push(@ConOut, $$Containers[$CurCon]); 

          if ($DebugOn) {
            log_msg("validate_con_names: Added $$Containers[$CurCon] to ".
                    "ConOut\n");
          }
        } elsif ($Force) {
          if ($DebugOn) {
            log_msg("validate_con_names: $$Containers[$CurCon] is not Open, ".
                    "but Force is specified, so skip it\n");
          }

          # Bug 20782683: remember that we skipped at least one candidate PDB; 
          # if we end up with an empty list, we will mention all specified 
          # PDBs were closed 
          $SkippedClosedPDBs = 1;
        } else {
          log_msg("validate_con_names: $$Containers[$CurCon] is not open\n");
          
          return ();
        }
      }
    } else {
      if ($DebugOn) {
        log_msg("validate_con_names: $$Containers[$CurCon] was not matched\n");
      }

      # add unmatched Container name to @ConOut if caller indicated that 
      # Containers whose names were specified are to be excluded from the set 
      # of Containers against which scripts/statements will be run
      if ($Exclude) {
        # Bug 18029946: will run scripts against this Container only if it is 
        # open
        if ($Open eq "Y") {
          push(@ConOut, $$Containers[$CurCon]); 

          if ($DebugOn) {
            log_msg("validate_con_names: Added $$Containers[$CurCon] to ".
                    "ConOut\n");
          }
        } elsif ($Force) {
          if ($DebugOn) {
            log_msg("validate_con_names: $$Containers[$CurCon] is not Open, ".
                    "but Force is specified, so skip it\n");
          }

          # Bug 20782683: remember that we skipped at least one candidate PDB; 
          # if we end up with an empty list, we will mention all specified 
          # PDBs were closed 
          $SkippedClosedPDBs = 1;
        } else {
          log_msg("validate_con_names: $$Containers[$CurCon] is not open\n");
          
          return ();
        }
      }
    }
  }

  # if any of specified Container names did not get matched, report an error
  if ($matched != @ConNameArr) {
    # Bug 18029946: print the list of unmatched Container names if were not 
    # told to ignore unmatched Container names or if debug flag is set
    if (!$Force || $DebugOn) {
      log_msg("validate_con_names: some specified Container names do not ".
              "refer to existing Containers:\n");
      for (keys %ConNameHash) {
        log_msg("\t$_\n");
      }

      if ($Force) {
        log_msg("validate_con_names: unmatched Container names ignored ".
                "because Force is specified\n");
      }
    }

    # Bug 18029946: unless told to ignore unmatched Container names, return 
    # an empty list which will cause the caller to report an error
    if (!$Force) {
      return ();
    }
  }

  # if @ConOut is empty (which could happen if we were asked to exclude every 
  # Container or if all existing PDBs which matched caller's criteria were 
  # closed)
  if (!@ConOut) {
    if ($SkippedClosedPDBs == 1) {
      log_msg("validate_con_names: all existing PDBs matching caller's ".
              "criteria were closed\n");
    } else {
      log_msg("validate_con_names: resulting Container list is empty\n");
    }

    return ();
  }

  if ($DebugOn) {
    log_msg("validate_con_names: resulting Container set consists of the ".
            "following Containers {\n");
    foreach (@ConOut) {
      log_msg("validate_con_names:\t$_\n");
    }
    log_msg("validate_con_names: }\n");
  }

  return @ConOut;
}

#
# split_root_and_pdbs
#
# This subroutine will split an array of Container names a reference to which 
# was passed by the caller into Root (if any) variable and PDBs (if any) array 
# and will return them to the caller
#
# Parameters:
#   - an array of Container names (note: this subroutine assumes that the 
#     caller has verified that these names actually refer to Containers - its 
#     job is to spluit Root from everything else)
#   - name of the Root Container (we all know it is CDB$ROOT, but in case it 
#     ever changes, or we decide to use this array to separate one Container 
#     from the rest, we will ask the caller to provide us with the name)
#
# Returns:
#   A variable which will be set to $$rootName if a @$containers contained 
#   $$rootName and a reference to an array of names (if any) found in 
#   @$containers which did not match $$rootName
#
sub split_root_and_pdbs(\@\$) {
  my ($containers, $rootName) = @_;

  my $root;
  my @pdbs; 

  for (my $curCon = 0; $curCon <= $#$containers; $curCon++) {
    if ($$containers[$curCon] ne $$rootName) {
      push @pdbs, $$containers[$curCon];
    } else {
      $root = $$rootName;
    }
  }

  return ($root, \@pdbs);
}

#
# get_con_info - query V$CONTAINERS to get info about all Containers (currently
#                we are only getting names and open modes)
#
# parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - reference to an array of Container names; may be undef if the caller is 
#     not interested in Container names (e.g. because he has already obtained 
#     them) (OUT)
#   - reference to a hash of indicators of whether a given Container is 
#     open (OUT)
#   - indicator of whether we will be running user scripts (meaning that
#     CDB$ROOT and PDB$SEED should be skipped)
#   - indicator of whether to produce debugging info
#
sub get_con_info (\@$$$$$$) {
  my ($myConnect, $DoneCmd, $DoneFilePathBase, $ConNames, $IsOpen, $userScript,
      $debugOn) = @_;

  # con_id for the first Container whose name and open_mode needs to be fetched
  my $firstConId = $userScript ? 3 : 1;

  # NOTE: it is important that we do not fetch data from dictionary views 
  #       (e.g. DBA_PDBS) because views may yet to be created if this 
  #       procedure is used when running catalog.sql
  # NOTE: since a non-CDB may also have a row in CONTAINER$ with con_id#==0,
  #       we must avoid fetching a CONTAINER$ row with con_id==0 when looking 
  #       for Container names 
  #
  # Bug 18070841: "SET LINES" was added to make sure that both the PDB name 
  #               and the open_mode are on the same line.  500 is way bigger 
  #               than what is really needed, but it does not hurt anything.
  #
  # Bug 20307059: append mode to C:A:T:C:O:N<container-name> to ensure that the
  #               two do not get split across multiple lines

  my @GetConInfoStmts = (
    "connect ".$myConnect->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "set lines 500\n",
  );

  my @UpgPrioColExistsStmts = (
    "connect ".$myConnect->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || count(*) from sys.obj\$ o, sys.col\$ c ".
    "  where o.name = 'CONTAINER\$' and o.owner#=0 and o.type#=2 ".
    "    and o.obj# = c.obj# and c.name='UPGRADE_PRIORITY'\n/\n",
  );

  my $OrderByUpgPrioQuery = 
    "select \'C:A:T:C:O:N\' || v.name || ".
    "       decode(v.open_mode, 'MOUNTED', ' N', ' Y') ".
    "  from v\$containers v, sys.container\$ c ".
    "  where v.con_id = c.con_id# and v.con_id >= ".$firstConId.
    "  order by c.upgrade_priority, v.con_id\n/\n";

  my $OrderByConIdQuery = 
    "select \'C:A:T:C:O:N\' || v.name || ".
    "       decode(v.open_mode, 'MOUNTED', ' N', ' Y') ".
    "  from v\$containers v ".
    "  where v.con_id >= ".$firstConId.
    "  order by v.con_id\n/\n";

  # we would like to use CONTAINER$.UPGRADE_PRIORITY to order Containers 
  # whose names and open modes get fetched from V$CONTAINERS, but if we 
  # are upgrading to 12.2, that column nmay not have been created yet, in 
  # which case we have little choice but to order Containers by their CON_ID. 
  # In order to decide which query to use, we need to determine if 
  # CONTAINER$.UPGRADE_PRIORITY exists
  my ($upgPrioColCnt_ref, $Spool_ref) = 
    exec_DB_script(@UpgPrioColExistsStmts, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$upgPrioColCnt_ref || $#$upgPrioColCnt_ref != 0) {
    # count could not be obtained
    print_exec_DB_script_output("get_con_info(1)", $Spool_ref, 1);
    
    return 1;
  }

  # depending on the result returned by the above query, add an appropriate 
  # query to @GetConInfoStmts
  if ($$upgPrioColCnt_ref[0] eq "1") {
    push @GetConInfoStmts, $OrderByUpgPrioQuery;

    if ($debugOn) {
      log_msg("get_con_info: CONTAINER\$.UPGRADE_PRIORITY exists - use it and CON_ID to order Container names\n");
    }
  } else {
    push @GetConInfoStmts, $OrderByConIdQuery;

    if ($debugOn) {
      log_msg("get_con_info: CONTAINER\$.UPGRADE_PRIORITY does NOT exist - use CON_ID to order Container names\n");
    }
  }

  # each row consists of a Container name and Y/N indicating whether it is open
  my $rows_ref;

  ($rows_ref, $Spool_ref) = 
    exec_DB_script(@GetConInfoStmts, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$rows_ref || $#$rows_ref < 0) {
    # Container info could not be obtained
    print_exec_DB_script_output("get_con_info(2)", $Spool_ref, 1);
    
    return 1;
  }

  # split each row into a Container name and an indicator of whether is is open
  for my $row ( @$rows_ref ) {
    my ($name, $open) = split /\s+/, $row;
    push @$ConNames, $name if ($ConNames);
    $IsOpen->{$name} = $open;
  }

  return 0;
}

#
# get_fed_root_info - query V$CONTAINERS to obtain an indicator of whether 
#                     specified Container is a Federation Root and get its 
#                     CON ID
#
# parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - name of a supposed Federation Root (IN)
#   - reference to an indicator of whether a Container is a Federation 
#     Root (OUT)
#   - reference to CON ID of the specified Container (OUT)
#   - indicator of whether to produce debugging info
#
sub get_fed_root_info (\@$$$\$\$$) {
  my ($myConnect, $DoneCmd, $DoneFilePathBase, $ConName, $IsFedRoot, $ConId,
      $debugOn) = @_;

  my @GetFedRootInfoStmts = (
    "connect ".$myConnect->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || application_root, con_id from v\$containers where name = '".$ConName."'\n/\n",
  );

  # each row consists of a Container name and Y/N indicating whether it is open
  my ($rows_ref, $Spool_ref) = 
    exec_DB_script(@GetFedRootInfoStmts, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$rows_ref || $#$rows_ref != 0) {
    # App Root info could not be obtained
    print_exec_DB_script_output("get_fed_root_info", $Spool_ref, 1);
        
    return 1;
  }

  # split the row into a Federation Root indicator and CON ID (make sure a 
  # single row was fetched, let caller decide what to do if that was not the 
  # case)
  ($$IsFedRoot, $$ConId) = split /\s+/, $$rows_ref[0];

  return 0;
}

#
# get_fed_pdb_names - query V$CONTAINERS to get names of Federation PDBs 
#                     belonging to a Federation Root with specified CON ID
#
# parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - CON ID of a Federation Root (IN)
#   - reference to an array of Federation PDB names (OUT)
#   - indicator of whether to produce debugging info
#
sub get_fed_pdb_names (\@$$$\@$) {
  my ($myConnect, $DoneCmd, $DoneFilePathBase, $FedRootConId, $FedPdbNames,
      $debugOn) = @_;

  my @GetFedPdbNamesStmts = (
    "connect ".$myConnect->[0]."\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || name from v\$containers where application_root_con_id = $FedRootConId order by con_id\n/\n",
  );

  # each row consists of a Container name and Y/N indicating whether it is open
  my ($rows_ref, $Spool_ref) = 
    exec_DB_script(@GetFedPdbNamesStmts, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!@$rows_ref || $#$rows_ref < 0) {
    # App PDB names could not be obtained
    print_exec_DB_script_output("get_fed_pdb_names", $Spool_ref, 1);
    
    return 1;
  }

  for my $row ( @$rows_ref ) {
    push @$FedPdbNames, $row;
  }

  return 0;
}

#
# get_inst_procs - obtain number of SQL*Plus processes that can be started on 
#                  every OPEN instance
#
# parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns:
#   A hash consisting of 
#   <instance name> => <number of sessions that can connect to instance-name> 
#   entries;
#   return undef if pertinent info could not be found
#
sub get_inst_procs($$$$) {
  my ($myConnect, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @InstProcsStatements = (
    "connect $myConnect->[0]\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || inst.instance_name || \' \' || ".
           "least(cpu_param.value*2, sess_param.value-sess.num_sessions) ".
      "from gv\$instance inst, gv\$parameter cpu_param, ".
           "gv\$parameter sess_param, ".
           "(select inst_id, count(*) as num_sessions ".
              "from gv\$session group by inst_id) sess ".
      "where cpu_param.name=\'cpu_count\' ".
        "and sess_param.name=\'sessions\' ".
        "and sess_param.inst_id=sess.inst_id ".
        "and cpu_param.inst_id=sess.inst_id ".
        "and inst.inst_id=sess.inst_id\n/\n",
  );


  my ($retValues_ref, $Spool_ref) = 
    exec_DB_script(@InstProcsStatements, "C:A:T:C:O:N", 
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!$retValues_ref || !@$retValues_ref) {
    log_msg("get_inst_procs: no data was obtained\n");
    print_exec_DB_script_output("get_inst_procs", $Spool_ref, 1);
    return undef;
  }

  my %instProc;

  foreach my $row (@$retValues_ref) {
    # $row should consist of instance name and number of processes which will 
    # be allocated to run scripts on PDBs open on that instance
    my @vals = split(/  */, $row);

    if ($#vals != 1) {
      log_msg("get_inst_procs: data contained ".
              "an element ($row) with unexpected format\n");
      print_exec_DB_script_output("get_inst_procs", $Spool_ref, 1);
      return undef;
    }

    # remember how many sqlplus processes can be allocated to this instance
    my %numProcsHash = (NUM_PROCS => $vals[1]);

    # map instance name to a hash representing number of processes which can 
    # be started on this instance (to which an entry reprsenting an offset 
    # of the id of the first of these processes in @catcon_ProcIds will be 
    # added when we start these processes in start_processes)
    $instProc{$vals[0]} = \%numProcsHash;

    if ($debugOn) {
      log_msg("get_inst_procs: added $vals[0] => (NUM_PROCS => ".
              $vals[1].") entry to the hash\n");
    }
  }

  return \%instProc;
}

# handler for SIGINT while resetting PDB mode(s)
sub handle_sigint_for_pdb_mode () {
  log_msg("Caught SIGINT while changing PDB mode(s)\n");

  # reregister SIGINT handler in case we are running on a system where 
  # signal(3) acts in "the old unreliable System V way", i.e. clears the 
  # signal handler
  $SIG{INT} = \&handle_sigint_for_pdb_mode;

  return;
}

#
# reset_seed_pdb_mode - close PDB$SEED on all instances and open it in the 
#                       specified mode
#
# parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - mode in which PDB$SEED is to be opened
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# returns:
# - 0 if it appears that all went well
# - 1 if an error was encountered
# - 2 if PDB$SEED could not be opened in the requested mode (which would have 
#     to be something other that READ ONLY) because the CDB is open READ ONLY 
#     (bug 23106360)
#
# Bug 14248297: close PDB$SEED on all RAC instances. If the PDB$SEED is to be
# opened on all instances, then the 'seedMode' argument must specify it.
#
sub reset_seed_pdb_mode (\@$$$$) {
  my ($myConnect, $seedMode, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  # if resetting PDB$SEED to a mode other than READ ONLY, make sure the CDB 
  # itself is not open READ ONLY
  if ($seedMode !~ /READ ONLY/) {
    my $cdbOpenMode = 
      get_CDB_open_mode(@$myConnect, $DoneCmd, 
                        $DoneFilePathBase, $debugOn);

    if (!(defined $cdbOpenMode)) {
      log_msg("reset_seed_pdb_mode: unexpected error in ".
              "get_CDB_open_mode\n");
      return 1;
    }

    if ($cdbOpenMode eq 'READ ONLY') {
      log_msg("reset_seed_pdb_mode: WARNING: PDB\$SEED can not be opened in ".
              "$seedMode because the CDB is open READ ONLY");
      return 2;
    }
  }

  my @ResetSeedModeStatements = (
    "connect ".$myConnect->[0]."\n",
    qq#alter session set "_oracle_script"=TRUE\n/\n#,
    "alter pluggable database pdb\$seed close immediate instances=all\n/\n",
    "alter pluggable database pdb\$seed $seedMode\n/\n",
  );

  # temporarily reset $SIG{INT} to avoid dying while PDB$SEED is in transition
  my $saveIntHandlerRef = $SIG{INT};
  
  $SIG{INT} = \&handle_sigint_for_pdb_mode;

  if ($debugOn) {
    log_msg("reset_seed_pdb_mode: temporarily reset SIGINT handler\n");
  }

  my ($ignoreOutputRef, $spoolRef) = 
    exec_DB_script(@ResetSeedModeStatements, undef, $DoneCmd, 
                   $DoneFilePathBase, $debugOn);

  print_exec_DB_script_output("reset_seed_pdb_mode", $spoolRef, 0);

  # 
  # restore INT signal handler 
  #
  $SIG{INT} = $saveIntHandlerRef;

  if ($debugOn) {
    log_msg("reset_seed_pdb_mode: restored SIGINT handler\n");
  }

  return 0;
}

# values that can be used when requesting that PDBs be open in a certain 
# mode:
#   - 0 - leave it alone (caller responsible for opening it in the correct 
#         mode before calling catconExec and reopening it in READ ONLY 
#         mode before exiting)
#   - 1 - PDB$SEED must be open READ WRITE; if it is not, catconExec will 
#         reopen it READ WRITE and catconWrapUp will restore its mode 
#   - 2 - PDB$SEED must be open READ ONLY; if it is not, catconExec will 
#         reopen it READ ONLY and catconWrapUp will restore its mode 
#   - 3 - PDB$SEED must be open UPGRADE; if it is not open MIGRATE 
#         (currently we do not distinguish between UPGRADE and DOWNGRADE), 
#         catconExec will reopen it UPGRADE and catconWrapUp will restore 
#         its mode
#   - 4 - PDB$SEED must be open DOWNGRADE; if it is not open MIGRATE 
#         (currently we do not distinguish between UPGRADE and DOWNGRADE), 
#         catconExec will reopen it UPGRADE and catconWrapUp will restore 
#         its mode
use constant CATCON_PDB_MODE_UNCHANGED  => 0;
use constant CATCON_PDB_MODE_READ_WRITE => 1;
use constant CATCON_PDB_MODE_READ_ONLY  => 2;
use constant CATCON_PDB_MODE_UPGRADE    => 3;
use constant CATCON_PDB_MODE_DOWNGRADE  => 4;
# no value was supplied
use constant CATCON_PDB_MODE_NA         => 99;

# minimum/maximum CATCON_PDB_MODE_* values representing PDB mode
use constant CATCON_PDB_MODE_MIN        => CATCON_PDB_MODE_UNCHANGED;
use constant CATCON_PDB_MODE_MAX        => CATCON_PDB_MODE_DOWNGRADE;

# value of x$con.state if a PDB is mounted
use constant CATCON_PDB_MODE_MOUNTED    => 0;

# hash mapping CATCON_PDB_MODE_* value corresponding to a valid PDB mode 
# (READ WRITE, READ ONLY, UPGRADE) to a corresponding string

use constant PDB_MODE_CONST_TO_STRING  => {
    # caller cannot request this mode; however, a PDB may be in MOUNTED mode, 
    # and if we need to display a string representing that state, this mapping
    # will be used
    0   => "MOUNTED",

    # PDB modes that a user can request
    1   => "READ WRITE",
    2   => "READ ONLY",
    3   => "UPGRADE",
    # in RDBMS, we do not distinguish between UPGRADE and DOWNGRADE
    4   => "UPGRADE",
};

# and the inverse map (used to convert a string representing a mode into a
# corresponding CATCON_PDB_MODE_* value
use constant PDB_STRING_TO_MODE_CONST  => {
    # if no value was specified, the option flag will remain set to 0
    0            => CATCON_PDB_MODE_NA,

    'MOUNTED'    => CATCON_PDB_MODE_MOUNTED,
    'UNCHANGED'  => CATCON_PDB_MODE_UNCHANGED,
    'READ WRITE' => CATCON_PDB_MODE_READ_WRITE,
    'READ ONLY'  => CATCON_PDB_MODE_READ_ONLY,
    'MIGRATE'    => CATCON_PDB_MODE_UPGRADE,
    'UPGRADE'    => CATCON_PDB_MODE_UPGRADE,
    'DOWNGRADE'  => CATCON_PDB_MODE_DOWNGRADE,
};

# environment variable which may be set to a CATCON_PDB_MODE_* constant 
# representing a mode in which ALL pdbs should be opened before running 
# scripts against them
use constant ALL_PDB_MODE_ENV_TAG => "CATCON_ALL_PDB_MODE";

# validate PDB mode supplied by the caller
sub valid_pdb_mode($) {
  my ($mode) = @_;

  return ($mode =~ /[0-9]+/ && $mode >= catcon::CATCON_PDB_MODE_MIN && 
          $mode <= catcon::CATCON_PDB_MODE_MAX);
}


# while catcon.pl will prevent a user from specifying both --pdb_seed_mode 
# and --force_open_mode, I cannot guarantee that other users of catcon.pm 
# subroutines will do the same, so this subroutine will return mode in 
# which PDB$SEED should be open based on values stored in $catcon_SeedPdbMode 
# and $catcon_AllPdbMode
#
# NOTE: when determining mode in which PDB$SEED and the rest of PDBs should 
#       be open, it's important that we call seed_pdb_mode_to_use before 
#       all_pdb_mode_to_use() since the latter will set the "all PDBs" mode 
#       to a value other than CATCON_PDB_MODE_NA (even if no value for 
#       "all PDBs" mode was specified) which can then influence the mode in 
#       which we will open PDB$SEED
sub seed_pdb_mode_to_use($$) {
  my ($seed_mode, $all_pdb_mode) = @_;

  # if the caller specified mode in which all PDBs should be open or 
  # specified that they should be left in whatever mode they were, return 
  # that value
  if ($all_pdb_mode != CATCON_PDB_MODE_NA) {
    return $all_pdb_mode;
  }

  # if $all_pdb_mode was not set (by calling catconPdbMode()), see if
  # $CATCON_ALL_PDB_MODE was set to a valid value
  if (exists $ENV{ALL_PDB_MODE_ENV_TAG} && 
      valid_pdb_mode($ENV{ALL_PDB_MODE_ENV_TAG})) {
    return $ENV{ALL_PDB_MODE_ENV_TAG};
  }

  # PDB mode for PDB$SEED defaults to CATCON_PDB_MODE_READ_WRITE
  return ($seed_mode != CATCON_PDB_MODE_NA) ? $seed_mode 
                                            : CATCON_PDB_MODE_READ_WRITE;
}

# determine mode in which all PDBs need to be opened before running scripts
sub all_pdb_mode_to_use($) {
  my ($all_pdb_mode) = @_;

  # if $all_pdb_mode was set (by calling catconPdbMode()), use that value
  if ($all_pdb_mode != CATCON_PDB_MODE_NA) {
    return $all_pdb_mode;
  }

  # if $all_pdb_mode wasn't set, see if $CATCON_ALL_PDB_MODE was set to a 
  # valid value
  if (exists $ENV{ALL_PDB_MODE_ENV_TAG} && 
      valid_pdb_mode($ENV{ALL_PDB_MODE_ENV_TAG})) {
    return $ENV{ALL_PDB_MODE_ENV_TAG};
  }
  
  # if all else fails, PDB mode for all PDBs defaults to 
  # CATCON_PDB_MODE_UNCHANGED
  return CATCON_PDB_MODE_UNCHANGED;
}

# this subroutine will dump contents of a 
# instance-name => ref @pdb-names hash that 
# is used to describe which PDBs are open on which instances
sub inst_pdbs_hash_dump($$$) {
  my ($instPdbsHashRef, $funcName, $hashName) = @_;

  log_msg("$funcName: contents of $hashName hash:\n");

  if (!%$instPdbsHashRef) {
    log_msg("\tnone\n");

    return;
  }

  foreach my $inst (keys %$instPdbsHashRef) {
    log_msg("instance = $inst\n");

    my $pdbs = $instPdbsHashRef->{$inst};

    foreach my $pdb (@$pdbs) {
      log_msg("\tPDB = $pdb\n");
    }
  }
}

# this subroutine will dump contents of a 
# instance-name => number-of-processes hash that 
# is used to describe how many processes can be opened on each instance
sub inst_proc_hash_dump($$$) {
  my ($instProcHashRef, $funcName, $hashName) = @_;

  log_msg("$funcName: contents of $hashName hash:\n");

  foreach my $inst (keys %$instProcHashRef) {
    log_msg("instance $inst - ".$instProcHashRef->{$inst}." processes\n");
  }
}

# this subroutine will dump contents of a 
# (mode.' '. restricted) => ref {instance-name => ref @pdb-names} hash that 
# is used to describe how to reset PDB modes
sub pdb_mode_hash_dump($$$) {
  my ($pdbModeHashRef, $funcName, $hashName) = @_;

  log_msg("$funcName: contents of $hashName hash:\n");

  foreach my $modeAndRestr (keys %$pdbModeHashRef) {
    my @modeAndRestrArr = split(/  */, $modeAndRestr);
    my $modeString = PDB_MODE_CONST_TO_STRING->{$modeAndRestrArr[0]};
    my $restrString = 
      ($modeAndRestrArr[1] == 0) ? "unrestricted" : "restricted";

    log_msg("\tmode = $modeString $restrString\n");
      
    my $instToPdbMapRef = $pdbModeHashRef->{$modeAndRestr};

    foreach my $inst (keys %$instToPdbMapRef) {
      log_msg("\t  instance = $inst\n");

      my $pdbs = $instToPdbMapRef->{$inst};

      foreach my $pdb (@$pdbs) {
        log_msg("\t    PDB = $pdb\n");
      }
    }
  }
}

#
# curr_pdb_mode_info - obtain
#                      - numeric value corresponding to the mode in which the
#                        specified PDB is open:
#                          0 - MOUNTED, 1 - READ WRITE, 2 - READ ONLY
#                          3 - MIGRATE
#                      - an indicator of whether it is open RESTRICTED:
#                          0 - NO, 1 - YES, 2 - if PDB is MOUNTED
#                        and
#                      - the name of an instance to which this info
#                        applies (meaning that in RAC, multiple rows may be
#                        returned)
#
# parameters:
#   - connect strings - [0] used to connect to a DB,
#                       [1] can be used to produce diagnostic message
#   - PDB name
#   - a command to create a file whose existence will indicate that the
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns:
#   A hash consisting of instance-name => (mode, restricted) entries;
#   return undef if info pertaining to the specified PDB could not be found
#
sub curr_pdb_mode_info (\@$$$$) {
  my ($myConnect, $pdbName, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @SeedModeStateStatements = (
    "connect $myConnect->[0]\n",
    "set echo off\n",
    "set heading off\n",
    "select \'C:A:T:C:O:N\' || p.open_mode || ',' || ".
                              "nvl(p.restricted, 'NO') || ',' || ".
                              "i.instance_name ".
      "from gv\$pdbs p, gv\$instance i ".
      "where p.name='".$pdbName."' and p.inst_id = i.inst_id\n/\n",
  );


  my ($retValues_ref, $Spool_ref) =
    exec_DB_script(@SeedModeStateStatements, "C:A:T:C:O:N",
                   $DoneCmd, $DoneFilePathBase, $debugOn);

  if (!$retValues_ref || !@$retValues_ref) {
    log_msg("curr_pdb_mode_info: no data was obtained for PDB $pdbName\n");
    print_exec_DB_script_output("curr_pdb_mode_info", $Spool_ref, 1);
    return undef;
  }

  my %currPdbModeInfo;

  foreach my $row (@$retValues_ref) {
    # $row should consist of PDB mode (string), an indicator of whether
    # the PDB is open restricted, and an instance name, separated by commas
    my @vals = split(/,/, $row);

    if ($#vals != 2) {
      log_msg("curr_pdb_mode_info: data returned for PDB $pdbName contained ".
              "an element ($row) with unexpected format\n");
      print_exec_DB_script_output("curr_pdb_mode_info", $Spool_ref, 1);
      return undef;
    }

    # map instance name to (mode restricted)
    $currPdbModeInfo{$vals[2]} =
      PDB_STRING_TO_MODE_CONST->{$vals[0]}." ".($vals[1] eq "NO" ? 0 : 1);

    if ($debugOn) {
      log_msg("curr_pdb_mode_info: added $vals[2] => ".
              $currPdbModeInfo{$vals[2]}."(".$vals[0]." ".$vals[1].") ".
              "element to the hash\n");
    }
  }

  return \%currPdbModeInfo;
}

# force PDB(s) to be open in specified mode
#
# Parameters:
#   - array of PDB names (IN) 
#   - index of the first PDB to be processed (IN)
#   - index of the last PDB to be processed (IN)
#   - requested mode (IN)
#   - (mode, restricted) => {instance-name => @pdb-names} hash describing PDBs
#     whose modes were changed to the requested mode by this subroutine and 
#     which will need to be reset in catconWrapUp() (OUT)
#   - diagnostic info indicator (IN)
#   - verbose output indicator (IN)
#   - log file suffix - used to avoid conflicts in log file names (IN)
#   - log file path base (IN)
#   - connect string array (IN)
#   - "done" command (IN)
#   - indicator of whether _ORACLE_SCRPT needs to be set before (and cleared 
#     after) resetting PDB mode(s) (IN) 
#   - hash used to keep track of all PDBs which we know to be opened in 
#     requested mode and which may be updated to reflect additional PDBs 
#     specified in @$Containers[$firstPDB..$lastPDB] (IN)
#   - hash used to keep track of whether a given PDB is open; may be updated 
#     to reflect new PDBs opened in the desired mode (IN/OUT)
#   - RDBMS version; used to ensure that SQL statements that we generate are 
#     compatible with the version of the RDBMS to which we are connected (IN)
#   - an indicator of whether there are more PDBs to be opened following this 
#     invocation (meaning that if we are opening PDBs on multiple instances, 
#     we will close specified PDBs during this invocation but delay reopening 
#     them until there are no more PDBs to process at which point we will 
#     reopen all of them)
#   - a reference to a hash mapping names of instances to the number of 
#     processes allocated to each of these instances which will be set ONLY IF 
#     we were told to process PDBs using all available instances
#   - a reference to a hash mapping instance names to instance connect strings
#
# Returns:
#   (status, inst-to-PDB-Hash_ref)
#   where 
#     status may be one of
#       -1 - no PDBs had to be reopened, with inst-to-PDB-Hash_ref undefined
#        0 - some/all PDBs had to be reopened, with inst-to-PDB-Hash_ref, if 
#            defined, referencing a hash that can be used to determine which 
#            PDBs are open on each instance
#        1 - an unexpected error was encountered, with inst-to-PDB-Hash_ref 
#            undefined
sub force_pdb_modes(\@$$$$$$$$\@$$\%\%$$$$) {
  my ($Containers, $firstPDB, $lastPDB, $reqMode, $revertPdbModes,
      $debugOn, $verbose, $logFileSuffix, $logFilePathBase, $connString, 
      $doneCmd, $oracleScript, $pdbsOpenInReqMode, $isConOpen, 
      $dbmsVersion, $MorePdbs, $InstProcMap_ref, $InstConnStrMap_ref) = @_;

  # first check if @$Containers[$firstPDB..$lastPDB] includes any PDBs we 
  # haven't seen before
  my @newPDBs = 
    grep {! exists $pdbsOpenInReqMode->{$_}} @$Containers[$firstPDB..$lastPDB];
  
  if ($#newPDBs < 0) {
    # no new PDBs to process
    if ($debugOn) {
      log_msg("force_pdb_modes: PDBs whose mode has been previously verified ".
              "to match requested mode:\n\t".
              join(', ', sort (keys %$pdbsOpenInReqMode))."\n\n");
      log_msg("force_pdb_modes: PDBs whose names were supplied by the ".
              "caller:\n\t".join(', ', sort @$Containers[$firstPDB..$lastPDB]).
              "contained no new PDBs ... returning.\n");
    }

    return (-1, undef);
  } elsif ($debugOn) {
    log_msg("force_pdb_modes: PDBs whose mode has been previously verified ".
            "to match requested mode:\n\t".
            join(', ', sort (keys %$pdbsOpenInReqMode))."\n\n");
    log_msg("force_pdb_modes: PDBs whose names were supplied by the ".
            "caller:\n\t".join(', ', sort @$Containers[$firstPDB..$lastPDB]).
            "\n\n");
    log_msg("force_pdb_modes: PDBs which will have to be processed:\n\t".
            join(', ', @newPDBs)."\n\n");
  }

  my $doneFileNamePrefix = 
    done_file_name_prefix($logFilePathBase, $logFileSuffix);

  # Bug 20193612: if we decided to run scripts on all available instances 
  # (i.e. if %$InstProcMap_ref has been populated) we will try to spread 
  # PDBs across all available instances
  my $useMultipleInstances = 0;

  if (%$InstProcMap_ref) {
    if ($debugOn) {
      log_msg("force_pdb_modes: processing PDBs using all available ".
              "instances\n");
    }

    # number of instances on which the CDB is open
    my $numOpenInstances = scalar keys %$InstProcMap_ref;

    if ($numOpenInstances > 1) {
      $useMultipleInstances = 1;

      if ($debugOn) {
        log_msg <<msg;
force_pdb_modes: CDB is open on $numOpenInstances instances all of 
    which may be used to process PDBs
msg
      }
    } elsif ($debugOn) {
      log_msg <<msg;
force_pdb_modes: CDB is open on 1 instance which will be used to process PDBs
msg
    }
  }

  # string corresponding to the mode specified by the caller
  my $reqModeString = PDB_MODE_CONST_TO_STRING->{$reqMode};

  # if a PDB is open in UPGRADE mode and we decide to leave it alone 
  # (i.e. because the requested mode is UPGRADE, DOWNGRADE, or READ WRITE), 
  # we need to remember that this PDB should not be open on any other instance
  # (since a PDB can be open in UPGRADE mode on one instance)
  my %pdbsOpenInUpgradeMode;
          
  if ($debugOn) {
    log_msg <<msg;
force_pdb_modes: check if specified PDB(s) need to be reopened
    in $reqModeString ($reqMode) mode
msg
  }
   
  for (my $pdbIdx = 0; $pdbIdx <= $#newPDBs; $pdbIdx++) {
    # current PDB name
    my $pdbName = $newPDBs[$pdbIdx];

    if ($debugOn) {
      log_msg("force_pdb_modes: determine mode(s) in which PDB $pdbName is ".
              "open on all instances.\n");
    }

    # Bug 18011217: append _catcon_$logFileSuffix to $logFilePathBase to avoid 
    # conflicts with other catcon processes running on the same host
    my $currPdbModeInfo = 
      curr_pdb_mode_info(@$connString, $pdbName, $doneCmd, 
                         $doneFileNamePrefix, $debugOn);
    if (! (defined $currPdbModeInfo)) {
      log_msg("force_pdb_modes: unexpected error in curr_pdb_mode_info\n");
      return (1, undef);
    }

    # If on some instance the PDB is open in a mode different from that 
    # requested by the caller, remember to which mode it needs to be restored 
    # when we are done
    my @instances = sort keys %$currPdbModeInfo;

    if ($debugOn) {
      log_msg("force_pdb_modes: curr_pdb_mode_info has returned modes in ".
              "which PDB\n".
              "$pdbName is open on instances (".join(', ', @instances).")\n");
    }

    foreach my $inst (@instances) {
      my @modeAndRestr = split(/  */, $currPdbModeInfo->{$inst});
      my $currMode = $modeAndRestr[0];
      my $currModeString = PDB_MODE_CONST_TO_STRING->{$currMode};
      my $currRestrString = 
        ($modeAndRestr[1] == 0) ? "unrestricted" : "restricted";

      if ($debugOn) {
        log_msg <<msg;
force_pdb_modes: on instance $inst, PDB $pdbName is open
    $currModeString, $currRestrString
msg
      }

      #
      # NOTE: if the PDB is open in UPGRADE or DOWNGRADE mode, $currMode 
      # will be set to 3 since we don't currently distinguish between 
      # UPGRADE and DOWNGRADE
      #
      # Bug 18606911: if a PDB is open in UPGRADE mode and the caller 
      # asked that it be open in READ WRITE mode, there is no need to 
      # change the PDB's mode
      #
      # Bug 20193612: if we decided to spread PDBs across multiple 
      # OPEN instances, we will ignore whether the PDB is already open in the 
      # requested mode on some instance.  It is much simpler to close it 
      # across the board and then open it on the instance of our choosing
      if (   $useMultipleInstances ||
             ($reqMode != $currMode &&
              !(   (   $reqMode == CATCON_PDB_MODE_DOWNGRADE 
                    || $reqMode == CATCON_PDB_MODE_READ_WRITE)
                && $currMode == CATCON_PDB_MODE_UPGRADE))) {

        if ($debugOn) {
          log_msg <<msg;
force_pdb_modes: on instance $inst, PDB $pdbName will need 
    to be closed and possibly reopened in $reqModeString mode.
msg
        }

        # determine if %$revertPdbModes hash contains an entry for mode and 
        # restricted flag corresponding to $inst
        if (exists $revertPdbModes->{$currPdbModeInfo->{$inst}}) {
          # locate hash mapping instance names to PDB names for 
          # (mode restricted) pair
          my $instToPdbMapRef = $revertPdbModes->{$currPdbModeInfo->{$inst}};

          if ($debugOn) {
            log_msg <<msg;
force_pdb_modes: revertPdbModes contains an entry for 
    ($currModeString, $currRestrString) pair
msg
          }

          # if the mapping already includes an element for the current 
          # instance, add the current PDB to that mapping; otherwise, create 
          # a new mapping and add a reference to it to %$instToPdbMapRef
          if (exists $instToPdbMapRef->{$inst}) {
            my $pdbArrRef = $instToPdbMapRef->{$inst};

            if ($debugOn) {
              log_msg <<msg;
force_pdb_modes: revertPdbModes entry for ($currModeString, $currRestrString) 
    contains an entry for instance $inst; will add PDB $pdbName to 
    (@$pdbArrRef.)
msg
            }

            push @$pdbArrRef, $pdbName;
          } else {
            my @pdbArr;

            push @pdbArr, $pdbName;
            $instToPdbMapRef->{$inst} = \@pdbArr;

            if ($debugOn) {
              log_msg <<msg;
force_pdb_modes: revertPdbModes entry for ($currModeString, $currRestrString) 
    did not contain an entry for instance $inst; added $inst => @pdbArr to it
msg
            }
          }
        } else {
          my %instToPdbMap;
          my @pdbNameArr;

          # construct a hash mapping instance name to array consisting of the
          # PDB name
          push @pdbNameArr, $pdbName;
          $instToPdbMap{$inst} = \@pdbNameArr;

          # add to %$revertPdbModes an entry mapping (mode restricted) to 
          # a reference to the above hash 
          $revertPdbModes->{$currPdbModeInfo->{$inst}} = \%instToPdbMap;

          if ($debugOn) {
            log_msg <<msg;
force_pdb_modes: revertPdbModes did not contain an entry for 
    ($currModeString, $currRestrString) pair which was then created
    and initialized with $inst => @pdbNameArr
msg
          }
        }
      } elsif ($currMode == CATCON_PDB_MODE_UPGRADE) {
        # remember that this PDB is open in UPGRADE mode so we should not try 
        # to open it on any other instance
        $pdbsOpenInUpgradeMode{$pdbName} = $inst;

        if ($debugOn) {
          log_msg <<msg;
force_pdb_modes: PDB $pdbName is open in UPGRADE mode on instance $inst.
    It will remain open on that instance and will not be open on any other 
    instance, so we skip checking its status on any remaining instance
msg
        }

        # if the PDB is open in UPGRADE mode on this instance, it is 
        # guaranteed to be closed on all other instances, and we will not be 
        # trying to open it there, so there is no reason to check whether 
        # this PDB is open on remaining instances or to remember its state 
        # since it will not (and cannot) be changed 
        last;
      }
    }
  }

  # if there are no PDBs whose mode needs to be reset, we are done
  if (!%$revertPdbModes) {
    if ($debugOn) {
      log_msg("force_pdb_modes: no PDB modes need changing ... returning.\n");
    }

    # we need to update the hash of PDBs which are known to be opened in the 
    # requested mode
    %$pdbsOpenInReqMode = (%$pdbsOpenInReqMode, map {($_, 1)} @newPDBs);

    return (-1, undef);
  }

  if ($debugOn) {
    pdb_mode_hash_dump($revertPdbModes, "force_pdb_modes", "revertPdbModes");
  }

  if ($debugOn) {
    log_msg("force_pdb_modes: constructing hash to reset PDB modes\n");
  }

  # next we use contents of %$revertPdbModes to construct a hash which will 
  # be passed to reset_pdb_modes() to reopen in required mode all PDBs that 
  # need to be so reopened.
  #
  # To that end, we will examine every entry in %$revertPdbModes and populate
  # hash mapping instance name to an array of PDB names
  #
  # When we finish processing %$revertPdbModes, we will construct a hash 
  # that looks like $catcon_RevertUserPdbModes (but has a single entry - 
  # specifying the requested mode) and pass it to reset_pdb_modes

  # will be populated with inst => PDB array entries as we traverse 
  # %$revertPdbModes
  my %instToPdbMap;

  # (mode, restricted) pairs of PDBs whose mode needs to be changed
  my @modesAndRestr = sort keys %$revertPdbModes;

  if ($debugOn) {
    log_msg("force_pdb_modes: (mode, restricted) pairs of PDBs whose mode ".
            "needs to be changed: ".join(', ', @modesAndRestr)."\n");
  }

  foreach my $modeAndRestr (@modesAndRestr) {
    if ($debugOn) {
      log_msg("force_pdb_modes: processing (mode, restricted) pair: ".
              "$modeAndRestr\n");
    }

    # instances and PDBs on those instances whose mode needs to be reset
    my $instToPdbMapRef = $revertPdbModes->{$modeAndRestr};

    my @instances = sort keys %$instToPdbMapRef;

    if ($debugOn) {
      log_msg("force_pdb_modes: instances on which some PDBs are opened ".
              "$modeAndRestr: ".join(', ', @instances)."\n");
    }

    foreach my $inst (@instances) {
      if ($debugOn) {
        log_msg("force_pdb_modes: processing PDBs on instance $inst\n");
      }

      my $pdbsToConsider = $instToPdbMapRef->{$inst};
      if ($debugOn) {
        log_msg("force_pdb_modes: pdbs from instToPdbMapRef ".
                "whose mode may need to be changed on instance $inst: ".
                join(', ', @{$pdbsToConsider})."\n");
      }

      # PDBs whose mode will need to be reset; will be set to 
      # @$pdbsToConsider minus <PDBs represented in %pdbsOpenInUpgradeMode>
      my @pdbsToReopen;

      if (%pdbsOpenInUpgradeMode) {
        # if some PDBs were open in UPGRADE mode and we decided to leave them
        # alone, before we try to add PDB names found in @$pdbsToConsider to 
        # @$pdbsRef, make sure none of them are already open in UPGRADE 
        # mode or have been previously added to $instToPdbMap on some 
        # instance and so should not be reopened on this instance
        if ($debugOn) {
          if ($reqMode == CATCON_PDB_MODE_UPGRADE) {
            log_msg("force_pdb_modes: opening PDBs in $reqModeString ".
                    "mode;\n");
          } else {
            log_msg("force_pdb_modes: opening PDBs in $reqModeString ".
                    "mode, but some PDBs will be left in UPGRADE mode;\n");
          }

          log_msg("check if any PDBs in pdbsToConsider should NOT have ".
                  "their mode reset\n");
        }

        # names of PDBs which should not be reopened on this instance; will 
        # be set to PDBs that both are in @$pdbsToConsider and have entries 
        # in %pdbsOpenInUpgradeMode
        my @pdbsNotToReopen = 
          grep {exists $pdbsOpenInUpgradeMode{$_}} @$pdbsToConsider;

        if ($#pdbsNotToReopen >= 0) {
          # at least some of the PDBs in @$pdbsToConsider should not be 
          # reopened
          if ($#pdbsNotToReopen == $#$pdbsToConsider) {
            if ($debugOn) {
              log_msg <<msg;
force_pdb_modes: all PDBs in pdbsToConsider are open for UPGRADE - no PDBs 
    need to be reopened in instance $inst
msg
            }
              
            next; # move on to the next instance
          } else {
            @pdbsToReopen = 
              grep {! exists $pdbsOpenInUpgradeMode{$_}} @$pdbsToConsider;

            if ($debugOn) {
              log_msg("force_pdb_modes: some PDBs (".
                      join(', ', @pdbsNotToReopen).")\n".
                      "in pdbsToConsider are open for UPGRADE;\n".
                      "remaining PDBs (".
                      join(', ', @pdbsToReopen).")\n". 
                      "need to be reopened in instance $inst\n");
            }
          }
        } else {
          if ($debugOn) {
            log_msg("force_pdb_modes: no PDBs from pdbsToConsider are ".
                    " open in UPGRADE mode, so PDBs (".
                    join(', ', @{$pdbsToConsider}).") ". "will be reopened\n");
          }

          @pdbsToReopen = @$pdbsToConsider;
        }
      } else {
        if ($debugOn) {
          log_msg("force_pdb_modes: no PDBs are open in UPGRADE mode, ".
                  "so all PDBs (".join(', ', @{$pdbsToConsider}).
                  ") will be reopened\n");
        }

        @pdbsToReopen = @$pdbsToConsider;
      }

      if (exists $instToPdbMap{$inst}) {
        if ($debugOn) {
          log_msg("force_pdb_modes: instToPdbMap contains an entry for ".
                  "inst=$inst\n");
        }

        # add list of PDBs from $revertPdbModes to the list in %instToPdbMap
        my $pdbsRef = $instToPdbMap{$inst};
        if ($debugOn) {
          log_msg("force_pdb_modes: pdbs in instToPdbMap entry: ".
                  join(', ', @{$pdbsRef})."\n");
        }

        push(@$pdbsRef, @pdbsToReopen);

        if ($debugOn) {
          log_msg("force_pdb_modes: resulting pdbs in instToPdbMap entry: ".
                  join(', ', @{$pdbsRef})."\n");
        }
      } else {
        if ($debugOn) {
          log_msg("force_pdb_modes: pdbs from pdbsToReopen (".
                  join(', ', @pdbsToReopen).") will be assigned to ".
                  "instToPdbMap{$inst}\n");
        }

        $instToPdbMap{$inst} = \@pdbsToReopen;
      }
    }
  }

  # this hash will be used to map (mode.' '.0) to a reference to a hash 
  # mapping instance names to a reference to array of names of PDBs on these 
  # instances whose [PDBs'] mode needs to be reset
  my %resetPdbModes;
  
  my $ret;

  if ($useMultipleInstances) {
    # Bug 20193612: if we will be processing PDBs using multiple RAC 
    # instances, we need to first close all PDBs on all Instances where they 
    # are open and then open them evenly across the available instances
    $resetPdbModes{CATCON_PDB_MODE_MOUNTED.' '.0} = \%instToPdbMap;

    if ($debugOn) {
      log_msg("force_pdb_modes: processing PDBs using multiple instances; ".
              "first all PDBs will be closed\n");
      log_msg("\nforce_pdb_modes: calling reset_pdb_modes\n");
    }

    # close all PDBs

    # Bug 18011217: append _catcon_$logFileSuffix to 
    # $logFilePathBase to avoid conflicts with other catcon 
    # processes running on the same host
    $ret = 
      reset_pdb_modes($connString, \%resetPdbModes, $doneCmd, 
                      $doneFileNamePrefix, 
                      $oracleScript, $dbmsVersion, 0, $InstConnStrMap_ref,
                      $debugOn);

    # we do not expect reset_pdb_modes to return 2 because we did not try to 
    # open any PDBs
    if ($ret == 1) {
      log_msg("force_pdb_modes: unexpected error reported by ".
              "reset_pdb_modes\n");
      return (1, undef);
    }

    # if we will be processing PDBs using multiple instances, and there are 
    # more PDBs to be processed (in a subsequent invocation), we will avoid 
    # opening PDBs during this invocation because they will need to be closed 
    # and reopened during the subsequent invocation
    if ($MorePdbs) {
      if ($debugOn) {
        log_msg("force_pdb_modes: having closed specified PDBs, delay\n".
                "\treopening them until the final invocation of this ".
                "subroutine\n");
      }

      # %instToPdbMap needs to be undefined to let the caller know that 
      # PDBs specified by the caller were not really open
      undef %instToPdbMap;

      # Note that by jumping to this label we temporarily cause 
      # %$pdbsOpenInReqMode and %$isConOpen to reflect reality that is yet 
      # to happen, i.e. that PDBs represented by @newPDBs have been opened in 
      # the requested mode.  The reality will catch up once this subroutine is 
      # called to reopen remaining PDBs.
      goto force_pdb_modes_done;
    }

    # delete previously added entry from %resetPdbModes since we will be 
    # calling reset_pdb_modes once more (to open PDBs on available instances), 
    # and we do not want it to try to close them
    delete $resetPdbModes{CATCON_PDB_MODE_MOUNTED.' '.0};

    # next we will purge %instToPdbMap of PDB names for all instances and 
    # then distribute PDBs across instances taking into consideration number 
    # of processes allocated for every instance

    if ($debugOn) {
      log_msg("force_pdb_modes: remove PDB names from instToPdbMap\n");
    }

    foreach my $inst (sort keys %instToPdbMap) {
      $#{$instToPdbMap{$inst}} = -1;
      if ($debugOn) {
        log_msg("\tclear PDBs associated with instance $inst ".
                "from instToPdbMap\n");
      }
    }
    
    # it is possible that some (but definitely not all) entries in
    # %$InstProcMap_ref represent the fact that no processes could be 
    # allocated to an instance. For purposes of assigning PDBs to instances, 
    # it will simplify things greatly if such entries were eliminated before 
    # attempting the assignment.  To that end, we will construct a hash mapping
    # names of instances which have some processes allocated for them to the
    # number of processes allocated to each of these instances
    my %procsOnAvailInst;
    
    foreach my $inst (keys %$InstProcMap_ref) {
      if (! exists $InstProcMap_ref->{$inst}->{NUM_PROCS}) {
        log_msg("force_pdb_modes: InstProcMap_ref for instance $inst has ".
                "no NUM_PROCS entry\n");
        return (1, undef);
      }

      if ($InstProcMap_ref->{$inst}->{NUM_PROCS} > 0) {
        $procsOnAvailInst{$inst} = $InstProcMap_ref->{$inst}->{NUM_PROCS};
      }
    }

    # as we assign PDBs to instances, we want to decrement number of processes 
    # still available for that instance so that instances with fewer processes
    # allocated for them get fewer PDBs assigned to them.  To accomplish this, 
    # we make a copy of %procsOnAvailInst which we can modify without losing
    # info about the number of processes allocated to every open instance
    my %tempInstProcs;
    
    # names of instances for which sqlplus processes have been allocated
    my @instances;

    # index into @instances
    my $instIdx;

    # PDBs that need to be opened belong to a union of key %$pdbsOpenInReqMode 
    # (i.e. PDBs which were open or were discovered to be open in the 
    # requested mode during previous invocation of force_pdb_modes) and 
    # @newPDBs (PDBs which newly specified during this invocation)
    my @pdbsToOpen = keys %$pdbsOpenInReqMode;
    push @pdbsToOpen, @newPDBs;

    if ($debugOn) {
      log_msg("force_pdb_modes: preparing to assign PDBs (".
              join(', ', @pdbsToOpen).") to instances (".
              join(', ', (keys %procsOnAvailInst)).")\n");
    }

    foreach my $pdbName (@pdbsToOpen) {
      if (!@instances || $#instances < 0) {
        # there were more PDBs than there were processes started for all 
        # instances, so we need to start another round of assignments of PDBs 
        # to instances
        %tempInstProcs = %procsOnAvailInst;
        @instances = keys %procsOnAvailInst;
        $instIdx = 0;

        if ($debugOn) {
          log_msg("force_pdb_modes: starting a new round of PDB to instance ".
                  "assignments\n");
        }
      } elsif ($instIdx > $#instances) {
        $instIdx = 0;

        if ($debugOn) {
          log_msg("force_pdb_modes: reached the end of instance list; ".
                  "starting over\n");
        }
      }      

      if ($debugOn) {
        log_msg("force_pdb_modes: instance-to-#-of-processes mapping:\n");
        foreach my $inst (@instances) {
          log_msg("\t$inst : ".$tempInstProcs{$inst}."\n");
        }
      }

      # current instance name
      my $inst = $instances[$instIdx];
 
      if ($debugOn) {
        log_msg("force_pdb_modes: preparing to assign PDB $pdbName to ".
                "instance $inst (instIdx = $instIdx)\n");
      }

      # at this point we expect that $tempInstProcs{$inst} is > 0 (i.e. the 
      # current instance has at least one remaining process allocated for it 
      # which will be available to process the current PDB)
      if (! exists $tempInstProcs{$inst}) {
        log_msg("force_pdb_modes: tempInstProcs contains no entry for ".
                "instance $inst\n");
        return (1, undef);
      } elsif ($tempInstProcs{$inst} <= 0) {
        log_msg("force_pdb_modes: unexpected number of processes (".
                $tempInstProcs{$inst}.") in tempInstProcs for ".
                "instance $inst\n");
        return (1, undef);
      }

      if (exists $instToPdbMap{$inst}) {
        my $pdbsRef = $instToPdbMap{$inst};
        if ($debugOn) {
          log_msg("force_pdb_modes: instToPdbMap contains an entry for ".
                  "inst=$inst (as expected)\n");
          log_msg("force_pdb_modes: before adding current PDB, pdbs in ".
                  "instToPdbMap entry for instance $inst: ".
                  join(', ', @{$pdbsRef})."\n");
        }

        # add the current PDB to the list of PDBs which will be open on the
        # current instance
        push(@$pdbsRef, $pdbName);

        if ($debugOn) {
          log_msg("force_pdb_modes: added $pdbName to instToPdbMap entry ".
                  "for instance $inst\n");
        }
      } else {
        # this is unexpected
        log_msg("force_pdb_modes: instToPdbMap did not contain an entry for ".
                "instance $inst\n");
        return (1, undef);
      }

      # decrement number of processes available for this instance; if there 
      # are no more processes left for this instance, remove its name from 
      # @instances
      if (!--$tempInstProcs{$inst}) {
        if ($debugOn) {
          log_msg("force_pdb_modes: no more processes left for instance ".
                  "$inst\n");
        }

        splice @instances, $instIdx, 1;

        # $instIdx should not be advanced since it is pointing to the 
        # instance name (if any) following the one to which we have just 
        # assigned the PDB 
      } else {
        $instIdx++;

        if ($debugOn && $instIdx <= $#instances) {
          log_msg("force_pdb_modes: advancing to instance ".
                  $instances[$instIdx]." (instIdx = $instIdx)\n");
        }
      }
    }
  }
  
  $resetPdbModes{$reqMode.' '.0} = \%instToPdbMap;

  if ($debugOn) {
    log_msg("\nforce_pdb_modes: calling reset_pdb_modes\n");
  }

  # finally reset PDB mode(s)

  # Bug 18011217: append _catcon_$logFileSuffix to 
  # $logFilePathBase to avoid conflicts with other catcon 
  # processes running on the same host
  $ret = 
    reset_pdb_modes($connString, \%resetPdbModes, $doneCmd, 
                    $doneFileNamePrefix, 
                    $oracleScript, $dbmsVersion, !$useMultipleInstances,
                    $InstConnStrMap_ref, $debugOn);

  # bug 23106360: reset_pdb_modes may return 
  # - 1 if an error was encountered, 
  # - 2 if it could not reopen PDB(s) in requested mode because it 
  #     would be incompatible with the mode in which CDB is open, and
  # - 0 if everything appears to have gone well
  #
  # to make it possible for DBUA to work with no changes against a 
  # standby CDB, we will not treat 2 as an error (but we will remember 
  # that we did not change PDB(s)' mode, and so we do not need to 
  # revert it before exiting catcon)
  if ($ret == 1) {
    log_msg("force_pdb_modes: unexpected error reported by reset_pdb_modes\n");
    return (1, undef);
  }

force_pdb_modes_done:

  if ($ret == 0) {
    if ($debugOn || $verbose) {
      log_msg("force_pdb_modes: reset_pdb_modes completed successfully\n");
    }

    # we need to update the hash of PDBs which we have successfully processed 
    # and the hash of PDBs which are known to be opened
    %$pdbsOpenInReqMode = (%$pdbsOpenInReqMode, map {($_, 1)} @newPDBs);
    %$isConOpen = (%$isConOpen, map {($_, "Y")} @newPDBs);
  } else {
    if ($debugOn || $verbose) {
      log_msg <<msg;
force_pdb_modes: reset_pdb_modes could not change PDB modes to $reqModeString 
    mode because it would conflict with the mode in which the CDB is open
msg
    }
  }

  if ($debugOn) {
    log_msg("force_pdb_modes: returning:\nret = $ret\n");
    inst_pdbs_hash_dump(\%instToPdbMap, "force_pdb_modes", "instToPdbMap");
  }

  return ($ret, \%instToPdbMap);
}

#
# exec_reset_pdb_mode_stmts - execute an array of statements to change mode of 
#                             one or more PDBs
#
# Parameters:
#   Stmts - reference to an array of statements to execute
#   DoneCmd - a command to create a file whose existence will indicate that the
#     last statement of the script has executed (needed by exec_DB_script())
#     (IN)
#   DoneFilePathBase - base for a name of a "done" file (see above) (IN)
#   debugOn - indicator controlling generation of diagnostic info
#
# Returns:
#   0 if statements appear to have executed successfully; 1 otherwise
#
sub exec_reset_pdb_mode_stmts($$$$) {
  my ($Stmts, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  # temporarily reset $SIG{INT} to avoid dying while PDB$SEED is in transition
  my $saveIntHandlerRef = $SIG{INT};
  
  $SIG{INT} = \&handle_sigint_for_pdb_mode;

  if ($debugOn) {
    log_msg("exec_reset_pdb_mode_stmts: temporarily reset SIGINT handler\n");
  }

  my ($ignoreOutputRef, $spoolRef) = 
    exec_DB_script(@$Stmts, undef, $DoneCmd, $DoneFilePathBase, $debugOn);

  if ($debugOn) {
    print_exec_DB_script_output("exec_reset_pdb_mode_stmts", $spoolRef, 0);
  }

  # 
  # restore INT signal handler 
  #
  $SIG{INT} = $saveIntHandlerRef;

  if ($debugOn) {
    log_msg("exec_reset_pdb_mode_stmts: restored SIGINT handler\n");
  }

  # check if any errors were reported when executing SQL statements 
  # assembled above
  if ($debugOn) {
    log_msg("exec_reset_pdb_mode_stmts: examine spool array returned by ".
            "exec_DB_script\n".
            "to determine if it contains any error messages\n");
  }

  for my $spoolLine (@$spoolRef) {
    # look for lines reporting an error message other than ORA-65020 which 
    # gets issued if a PDB we are trying to close is already closed, which 
    # we can ignore (because if running against a 12.1 CDB we cannot specify 
    # FORCE in ALTER PDB CLOSE)
    if ($spoolLine =~ "^ORA-[0-9]" && $spoolLine !~ "^ORA-65020") {
      log_msg("exec_reset_pdb_mode_stmts: error reported while attempting to ".
              "reset PDB modes:\n");
      log_msg("  output produced by ALTER PDB statements:\n");

      for my $errLine (@$spoolRef) {
        log_msg("\t$errLine\n");
      }

      return 1;
    }
  }

  return 0;
}

# 
# get_inst_conn_string - generate a connect string for connecting to a 
#                        specified instance
#
# parameters:
#   connStr - user-supplied connect string
#   InstConnStrMap_ref - a reference to a hash mapping instance names to 
#     connect strings
#   inst - name of the instance to which we need to connect
#   debugOn - indicator of whether to produce debugging info
#
sub get_inst_conn_string($$$$) {
  my ($connStr, $InstConnStrMap_ref, $inst, $debugOn) = @_;
  
  if ($connStr  eq '/ AS SYSDBA') {
    # if using OS authentication, just return the connect string since we 
    # cannot make use of instance-specific connect string
    return $connStr;
  }

  my $numInstances = scalar keys %$InstConnStrMap_ref;

  # Bug 25366291: in some cases %$InstConnStrMap_ref may be left undefined, 
  #   in which case all processing will occur on the default instance
  if ($numInstances <= 1) {
    # if a DB is running on one instance, just return the connect string since 
    # we need not use instance-specific connect string
    return $connStr;
  }

  my $asSysdba = 0;
          
  # if the caller-supplied connact string includes AS SYSDBA, we need 
  # to strip it off before adding an instance-specific connect string 
  # and then add it back on
  if ($connStr =~ 'AS SYSDBA') {
    $connStr =~ s/AS SYSDBA//g;
    $asSysdba = 1;
  }
  
  # append instance-specific connect string
  $connStr .= "@".$InstConnStrMap_ref->{$inst};
  
  # append AS SYSDBA if we stripped it above
  if ($asSysdba) {
    $connStr .= " AS SYSDBA"; 
  }
  
  return $connStr;
}

#
# reset_pdb_modes - close specified PDBs on all instances and reopen them in 
#                   specified modes on specified instances.
#
# parameters:
#   - reference to an array of internal connect strings - [0] used to connect 
#     to a DB, [1] can be used to produce diagnostic message (IN)
#   - reference to a (mode.' '. restricted) => ref {instance-name => 
#     ref @pdb-names} hash describing PDBs whose mode needs to be reset (IN)
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#     (IN)
#   - base for a name of a "done" file (see above) (IN)
#   - indicator of whether _ORACLE_SCRPT needs to be set before (and cleared 
#     after) resetting PDB mode(s) (IN) 
#   - RDBMS version; used to ensure that SQL statements that we generate are 
#     compatible with the version of the RDBMS to which we are connected (IN)
#   - indicator of whether PDBs should be opened using the default instance;
#     will cause this function to open PDBs without specifying the 
#     INSTANCES-clause; 
#     will be used if opening PDBs for running scripts if the user did not 
#     instruct us to use all available instances or if there is only one 
#     instance to be had (IN)
#   - a reference to a hash mapping instance names to instance connect strings 
#     (IN)
#   - indicator of whether to produce debugging info (IN)
#
# returns:
# - 0 if it appears that all went well
# - 1 if an error was encountered
# - 2 if PDB(s) could not be opened in the requested mode (which would have 
#     to be something other that READ ONLY) because the CDB is open READ ONLY 
#     (bug 23106360)
#
# Bug 14248297: close PDB$SEED on all RAC instances. If the PDB$SEED is to be
# opened on all instances, then the 'seedMode' argument must specify it.
#
sub reset_pdb_modes($$$$$$$$$) {
  my ($myConnect, $pdbModeHashRef, $DoneCmd, $DoneFilePathBase, $OracleScript, 
      $dbmsVersion, $openUsingDfltInst, $InstConnStrMap_ref, $debugOn) = @_;
  
  if ($debugOn) {
    pdb_mode_hash_dump($pdbModeHashRef, "reset_pdb_modes", "pdbModeHashRef")
  }

  # %$pdbModeHashRef is keyed on (mode.' '. restricted)
  my @modeInfoArr = keys %$pdbModeHashRef;

  # used to remmeber if we were called to close all PDBs
  my $closingAllPdbs = 1;

  # if asked to open some PDB(s) to a mode other than READ ONLY, make sure 
  # the CDB itself is not open READ ONLY
  foreach my $modeInfo (@modeInfoArr) {
    my @modeAndRestr = split(/  */, $modeInfo);
    my $pdbMode = $modeAndRestr[0];

    if ($closingAllPdbs && $pdbMode != CATCON_PDB_MODE_MOUNTED) {
      $closingAllPdbs = 0;
    }

    if ($debugOn) {
      if ($pdbMode == CATCON_PDB_MODE_MOUNTED) {
        log_msg("reset_pdb_modes: processing pdbModeHashRef entry for PDBs ".
                "which need to be closed\n");
      } else {
        my $modeString = PDB_MODE_CONST_TO_STRING->{$pdbMode};
        my $restrString = 
          ($modeAndRestr[1] == 0) ? "unrestricted" : "restricted";

        log_msg("reset_pdb_modes: processing pdbModeHashRef entry for PDBs ".
                "which need to be reopened $modeString $restrString\n");
      }
    }

    if ($pdbMode != CATCON_PDB_MODE_MOUNTED && 
        $pdbMode != CATCON_PDB_MODE_READ_ONLY) {
      if ($debugOn) {
        log_msg("reset_pdb_modes: call get_CDB_open_mode to determine if ".
                "the PDB(s) may be opened in desired mode\n");
      }

      my $cdbOpenMode = 
        get_CDB_open_mode(@$myConnect, $DoneCmd, $DoneFilePathBase, $debugOn);

      if (!(defined $cdbOpenMode)) {
        log_msg("reset_pdb_modes: unexpected error in get_CDB_open_mode\n");
        return 1;
      }

      if ($cdbOpenMode eq 'READ ONLY') {
        log_msg("reset_pdb_modes: WARNING: PDBs cannot be opened in ".
                PDB_MODE_CONST_TO_STRING->{$pdbMode}.
                " mode because the CDB is open READ ONLY");
        return 2;
      }

      # if we got here, the CDB must be open READ WRITE/UPGRADE, so we 
      # passed the check and there is no reason to process the rest of 
      # entries (if any) in @modeInfoArr
      if ($debugOn && $#modeInfoArr) {
        log_msg("reset_pdb_modes: CDB is open in $cdbOpenMode mode which ".
                "will satisfy all remaining pdbModeHashRef entries\n");
      }

      last;
    }
  }

  # Bug 25061922: remember if we are connected to a 12.1 CDB
  my $cdb_12_1 = ($dbmsVersion =~ "^12.1") ? 1 : 0;
  my $cdb_12_1_0_1 = $cdb_12_1 && ($dbmsVersion =~ "^12.1.0.1");

  # if a 12.1 CDB is running on just one instance, we do not need to use an 
  # instance-specific connect string to connect to the DB
  #
  # NOTE: Bug 25366291: in some cases %$InstConnStrMap_ref may be left 
  #   undefined, in which case we will force all processing to occur on the 
  #   default instance
  my $numInstConnStrings = scalar keys %$InstConnStrMap_ref;

  # if caller wants us to use "/ AS SYSDBA" to connect to a 12.1 CDB running 
  # on multiple RAC instances, we cannot use instance-specific connect 
  # strings and will resort to using INSTANCES=ALL in ALTER PDB CLOSE 
  # statements
  my $usingOsAuthentication = ($myConnect->[0] eq '/ AS SYSDBA');

  # if running against a post-12.1 CDB or a 12.1 CDB running on a single
  # instance or the connect string supplied by the caller relies on OS 
  # authentication (/ AS SYSDBA), start constructing a set of statements to be 
  # executed. 
  #
  # Otherwise, connect statement will need to use an instance-specific 
  # connect string for every instance and so will be constructed in 
  # the loop that iterates over instances
  my @ResetPdbModeStatements;
  
  if (!$cdb_12_1 || $numInstConnStrings <= 1 || $usingOsAuthentication) {
    if ($debugOn) {
      log_msg("reset_pdb_modes: setting first statements in ".
              "ResetPdbModeStatements for\n".
              "  PDBs open on all instances because\n");
      if (!$cdb_12_1) {
        log_msg("\twe are connected to a post-12.1 CDB\n");
      } elsif ($numInstConnStrings == 1) {
        log_msg("\tCDB is open on only one instance\n");
      } elsif ($numInstConnStrings == 0) {
        log_msg("\tinstance connect strings could not be fetched, so all\n".
                "\tprocessing will occur on the default instance\n");
      } else {
        log_msg("\tuser-supplied connect string relies on OS ".
                "authentication\n");
      }
    }

    @ResetPdbModeStatements = ("connect ".$myConnect->[0]."\n");
    if ($debugOn) {
      log_msg("reset_pdb_modes: added\n".
              "\tconnect ".$myConnect->[1]."\n");
    }

    # if changing PDB mode requires that _oracle_script be set, do it
    if ($OracleScript) {
      push @ResetPdbModeStatements, 
        qq#alter session set "_oracle_script"=TRUE\n/\n#;
      if ($debugOn) {
        log_msg("reset_pdb_modes: added\n".
                "\t".$ResetPdbModeStatements[1]."\n");
      }
    }
  }
  
  # Bugs 25132308, 25117295: if connected to a 12.1* CDB, we will be 
  #   generating a separate script for PDBs connected to every instance 
  #   (except when opening PDBs if we were instructed to open them all on the 
  #   default instance), and each of them will need to start with connect, 
  #   optionally followed by ALTER SESION SET "_ORACLE_SCRIPT"... which we 
  #   will store in this array
  my @firstStmts_12_1;

  if ($cdb_12_1) {
    # save statements with which a script closing or opening PDBs on a 
    # given instance needs to start, assuming we added some statements to 
    # @ResetPdbModeStatements; otherwise, we will be generating connect 
    # statement, and possibly ALTER SESSION SET "_oracle_script" for every 
    # instance on which a 12.1 CDB is running and to which we need to connect 
    if (@ResetPdbModeStatements) {
      @firstStmts_12_1  = @ResetPdbModeStatements;
      
      if ($debugOn) {
        log_msg("reset_pdb_modes: copied ResetPdbModeStatements to ".
                "firstStmts_12_1\n");
      }
    }

    if ($debugOn) {
      log_msg("reset_pdb_modes: processing PDBs in a ".
              ($cdb_12_1_0_1 ? "12.1.0.1" : "12.1.0.2")." CDB\n");
    }
  }

  # first we need to generate statements to close specified PDBs on all 
  # specified instances
  if ($debugOn) {
    log_msg("reset_pdb_modes: generate statements to close specified PDBs on ".
            "all specified instances:\n");
  }

  foreach my $modeInfo (@modeInfoArr) {
    # instances and PDBs on those instances whose mode needs to be reset
    my $instToPdbMapRef = $pdbModeHashRef->{$modeInfo};
    my @instances = keys %$instToPdbMapRef;

    foreach my $inst (@instances) {
      if ($#{$instToPdbMapRef->{$inst}} == -1) {
        if ($debugOn) {
          log_msg("  no PDBs to close on instance $inst - skip it\n");
        }
        next;
      }

      if ($cdb_12_1) {
        if ($debugOn) {
          log_msg("  for ".($cdb_12_1_0_1 ? "12.1.0.1" : "12.1.0.2").
                  " instance $inst:\n");
        }

        if (!@firstStmts_12_1) {
          # need to generate a connect statement using an instance-specific 
          # connect string and possibly set _oracle_script
          
          my $instConnStr = 
            get_inst_conn_string($myConnect->[0], $InstConnStrMap_ref, $inst, 
                                 $debugOn);
          if ($debugOn) {
            my $instConnStrDbg = 
              get_inst_conn_string($myConnect->[1], $InstConnStrMap_ref, $inst,
                                   $debugOn);
            
            log_msg("    will connect as $instConnStrDbg\n");
          }

          @ResetPdbModeStatements = ("connect ".$instConnStr."\n",);

          # if changing PDB mode requires that _oracle_script be set, do it
          if ($OracleScript) {
            push @ResetPdbModeStatements, 
              qq#alter session set "_oracle_script"=TRUE\n/\n#;

            if ($debugOn) {
              log_msg("    and set _oracle_script\n");
            }
          }
        }
      }

      # Bugs 25132308, 25117295: in 12.1.0.1, FORCE cannot be specified with 
      #   ALTER PDB CLOSE and in 12.1.0.2 it cannot be specified with 
      #   multiple instances.  To address these issues (and to simplify code),
      #   I will modify code dealing with 12.1* CDB to
      #     - not use the INSTANCES clause at all and 
      #     - if connected to a 12.1.0.1 CDB, to not specify FORCE
      #   Instead, if connected to a 12.1* CDB, I collect statements 
      #   pertaining to PDBs that need to be closed on a given instance and 
      #   execute them against that instance (by connecting to the DB using a 
      #   connect string corresponding to $inst - fix for bug 25315864) 
      #   rather than assembling statements closing PDBs across all instances.
      #
      #   In addition, rather than issuing a single ALTER PDB statement for 
      #   all PDBs on a given instance, we will issue a separate ALTER PDB 
      #   statement for every PDB because this will 
      #     - ensure that if one of 12.1.0.1 PDBs is already closed, the rest 
      #       of PDBs will not be affected by the error and
      #     - prevent us from generating an overly long statement if 
      #       there are lots of PDBs that need to be closed
      #
      # NOTE: fix for bugs 25132308 and 25117295 pretty much replaces changes 
      #       made to address bug 25061922

      # Bug 25315864: if the caller supplied "/ AS SYSDBA" as the connect 
      #               string, we cannot use instance-specific connect strings 
      #               to connect to individual instances. In such cases, we 
      #               will connect to the default instance and add 
      #               INSTANCES=ALL to ALTER PDB CLOSE statements (which is 
      #               what we used to do in reset_seed_pdb_mode)
      foreach my $pdb (@{$instToPdbMapRef->{$inst}}) {
        # need to issue ALTER PDB CLOSE IMMEDIATE regardless of whether we 
        # are connected to a 12.1* CDB
        my $stmt = "alter pluggable database $pdb close immediate";
        
        if ($cdb_12_1) {
          # if connected to a 12.1 RAC CDB running on multiple instances and 
          # the connect string specified by the caller relies on OS 
          # authentication, add INSTANCES=ALL
          #
          # Bug 25366291: ditto if we failed to obtain instance connect strings
          if (($numInstConnStrings > 1 && $usingOsAuthentication) ||
              $numInstConnStrings == 0) {
            $stmt .= " instances = all";
          } elsif (!$cdb_12_1_0_1) {
            # unless we are connected to a 12.1.0.1 CDB (where 
            # ALTER PDB CLOSE FORCE is not supported) OR we had to add 
            # INSTANCES=ALL above (in 12.1.0.2 specifying FORCE when
            # closing a PDB on multiple instances results in a confusing 
            # ORA-65145), add FORCE
            $stmt .= " force";
          }

          $stmt .= "\n/\n";
        } else {
          # for post 12.1 CDBs, we will specify both FORCE and 
          # INSTANCES-clause and execute ALTER PDB statements for all 
          # instances as one script
          $stmt .= " force instances=('$inst')\n/\n";
        }
        
        push @ResetPdbModeStatements, $stmt;

        if ($debugOn) {
          log_msg("\t$stmt\n");
        }
      }

      # if operating against a 12.1.* CDB, statements in 
      # @ResetPdbModeStatements need to be run against instance $inst
      if ($cdb_12_1) {
        if ($debugOn) {
          log_msg("reset_pdb_modes: execute statements to close PDBs on ".
                  "instance $inst of a 12.1 CDB\n");
        }

        if (exec_reset_pdb_mode_stmts(\@ResetPdbModeStatements, $DoneCmd, 
              $DoneFilePathBase."_close_".$inst."_pdbs", $debugOn)) {
          log_msg("reset_pdb_modes: unexpected error in ".
                  "exec_reset_pdb_mode_stmts\n");
          return 1;
        }
        
        # having executed statements to close PDBs on this instance, set 
        # @ResetPdbModeStatements to @firstStmts_12_1 for the next series of 
        # statements to close or open PDBs (and if statements to connect to 
        # the CDB and possibly set _oracle_script have to be generated anew 
        # for every instance, this simply undefines @ResetPdbModeStatements, 
        # preparing it for the next series)
        @ResetPdbModeStatements = @firstStmts_12_1;
      }
    }
  }

  if ($closingAllPdbs) {
    if ($debugOn) {
      log_msg("reset_pdb_modes: called to close all specified PDBs\n".
              "\tskip code trying to generate statements to reopen PDBs\n");
    }

    goto issueResetPdbModesStmts;
  }

  # Bug 25366291: if instance connect strings could not be obtained, all PDBs 
  #               will be open using the default instance
  if ($numInstConnStrings == 0 && !$openUsingDfltInst) {
    $openUsingDfltInst = 1;

    if ($debugOn) {
      log_msg("reset_pdb_modes: instance connect strings are not available, ".
              "so all PDBs\n".
              "\twill be opened on the default instance, even though the\n".
              "\tcaller has not requested it\n");
    }
  }

  # this hash will be used to keep track of 
  # - PDBs for which we have already generated statements to open them in 
  #   UPGRADE mode so as to avoid trying to open them in that mode on any 
  #   other instance (because a PDB can be open in UPGRADE mode on only 1 
  #   instance)
  # - PDBs for which we have already generated statements to open them in the 
  #   default instance if the caller has instructed us to do so, so as to 
  #   avoid issuing multiple ALTER PDB OPEN statements for the same PDB on 
  #   the default instance
  my %pdbsAlreadyOpen;

  # next, generate statements to open all specified PDBs in desired mode on 
  # specified instances
  if ($debugOn) {
    log_msg("reset_pdb_modes: generate statements to open specified PDBs in ".
            "desired modes on specified instances:\n");
  }

  # if connected to a 12.1 CDB and 
  # - the user-supplied connect string relies on OS authentication or 
  # - we were told to open PDBs using the default instance
  # all ALTER PDB OPEN statements will be executed as a part of one script 
  # which means that the initial statements (CONNECT and possibly 
  # ALTER SESSION SET _oracle_script) need to be generated only once, before 
  # we start generating ALTER PDB OPEN statements. 
  #
  # If @firstStmts_12_1 is defined (as would be the case if the user-supplied 
  # connect string relies on OS authentication), these initial statements have 
  # already been added to @ResetPdbModeStatements above; otherwise, we need 
  # to do it now
  if ($cdb_12_1 && !@firstStmts_12_1 && $openUsingDfltInst) {
    if ($debugOn) {
      log_msg("reset_pdb_modes: setting first statements in ".
              "ResetPdbModeStatements for\n".
              "                 opening PDBs using the default instance of a ".
              ($cdb_12_1_0_1 ? "12.1.0.1" : "12.1.0.2")." CDB:\n");
    }

    @ResetPdbModeStatements = ("connect ".$myConnect->[0]."\n",);

    if ($debugOn) {
      log_msg("  will connect as ".$myConnect->[1]."\n");
    }

    # if changing PDB mode requires that _oracle_script be set, do it
    if ($OracleScript) {
      push @ResetPdbModeStatements, 
        qq#alter session set "_oracle_script"=TRUE\n/\n#;

      if ($debugOn) {
        log_msg("  and set _oracle_script\n");
      }
    }
  }

  foreach my $modeInfo (@modeInfoArr) {
    my @modeAndRestr = split(/  */, $modeInfo);
    my $pdbMode  = $modeAndRestr[0];
    my $restr = $modeAndRestr[1];
    my $modeString = PDB_MODE_CONST_TO_STRING->{$pdbMode};
    my $restrString = ($restr) ? " restricted" : "";

    if ($debugOn) {
      log_msg("\tPDBs which need to be opened $modeString $restrString - \n");
    }

    # if this entry indicates that PDBs are to be closed, they have already 
    # been closed, so we can skip this entry
    if ($pdbMode == CATCON_PDB_MODE_MOUNTED) {
      next;
    }

    # instances and PDBs on those instances whose mode needs to be reset
    my $instToPdbMapRef = $pdbModeHashRef->{$modeInfo};
    my @instances = keys %$instToPdbMapRef;
    foreach my $inst (@instances) {
      if ($#{$instToPdbMapRef->{$inst}} == -1) {
        if ($debugOn) {
          log_msg("  no PDBs to reopen on instance $inst - skip it\n");
        }
      
        next;
      }

      if ($cdb_12_1) {
        if ($debugOn) {
          log_msg("  for ".($cdb_12_1_0_1 ? "12.1.0.1" : "12.1.0.2").
                  " instance $inst:\n");
        }
        
        # if a 12.1 CDB is open on multiple instances and the connect string 
        # supplied by the caller does not rely on OS authentication, 
        # @firstStmts_12_1 will not be defined since while we were closing 
        # PDBs, we needed to generate CONNECT statements using 
        # instance-specific connect strings.  During the opening stage, the 
        # same reasoning applies, unless we were told to open PDBs using the 
        # default instance
        if (!@firstStmts_12_1 && !$openUsingDfltInst) {
          my @instConnStr;

          # need to generate a connect statement using an instance-specific 
          # connect string
          $instConnStr[0] = 
            get_inst_conn_string($myConnect->[0], $InstConnStrMap_ref, $inst,
                                 $debugOn);
          if ($debugOn) {
            $instConnStr[1] = 
              get_inst_conn_string($myConnect->[1], $InstConnStrMap_ref, 
                                   $inst, $debugOn);
          }

          @ResetPdbModeStatements = ("connect ".$instConnStr[0]."\n",);

          if ($debugOn) {
            log_msg("    will connect as ".$instConnStr[1]."\n");
          }

          # if changing PDB mode requires that _oracle_script be set, do it
          if ($OracleScript) {
            push @ResetPdbModeStatements, 
              qq#alter session set "_oracle_script"=TRUE\n/\n#;

            if ($debugOn) {
              log_msg("    and set _oracle_script\n");
            }
          }
        }
      }

      my @pdbsToOpen;

      if ($openUsingDfltInst || $pdbMode == CATCON_PDB_MODE_UPGRADE ||
         ($cdb_12_1 && $usingOsAuthentication)) {
        # determine if any of the PDBs that should be open only once (because 
        # - we were instructed to open all PDBs on the default instance or 
        # - we are opening them for UPGRADE or 
        # - we are connected to a 12.1 CDB and the user-supplied connect 
        #   string relies on OS authentication, so we cannot connect to a 
        #   specific RAC instance) 
        # have already been opened
        my @newPDBs = 
          grep {! exists $pdbsAlreadyOpen{$_}} @{$instToPdbMapRef->{$inst}};

        if ($#newPDBs < 0) {
          # no new PDBs to open
          if ($debugOn) {
            log_msg("reset_pdb_modes: all PDBs (".
                    join(',', @{$instToPdbMapRef->{$inst}}).")\n".
                    "\tcorresponding to instance $inst have ".
                    "already been opened ");

            if ($pdbMode == CATCON_PDB_MODE_UPGRADE) {
              log_msg("in UPGRADE mode\n");
            } elsif ($openUsingDfltInst) {
              log_msg("on the default instance\n");
            } else {
              log_msg("in a 12.1 CDB using OS authentication\n");
            }
          }

          next;
        } elsif ($debugOn) {
          if ($#newPDBs < $#{$instToPdbMapRef->{$inst}}) {
            my @alreadyOpened = grep {! exists $pdbsAlreadyOpen{$_}} 
              @{$instToPdbMapRef->{$inst}};

            log_msg("reset_pdb_modes: of PDBs (".
                    join(',', @{$instToPdbMapRef->{$inst}}).")\n".
                    "\tcorresponding to instance $inst, some ( ".
                    join(',', @alreadyOpened).")\n".
                    "have already been opened ");

            if ($pdbMode == CATCON_PDB_MODE_UPGRADE) {
              log_msg("in UPGRADE mode\n");
            } elsif ($openUsingDfltInst) {
              log_msg("on the default instance\n");
            } else {
              log_msg("in a 12.1 CDB using OS authentication\n");
            }

            log_msg("\tso only (". join(',', @newPDBs).") will be opened\n");
          } else {
            log_msg("reset_pdb_modes: of PDBs (".
                    join(',', @{$instToPdbMapRef->{$inst}}).")\n".
                    "\tcorresponding to instance $inst, none have already ".
                    "been opened ");

            if ($pdbMode == CATCON_PDB_MODE_UPGRADE) {
              log_msg("in UPGRADE mode ");
            } elsif ($openUsingDfltInst) {
              log_msg("on the default instance ");
            } else {
              log_msg("in a 12.1 CDB using OS authentication ");
            }

            log_msg("so all will need to be opened\n");
          }
        }

        @pdbsToOpen = @newPDBs;

        # remember that PDBs listed in @newPDBs should not be opened again
        %pdbsAlreadyOpen = (%pdbsAlreadyOpen, map {($_, 1)} @newPDBs);
      } else {
        @pdbsToOpen = @{$instToPdbMapRef->{$inst}};
      }

      # Rather than issuing a single ALTER PDB statement for 
      # all PDBs on a given instance, we will issue a separate ALTER PDB 
      # statement for every PDB because this will prevent us from generating 
      # an overly long statement if there are lots of PDBs that need to be 
      # opened

      foreach my $pdb (@{$instToPdbMapRef->{$inst}}) {
        # need to issue ALTER PDB OPEN <mode> [restricted] regardless of 
        # whether we are operating against a 12.1* CDB
        my $stmt = "alter pluggable database $pdb open ".
          PDB_MODE_CONST_TO_STRING->{$pdbMode}.$restrString;
        
        if ($openUsingDfltInst || $cdb_12_1) {
          # if we were instructed to open PDBs on the default instance, do not 
          # specify INSTANCES-clause
          #
          # Bug 25061922: ditto if connected to a 12.1 CDB, where a bug 
          #   precludes us from specifying instance names
          $stmt .= "\n/\n";
        } else {
          $stmt .= " instances=('$inst')\n/\n";
        }

        push @ResetPdbModeStatements, $stmt;

        if ($debugOn) {
          log_msg("\t\t$stmt\n");
        }
      }

      # Bug 25061922, 25315864: if 
      #   - connected to a 12.1 CDB and 
      #   - we were not instructed to open all PDBs on the default instance, 
      #     and
      #   - the user-supplied connect string does not rely on OS authentication
      #     (so we can connect to a specific RAC instance)
      #   we will execute statements pertaining to PDBs that need to be 
      #   opened on a given instance and execute them against that instance 
      #   (which was assured when we constructed a CONNET statement using an 
      #   instance-specific connect string) rather than assembling statements 
      #   opening PDBs across all instances.  
      if ($cdb_12_1 && !$openUsingDfltInst && !$usingOsAuthentication) {
        if ($debugOn) {
          log_msg("reset_pdb_modes: execute statements to open PDBs on ".
                  "instance $inst of a 12.1 CDB\n");
        }

        if (exec_reset_pdb_mode_stmts(\@ResetPdbModeStatements, $DoneCmd, 
              $DoneFilePathBase."_open_".$inst."_pdbs", $debugOn)) {
          log_msg("reset_pdb_modes: unexpected error in ".
                  "exec_reset_pdb_mode_stmts\n");
          return 1;
        }
        
        # having executed statements to open PDBs on this instance, set 
        # to @firstStmts_12_1 for the next series of statements to open PDBs
        # (and if statements to connect to the CDB and possibly set 
        # _oracle_script have to be generated anew for every instance, this 
        # simply undefines @ResetPdbModeStatements, preparing it for the next 
        # series)
        @ResetPdbModeStatements = @firstStmts_12_1;
      }
    }
  }

issueResetPdbModesStmts:

  # NOTE: if 
  #       - operating against a 12.1 CDB and 
  #       - we were not instructed to open PDBs on the default instance and 
  #       - the user-supplied connect string does  not rely on OS 
  #         authentication (so we can connect to a specific RAC instance), 
  #       statements to open PDBs have already  been executed; otherwise, we 
  #       execute them now
  if ((!$cdb_12_1 || $openUsingDfltInst || $usingOsAuthentication) && 
      exec_reset_pdb_mode_stmts(\@ResetPdbModeStatements, $DoneCmd, 
                                $DoneFilePathBase, $debugOn)) {
    log_msg("reset_pdb_modes: unexpected error in ".
            "exec_reset_pdb_mode_stmts\n");
    return 1;
  }

  return 0;
}

#
# shutdown_db - shutdown the database in specified mode
#
# parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - shutdown mode
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
sub shutdown_db (\@$$$$) {
  my ($myConnect, $shutdownMode, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @ShutdownStatements = (
    "connect ".$myConnect->[0]."\n",
    "SHUTDOWN $shutdownMode\n",
  );

  exec_DB_script(@ShutdownStatements, undef, $DoneCmd, 
                 $DoneFilePathBase, $debugOn);
}

#
# kill_sqlplus_sessions - kill all SQL*Plus sessions started by us
#
# parameters:
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - name of the "kill all sessions script"
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
sub kill_sqlplus_sessions (\@$$$$) {
  my ($myConnect, $killSessScript, $DoneCmd, $DoneFilePathBase, 
      $debugOn) = @_;

  my @KillSessionsStatements = (
    "connect ".$myConnect->[0]."\n",
    "set echo on\n",
    "@@".$killSessScript."\n",
  );

  my ($ignoreOutputRef, $spoolRef) = 
    exec_DB_script(@KillSessionsStatements, undef, $DoneCmd, 
                   $DoneFilePathBase, $debugOn);

  if ($debugOn) {
    print_exec_DB_script_output("kill_sqlplus_sessions", $spoolRef, 0);
  }
}

# validate_script_path
#
# Parameters:
#   $FileName - name of file to validate
#   $Dir      - directory, if any, in which the file is expected to be found
#   $Windows  - an indicator of whether we are running under Windows
#   $IgnoreValidErr - if TRUE, validation errors will be reported but ignored
#   $DebugOn  - an indicator of whether to produce diagnostic messages
#
# Description:
#   construct file's path using its name and optional directory and determine 
#   if it exists and is readable
#
# Returns:
#   file's path
sub validate_script_path ($$$$$) {
  my ($FileName, $Dir, $Windows, $IgnoreValidErr, $DebugOn) = @_;

  my $Path;                                    # file path
  my $Slash = $Windows ? "\\" : "/";           # set OS file separator
  my $IgnoredErrors = 0;                       # were any errors ignored
    
  if ($DebugOn) {
    log_msg("validate_script_path: getting ready to construct path for script $FileName\n");
  }

# prepend directory if not already full path
  if ($Dir && (index($FileName,$Slash) <0)) {  
    $Path = $Dir.$Slash.$FileName;
  } else {
    $Path = $FileName;
  }

  if ($DebugOn) {
    log_msg("validate_script_path: getting ready to validate script $Path\n");
  }

  stat($Path);

  if (! -e _ || ! -r _) {
    log_msg("validate_script_path: sqlplus script $Path does not exist or is unreadable\n");
    if (!$IgnoreValidErr) {
      return undef;
    } else {
      $IgnoredErrors = 1;
    }
  }

  if (!($IgnoredErrors || -f $Path)) {
    log_msg("validate_script_path: supposed sqlplus script $Path is not a regular file\n");
    if (!$IgnoreValidErr) {
      return undef;
    } else {
      $IgnoredErrors = 1;
    }
  }

  if ($DebugOn) {
    if (!$IgnoredErrors) {
      log_msg("validate_script_path: successfully validated script $Path\n");
    } else {
      log_msg("validate_script_path: errors encountered and ignored while validating script $Path\n");
    }
  }

  return $Path;
}

#
# err_logging_tbl_stmt - construct and issue SET ERRORLOGGING ON [TABLE ...]
#                        statement, if any, that needs to be issued to create 
#                        an Error Logging table
#
# parameters:
#   - an indicator of whether to save error logging information, which may be 
#     set to ON to create a default error logging table or to the name of an 
#     existing error logging table (IN)
#   - reference to an array of file handles containing a handle to which to 
#     send SET ERRORLOGGING statement (IN)
#   - process number (IN)
#   - an indicator of whether we are running a user script (meaning that 
#     _ORACLE_SCRIPT was not set and does not need to temporarily reset in 
#     this subroutine) (IN)
#   - name of a Federation Root if it was supplied by the caller of catcon to 
#     indicate that scripts are to be run against Containers comprising a 
#     Federation rooted in that Container; if set, _federation_script may be 
#     set and so needs to be temporarily reset in this subroutine (IN)
#   - an indicator of whether to produce debugging info (IN)
#
sub err_logging_tbl_stmt ($$$$$$) {
  my ($ErrLogging, $FileHandles_REF, $CurProc, $UserScript, $FedRoot,
      $DebugOn) =  @_;

  # NOTE:
  # I have observed that if you issue 
  #   SET ERRORLOGGING ON IDENTIFIER ...
  # in the current Container after issuing
  #   SET ERRORLOGGING ON [TABLE ...] in a different Container, 
  # the error logging table will not be created in the current 
  # Container.  
  # 
  # To address this issue, I will first issue 
  #   SET ERRORLOGING ON [TABLE <table-name>]
  # and only then issue SET ERRORLOGGING ON [IDENTIFIER ...]
  #
  # If running an Oracle-supplied script, in order to make sure that the 
  # error logging table does not get created as a Metadata Link, I will 
  # temporarily reset 
  # _ORACLE_SCRIPT parameter before issuing 
  #   SET ERRORLOGING ON [TABLE <table-name>]
  #
  # Similarly, in order to make sure that the error logging table does not 
  # get created as a Federation Metadata Link if running a script against 
  # Containers comprising a Federation rooted in a specified Container, I 
  # will temporarily reset _FEDERATION_SCRIPT before issuing 
  #   SET ERRORLOGING ON [TABLE <table-name>]

  my $ErrLoggingStmt;

  if ($ErrLogging) {

    if ($DebugOn) {
      log_msg("err_logging_tbl_stmt: ErrLogging = $ErrLogging\n");
    }

    # if ERRORLOGGING is to be enabled, construct the SET ERRORLOGGING 
    # statement which will be sent to every process
    $ErrLoggingStmt = "SET ERRORLOGGING ON ";

    # Bug 25259127: $ErrLogging should not be quoted because it may specify 
    # owner.table. If a user is concerned about case folding, he needs to 
    # appropriately quote value of the parameter supplied to catcon
    if ((lc $ErrLogging) ne "on") {
      $ErrLoggingStmt .= qq#TABLE $ErrLogging#;
    }

    if ($DebugOn) {
      my $msg = <<msg;
err_logging_tbl_stmt: ErrLoggingStmt = $ErrLoggingStmt
    temporarily resetting _parameter before issuing 
    SET ERRORLOGGING ON [TABLE ...] in process $CurProc
msg
      log_msg($msg);
    }

    # if not running a user script, 
    #   _ORACLE_SCRIPT will be set, so reset it
    # else if Federation Root name was supplied
    #   _FEDERATION_SCRIPT will be set, so reset it
    if (!$UserScript) {
      printToSqlplus("err_logging_tbl_stmt", $FileHandles_REF->[$CurProc],
                     qq#ALTER SESSION SET "_ORACLE_SCRIPT"=FALSE#,
                     "\n/\n", $DebugOn);
    } elsif ($FedRoot){
      printToSqlplus("err_logging_tbl_stmt", $FileHandles_REF->[$CurProc],
                     qq#ALTER SESSION SET "_FEDERATION_SCRIPT"=FALSE#,
                     "\n/\n", $DebugOn);
    }

    # send SET ERRORLOGGING ON [TABLE ...] statement
    if ($DebugOn) {
      log_msg("err_logging_tbl_stmt: sending $ErrLoggingStmt to process $CurProc\n");
    }

    printToSqlplus("err_logging_tbl_stmt", $FileHandles_REF->[$CurProc],
                   $ErrLoggingStmt, "\n", $DebugOn);

    if ($DebugOn) {
      log_msg("err_logging_tbl_stmt: setting _parameter after issuing SET ERRORLOGGING ON [TABLE ...] in process $CurProc\n");
    }

    # if not running a user script, 
    #   _ORACLE_SCRIPT was set, so restore it
    # else if Federation Root name was supplied
    #   _FEDERATION_SCRIPT was set, so restore it
    if (!$UserScript) {
      printToSqlplus("err_logging_tbl_stmt", $FileHandles_REF->[$CurProc],
                     qq#ALTER SESSION SET "_ORACLE_SCRIPT"=TRUE#,
                     "\n/\n", $DebugOn);
    } elsif ($FedRoot){
      printToSqlplus("err_logging_tbl_stmt", $FileHandles_REF->[$CurProc],
                     qq#ALTER SESSION SET "_FEDERATION_SCRIPT"=TRUE#,
                     "\n/\n", $DebugOn);
    }
  }

  return $ErrLoggingStmt;
}

#
# start_processes - start processes
#
# parameters:
#   - number of processes to start (IN)
#   - base for constructing log file names (IN)
#   - reference to an array of file handles; will be obtained as a 
#     side-effect of calling open() (OUT)
#   - reference to an array of process ids; will be obtained by calls 
#     to open (OUT)
#   - reference to an array of Container names; array will be empty if 
#     we are operating against a non-Consolidated DB (IN)
#   - name of the Root Container; undefined if operating against a non-CDB (IN)
#   - connect string used to connect to a DB (IN)
#   - an indicator of whether SET ECHO ON should be sent to SQL*Plus 
#     process (IN)
#   - an indicator of whether to save ERRORLOGGING information (IN)
#   - an indicator of whether to produce debugging info (IN)
#   - an indicator of whether processes are being started for the first 
#     time (IN)
#   - an indicator of whether called by cdb_sqlexec.pl
#   - name of a Federation Root if called to run scripts against all 
#     Containers comprising a Federation rooted in the specified Container
#   - command used to generate a "done" file
#   - "disable lockdown profile" indicator
#   - index into @$ProcIds of a dead process which is being replaced
#     - -1 if not dealing with a dead process, so $NumProcs processes need to 
#       be started and their ids need to be added to @$ProcIds
#     - otherwise, only one process will be started and its id will be stored 
#       in $ProcIds->[$DeadProc] 
#   - if $DeadProc != -1, id of a process that died unexpectedly and will be 
#     replaced by the process started here; undef (and should not be 
#     referenced) otherwise
#   - name of a script that needs to be invoked to add 
#       ALTER SYSTEM KILL SESSION 
#     statement for every process that we start to the "kill session script" 
#   - "kill all sessions script" name
#   - reference to a hash mapping instance names to a hash representing useful
#     facts aboiut processes associated with it (namely, number of SQL*Plus 
#     processes that will be allocated for that instance) 
#     when processing PDBs using all available instances; 
#
#     this subroutine will add a hash representing an offset into @$ProcIds of 
#     the first sqlplus process created for a given instance (IN/OUT)
#   - reference to a hash mapping ids of sqlplus processes to names of 
#     instances for which they were allocated when processing PDBs 
#     using all available instances; existing entry in this hash will be 
#     used if replacing a dead process; new entries will be added as new 
#     processes are started (IN/OUT)
#   - a reference to a hash mapping instance names to instance connect strings 
#     (IN)
#
# Returns: 1 if an error is encountered
#
sub start_processes ($$\@\@\@$\@$$$$$$$$$$$$$$$) {
  my ($NumProcs, $LogFilePathBase, $FileHandles, $ProcIds, $Containers, $Root,
      $ConnectString, $EchoOn, $ErrLogging, $DebugOn, $FirstTime, 
      $UserScript, $FedRoot, $DoneCmd, $DisableLockdown,
      $DeadProc, $DeadProcId, $genKillSessScript, $killAllSessScript,
      $InstProcMap_ref, $ProcInstMap_ref, $InstConnStrMap_ref) =  @_;

  my $ps;

  my $CurrentContainerQuery = qq#select '==== Current Container = ' || SYS_CONTEXT('USERENV','CON_NAME') || ' Id = ' || SYS_CONTEXT('USERENV','CON_ID') || ' ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') || ' ====' AS now_connected_to from sys.dual;\n/\n#;

  # indexes of the first and last elements of @$ProcIds and @$FileHandles 
  # which will be populated with ids of newly started processes and their 
  # correspoinding file handles
  my $FirstProcIdx;
  my $LastProcIdx;

  # if processing PDBs using all available instances, we need to keep track 
  # of the instance for which every process is allocated as well as how many 
  # more processes will be allocated to the current instance
  my @instances;

  # index into @instances of the instance for which sqlplus processes are 
  # being created
  my $currInstIdx;

  # name of the current instance
  my $inst;

  # number of processes yet to be started for the current instance
  my $numProcsForCurrInst;

  # ids of processes allocated for the current instance
  my @procsForCurrInst;
  
  if ($DeadProc == -1) {
    # will start $NumProcs processes
    $FirstProcIdx = 0;
    $LastProcIdx = $NumProcs - 1;

    if ($DebugOn) {
      log_msg("start_processes: will start $NumProcs processes\n");
    }
    
    if (%$InstProcMap_ref) {
      if ($DebugOn) {
        log_msg("start_processes: will create processes for instances used ".
                "to process PDBs\n");
      }
      
      @instances = keys %$InstProcMap_ref;
      
      if ($#instances < 0) {        
        log_msg("start_processes: instance-to-process map contains no ".
                "entries\n");
        return 1;
      }

      if ($DebugOn) {
        if (%$InstProcMap_ref) {
          log_msg("start_processes: before any processes were started, ".
                  "contents of\n".
                  "InstProcMap_ref were:\n");

          foreach my $inst (sort keys %$InstProcMap_ref) {
            foreach my $k (keys %{$InstProcMap_ref->{$inst}}) {
              log_msg("\t$inst => ".
                      "($k => $InstProcMap_ref->{$inst}->{$k})\n");
            }
          }
        }
      }

      # start with the first instance; it's possible that no processes can be 
      # allocated for it - this will be dealt with before we start creating 
      # processes
      $currInstIdx = 0;
      $inst = $instances[$currInstIdx];
      
      # remember how many processes need to be created for this instance
      $numProcsForCurrInst = $InstProcMap_ref->{$inst}->{NUM_PROCS};
    
      # remember the offset into @$ProcIds where the id of first process 
      # (if any) for this instance will be stored
      if ($numProcsForCurrInst > 0) {
        $InstProcMap_ref->{$inst}->{FIRST_PROC} = $FirstProcIdx;

        if ($DebugOn) {
          log_msg("start_processes: will create $numProcsForCurrInst ".
                  "processes for instance $inst\n".
                  "with id of the first such process stored at offset ".
                  "$FirstProcIdx in ProcIds\n");

          log_msg("start_processes: contents of ".
                  "InstProcMap_ref->{$inst}:\n");

          foreach my $k (keys %{$InstProcMap_ref->{$inst}}) {
            log_msg("\t$inst => ".
                    "($k => $InstProcMap_ref->{$inst}->{$k})\n");
          }
        }
      } else {
        # if no processes will be created for this instance, there is no need 
        # to have an entry representing offset of the first process created 
        #for it
        delete $InstProcMap_ref->{$inst}->{FIRST_PROC}
          if (exists $InstProcMap_ref->{$inst}->{FIRST_PROC});

        if ($DebugOn) {
          log_msg("start_processes: will create 0 processes for ".
                  "instance $inst\n");
        }
      }
    }
  } else {
    # will start only 1 process and store its info in $DeadProc elements of 
    # @$ProcIds and @$FileHandles
    $FirstProcIdx = $LastProcIdx = $DeadProc;

    if ($DebugOn) {
      log_msg("start_processes: replacing dead process; will start one ".
              "process and store its id in ProcIds[$DeadProc]\n");
    }
    
    if (%$InstProcMap_ref) {
      # remember the instance for which the newly dead process was created 
      # (and for which the new process will be started)
      if (! exists $ProcInstMap_ref->{$DeadProcId}) {
        log_msg("start_processes: instance for which the dead process ".
                "$DeadProcId was allocated is not known\n");
        return 1;
      }

      $inst = $ProcInstMap_ref->{$DeadProcId};
      if ($DebugOn) {
        log_msg("start_processes: replacing process $DeadProcId which was ".
                "created for instance $inst\n");
      }
      
      # replacing 1 process; this will not affect the offset into @$ProcIds 
      # of the first process allocated for this instance
      $numProcsForCurrInst = 1;
    }
  }

  # 14787047: Save STDOUT so we can restore it after we start each process
  if (!open(SAVED_STDOUT, ">&STDOUT")) {
    log_msg("start_processes: failed to open ($!) SAVED_STDOUT\n");
    return 1;
  }

  # Keep Perl happy and avoid the warning "SAVED_STDOUT used only once"
  print SAVED_STDOUT "";

  # remember whether processes will be running against one or more Containers 
  # of a CDB, starting with the Root
  my $switchIntoRoot = (@$Containers && ($Containers->[0] eq $Root));

  for ($ps=$FirstProcIdx; $ps <= $LastProcIdx; $ps++) {
    # if creating processes for instances used to process PDBs, determine if 
    # there are more processes to be created for the current instance
    if (%$InstProcMap_ref) {
      if ($numProcsForCurrInst > 0) {
        if ($DebugOn) {
          log_msg("start_processes: $numProcsForCurrInst more processes ".
                  "to be started for instance $inst\n");
        }
      } else {
        if ($DebugOn) {
          log_msg("start_processes: no more processes to be started for ".
                  "instance $inst\n".
                 "will look for the next instance for which some at least ".
                  "one process can be created\n");
        }

        for ($currInstIdx++; $currInstIdx <= $#instances; $currInstIdx++) {
          $inst = $instances[$currInstIdx];
      
          # remember how many processes need to be created for this instance
          $numProcsForCurrInst = $InstProcMap_ref->{$inst}->{NUM_PROCS};
    
          # remember the offset into @$ProcIds where the id of first process 
          # (if any) for this instance will be stored
          if ($numProcsForCurrInst > 0) {
            $InstProcMap_ref->{$inst}->{FIRST_PROC} = $ps;

            if ($DebugOn) {
              log_msg("start_processes: will create $numProcsForCurrInst ".
                      "processes for instance $inst\n".
                      "with id of the first such process stored at offset ".
                      "$ps in ProcIds\n");

              log_msg("start_processes: contents of ".
                      "InstProcMap_ref->{$inst}:\n");

              foreach my $k (keys %{$InstProcMap_ref->{$inst}}) {
                log_msg("\t$inst => ".
                        "($k => $InstProcMap_ref->{$inst}->{$k})\n");
              }
            }

            last;
          } else {
            # if no processes will be created for this instance, there is no 
            # need to have an entry representing offset of the first process 
            # created for it
            delete $InstProcMap_ref->{$inst}->{FIRST_PROC}
              if (exists $InstProcMap_ref->{$inst}->{FIRST_PROC});
            
            if ($DebugOn) {
              log_msg("start_processes: no processes to be created for ".
                      "instance $inst\n");
            }
          }
        }

        if ($currInstIdx > $#instances) {
          # there appears to be a mismatch between the number of processes 
          # that the caller requested to start and the number of processes 
          # allocated to RAC instances
          log_msg("start_processes: number of processes (".
                  ($LastProcIdx - $FirstProcIdx + 1).") requested by the\n".
                  "caller exceeds the number of processes allocated to all ".
                  "available instances.\n");
          return 1;
        }
      }
    }

    my $LogFile = $LogFilePathBase.$ps.".log";

    # If starting for the first time, open for write; otherwise append
    if ($FirstTime) {
      if (!open (STDOUT,">", "$LogFile")) {
        log_msg("start_processes: failed to open ($!) STDOUT (1)\n");
        return 1;
      }
    } else {
      close (STDOUT);
      if (!open (STDOUT,"+>>", "$LogFile")) {
        log_msg("start_processes: failed to open ($!) STDOUT (2)\n");
        return 1;
      }
    }

    my $id = open ($FileHandles->[$ps], "|-", "sqlplus /nolog");

    if (!$id) {
      log_msg("start_processes: failed to open ($!) pipe to SQL*Plus\n");
      return 1;
    }

    # file handle for the current process
    my $fh = $FileHandles->[$ps];

    # if creating new processes, add new process id to @$ProcIds
    # otherwise, if creating a new process to replace a process that died 
    # unexpectedly, store the new process' id in the slot that used to be 
    # occupied by the dead process
    if ($DeadProc == -1) {
      push(@$ProcIds, $id);

      if ($DebugOn) {
        log_msg("start_processes: process $ps (id = $ProcIds->[$#$ProcIds]) ".
                "will use log file $LogFile\n");
      }
    } else {
      # before we replace dead process' id with the id of the newly started 
      # process, ensure that information about the process which died and is
      # being replaced gets recorded in the process' log file
      print $fh "prompt\n";
      print $fh "prompt WARNING: process $DeadProcId has died unexpectedly\n";
      print $fh "prompt          and was replaced with process $id\n";
      print $fh "prompt\n";
      $ProcIds->[$DeadProc] = $id;

      if ($DebugOn) {
        log_msg("start_processes: process $ps (id = $id) which will replace ".
                "newly dead process $DeadProcId will use log file $LogFile\n");
      }
    }
    
    # connect string passed to CONNECT statement
    my $connStr;

    # if processing PDBs using all available instances, decrement number 
    # of processes yet to be created for the current instance and update 
    # %$ProcInstMap_ref to indicate an instance for which this process 
    # was created
    if (%$InstProcMap_ref) {
      $numProcsForCurrInst--;

      # if replacing a dead process, remove mapping for its id from 
      # %$ProcInstMap_ref
      if ($DeadProc != -1) {
        delete $ProcInstMap_ref->{$DeadProcId};
      }

      $ProcInstMap_ref->{$id} = $inst;

      # apply an instance-specific connect string to the user-specified 
      # connect string
      $connStr = 
        get_inst_conn_string($ConnectString->[0], $InstConnStrMap_ref, $inst,
                             $DebugOn);    
    } else {
      # otherwise, use user-specified connect string
      $connStr = $ConnectString->[0];
    }

    # send initial commands to the sqlplus session
    print $fh "connect ".$connStr."\n";

    if ($DebugOn) {
      my $connStrDbg;

      if (%$InstProcMap_ref) {
        $connStrDbg = 
          get_inst_conn_string($ConnectString->[1], $InstConnStrMap_ref, $inst,
                             $DebugOn);
      } else {
        $connStrDbg = $ConnectString->[1];
      }

      log_msg(qq#start_processes: connected using $connStrDbg\n#);
    }

    # use ALTER SESSION SET APPLICATION MODULE/ACTION to identify this process
    printToSqlplus("start_processes", $fh,
                   qq#ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'#,
                   "\n/\n", $DebugOn);

    if ($DebugOn) {
      log_msg("start_processes: issued ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n");
    }

    printToSqlplus("start_processes", $fh,
                   qq#ALTER SESSION SET APPLICATION ACTION = 'started'#,
                   "\n/\n", $DebugOn);

    if ($DebugOn) {
      log_msg("start_processes: issued ALTER SESSION SET APPLICATION ACTION = 'started'\n");
    }

    # bug 21871308: create a "kill session script" for this process which will 
    # contain ALTER SYSTEM KILL SESSION statement to gracefully bring down 
    # this process
    my $killSessScript = kill_session_script_name($LogFilePathBase, $ps);

    printToSqlplus("start_processes", $fh, 
                   "@@".$genKillSessScript." ".$killSessScript." ".$id, 
                   "\n", $DebugOn);

    if ($DebugOn) {
      log_msg("start_processes: created a kill_session script (".
              $killSessScript.") for this ".
              "process\n");
    }

    if ($EchoOn) {
      printToSqlplus("start_processes", $fh, qq#SET ECHO ON#, "\n", $DebugOn);

      if ($DebugOn) {
        log_msg("start_processes: SET ECHO ON has been issued\n");
      }
    }
    
    # if running scripts against one or more Containers of a Consolidated 
    # Database, 
    # - if the Root is among Containers against which scripts will 
    #   be run (in which case it will be found in $Containers->[0]), switch 
    #   into it (because code in catconExec() expects that to be the case); 
    #   if scripts will not be run against the Root, there is no need to 
    #   switch into any specific Container - catconExec will handle that case 
    #   just fine all by itself
    # - turn on the "secret" parameter to ensure correct behaviour 
    #   of DDL statements; it is important that we do it after we issue 
    #   SET ERRORLOGGING ON to avoid creating error logging table as a 
    #   Common Table
    if (@$Containers) {
      if ($switchIntoRoot) {
        printToSqlplus("start_processes", $fh, 
                       qq#ALTER SESSION SET CONTAINER = $Root#, "\n/\n", 
                       $DebugOn);
        if ($DebugOn) {
          log_msg(qq#start_processes: switched into Container $Root\n#);
        }
      }
      
      # if not running a user script, 
      #   _ORACLE_SCRIPT needs to be set
      # else if Federation Root name was supplied
      #   _FEDERATION_SCRIPT needs to be set
      if (!$UserScript) {
        printToSqlplus("start_processes", $fh, 
                       qq#ALTER SESSION SET "_ORACLE_SCRIPT"=TRUE#, "\n/\n", 
                       $DebugOn);

        if ($DebugOn) {
          log_msg("start_processes: _oracle_script was set\n");
        }
      } elsif ($FedRoot) {
        printToSqlplus("start_processes", $fh, 
                       qq#ALTER SESSION SET "_FEDERATION_SCRIPT"=TRUE#, 
                       "\n/\n", $DebugOn);

        if ($DebugOn) {
          log_msg("start_processes: _federation_script was set\n");
        }
      }

      if ($DisableLockdown) {
        printToSqlplus("start_processes", $fh, 
                       qq#ALTER SESSION SET PDB_LOCKDOWN=''#, 
                       "\n/\n", $DebugOn);
        if ($DebugOn) {
          log_msg("start_processes: pdb_lockdown was disabled\n");
        }
      }
    }

    $fh->flush;
  }

  # we created a "kill session" script for every process we spawned to avoid 
  # possibiliy of contention between processes that were spawned.  Now that 
  # they have all been spawned, we are ready to coalesce them into a single 
  # "kill all sessions" script

  # open the "kill all sessions" script
  my $killAllSessFh;
  if (!open ($killAllSessFh,"+>>", "$killAllSessScript")) {
    log_msg("start_processes: failed to open ($!) kill_all_sessions script (".
            $killAllSessScript.")\n");
    return 1;
  }

  for ($ps=$FirstProcIdx; $ps <= $LastProcIdx; $ps++) {
    # open next "kill session" script
    my $killSessScript = kill_session_script_name($LogFilePathBase, $ps);
    my $killSessFh = openSpoolFile($killSessScript, $DebugOn);
    if (!$killSessFh) {
      log_msg("start_processes: failed to open ($!) kill_session script (".
              $killSessScript.")\n");
      # in some cases the spool file takes too long to materialize.  We cannot 
      # wait forever, and aborting processing seems too harsh, so we will 
      # settle for a warning message in the log file
      next;
    }
    
    print $killAllSessFh $_ while <$killSessFh>;

    close ($killSessFh);

    if ($DebugOn) {
      log_msg("start_processes: appended ".$killSessScript." to ".
              $killAllSessScript."\n");
    }

    # having concatenated this "kill session script", delete it
    sureunlink($killSessScript, $DebugOn);

    if ($DebugOn) {
      log_msg("start_processes: deleted ".$killSessScript."\n");
    }
  }

  close($killAllSessFh);
      
  if ($DebugOn) {
    if (!open ($killAllSessFh,"<", "$killAllSessScript")) {
      log_msg("start_processes: failed to open ($!) kill_all_sessions script ".
              "(".$killAllSessScript.")\n");
      return 1;
    }

    log_msg("start_processes: contents of kill_all_sessions script(".
            $killAllSessScript."):\n");

    log_msg("\t".$_) while <$killAllSessFh>;

    close($killAllSessFh);
  }

  # 14787047: Restore saved stdout
  close (STDOUT);
  open (STDOUT, ">&SAVED_STDOUT");

  return create_done_files($FirstProcIdx, $LastProcIdx, $LogFilePathBase,
                           $FileHandles, $ProcIds, 
                           $DebugOn, $DoneCmd);
}

#
# done_file_name_prefix - generate prefix of a name for a "done" file using 
#                         file name base and the id of a process to which 
#                         the "done" file belongs
#
sub done_file_name_prefix($$) {
    my ($FilePathBase, $ProcId) =  @_;

  return $FilePathBase."_catcon_".$ProcId;
}

#
# done_file_name - generate name for a "done" file using file name base 
#                  and the id of a process to which the "done" dile belongs
#
sub done_file_name($$) {
    my ($FilePathBase, $ProcId) =  @_;

  return done_file_name_prefix($FilePathBase, $ProcId).".done";
}

#
# child_log_file_name - generate name of a log file that will be used by a 
#                       child process forked in catconExec
#
sub child_log_file_name($$) {
    my ($FilePathBase, $ChildPid) =  @_;

  return $FilePathBase."_catcon_".$ChildPid.".lst";
}

#
# child_done_file_name - generate name of a file that will be used by a 
#                        child process forked in catconExec to communicate to 
#                        the parent that it is done
#
sub child_done_file_name($$) {
    my ($FilePathBase, $ChildPid) =  @_;

  return $FilePathBase."_catcon_".$ChildPid.".done";
}

#
# kill_session_script_name - generate name for a "kill session script" which 
#                            will be run if we need to gracefully terminate 
#                            SQL*Plus processes in the course of handling 
#                            SIGINT/SIGTERM/SIGQUIT (bug 21871308)
#
sub kill_session_script_name($$) {
  my ($FilePathBase, $ProcIdx) =  @_;

  return $FilePathBase."_catcon_kill_sess_".$$."_".$ProcIdx.".sql";
}

#
# create_done_files - Creates "done" files to be created for all processes.
#                     next_proc() will look for these files to determine
#                     whether a given process is available to take on the
#                     next script or SQL statement
#
# parameters:
#   - index of the first element of @$ProcIds_REF "done" files for which need 
#     to be created (IN)
#   - index of the last element of @$ProcIds_REF "done" files for which need 
#     to be created (IN)
#   - base for constructing log file names (IN)
#   - reference to an array of file handles (IN)
#   - reference to an array of process ids (IN)
#   - an indicator of whether to produce debugging info (IN)
#   - Done Command (IN)
#
sub create_done_files ($$$$$$$) {

    my ($FirstProcIdx, $LastProcIdx, $LogFilePathBase, $FileHandles_REF, 
        $ProcIds_REF, $DebugOn, $DoneCmd) =  @_;

    # Loop through processors
    for (my $CurProc = $FirstProcIdx; $CurProc <= $LastProcIdx; $CurProc++) {
      
      # file which will indicate that process $CurProc finished its work and 
      # is ready for more
      #
      # Bug 18011217: append _catcon_$ProcIds_REF->[$CurProc] to 
      # $LogFilePathBase to avoid conflicts with other catcon processes 
      # running on the same host
      my $DoneFile = 
        done_file_name($LogFilePathBase, $ProcIds_REF->[$CurProc]);

      # create a "done" file unless it already exists
      if (! -e $DoneFile) {
        # "done" file does not exist - cause it to be created
        printToSqlplus("create_done_files", $FileHandles_REF->[$CurProc],
                       qq/$DoneCmd $DoneFile/, "\n", $DebugOn);

        # flush the file so a subsequent test for file existence does 
        # not fail due to buffering
        $FileHandles_REF->[$CurProc]->flush;

        if ($DebugOn) {
          my $msg = <<msg;
create_done_files: sent "$DoneCmd $DoneFile"
    to process $CurProc (id = $ProcIds_REF->[$CurProc]) to indicate its 
    availability
msg
          log_msg($msg);
        }
      } elsif (! -f $DoneFile) {
        log_msg(qq#create_done_files: "done" file name collision: $DoneFile\n#);
        return 1;
      } else {
        if ($DebugOn) {
          log_msg(qq#create_done_files: "done" file $DoneFile already exists\n#);
        }
      }
    }
    return 0;
}

#
# end_processes - end process(es)
#
# parameters:
#   - index of the first process to end (IN)
#   - index of the last process to end (IN)
#   - reference to an array of file handles (IN)
#   - reference to an array of process ids; will be cleared of its 
#     elements (OUT)
#   - an indicator of whether to produce debugging info (IN)
#   - log file path base
#   - index into @$ProcIds of a dead process which is being replaced
#     - -1 if not dealing with a dead process, so all processes between 
#       $FirstProcIdx and $LastProcIdx will be ended and their entries in 
#       @$ProcIds will be deleted
#     - otherwise, only the process whose id is in $ProcIds->[$DeadProc] will 
#       be "ended" and its spot in @$ProcIds will be retained, albeit reset 
#       to -1
#   - reference to a hash mapping instance names to a hash representing useful
#     facts aboiut processes associated with it (namely, number of SQL*Plus 
#     processes that will be allocated for that instance and an offset into 
#     @$ProcIds of the first sqlplus process allocated to a given instance
#     when processing PDBs using all available instances; 
#
#     if all processes associated with a given instance are ended, the entry 
#     reprsenting ofset of the first process created or that instance will be 
#     deleted (IN/OUT)
#   - reference to a hash mapping ids of sqlplus processes to names of 
#     instances for which they were allocated;
#
#     entries mapping ids of processes that we "end" to instances for which 
#     they were created will be deleted (IN/OUT)
#
sub end_processes ($$\@\@$$$$$) {
  my ($FirstProcIdx, $LastProcIdx, $FileHandles, $ProcIds, $DebugOn,
      $LogFilePathBase, $DeadProc,
      $InstProcMap_ref, $ProcInstMap_ref) =  @_;

  my $ps;

  if ($DeadProc != -1) {
    # we were called to "end" a process which apepars to have died 
    # unexpectedly, so we only need to handle that process
    $FirstProcIdx = $LastProcIdx = $DeadProc;
  }

  if ($FirstProcIdx < 0) {
    log_msg("end_processes: FirstProcIdx ($FirstProcIdx) was less than 0\n");
    return 1;
  }

  if ($LastProcIdx < $FirstProcIdx) {
    log_msg("end_processes: LastProcIdx ($LastProcIdx) was less than ".
            "FirstProcIdx ($FirstProcIdx)\n");
    return 1;
  }

  if ($DebugOn) {
    if ($DeadProc == -1) {
      log_msg("end_processes: will end processes $FirstProcIdx to ".
              "$LastProcIdx\n");
    } else {
      log_msg("end_processes: will end dead process $DeadProc ".
              "(id=$ProcIds->[$DeadProc])\n");
    }
  }

  for ($ps = $FirstProcIdx; $ps <= $LastProcIdx; $ps++) {
    if ($FileHandles->[$ps]) {
      # LRG 19650060: if the caller didn't tell us that the process is dead,
      #               check if it is really alive since end_processes may be 
      #               called to end processes after some process was found to 
      #               be dead and the caller did not instruct us to recover 
      #               from SQL*Plus process failure
      my $procAlive;

      if ($DeadProc != -1) {
        $procAlive = 0;
      } else {
        if ($DebugOn) {
          log_msg("end_processes: check if process $ps (id=$ProcIds->[$ps]) ".
                  "is alive\n");
        }

        $procAlive = kill 0, $ProcIds->[$ps];

        if ($DebugOn) {
          log_msg("end_processes: process $ps (id=$ProcIds->[$ps]) is ".
                  ($procAlive ? "alive\n" : "dead\n"));
        }
      }

      # Bug 22887047: no point trying to write to a process that died (all 
      # you get out of it is a "Broken pipe" message and catcon dying)
      if ($procAlive) {
        # @@@@
        #print {$FileHandles->[$ps]} "PROMPT ========== PROCESS ENDED ==========\n";
        #
        #printToSqlplus("end_processes", $FileHandles->[$ps],
        #               "EXIT", "\n", $DebugOn);

        if ($DebugOn) {
          log_msg("end_processes: write EXIT to process $ps ".
                  "(id=$ProcIds->[$ps])\n");
        }

        print {$FileHandles->[$ps]} "EXIT\n";

        if ($DebugOn) {
          log_msg("end_processes: close file handle for process $ps ".
                  "(id=$ProcIds->[$ps])\n");
        }

        close ($FileHandles->[$ps]);
      }

      if ($DebugOn) {
        log_msg("end_processes: setting FileHandle[$ps] to undef\n");
      }

      $FileHandles->[$ps] = undef;
    } elsif ($DebugOn) {
      log_msg("end_processes: process $ps has already been stopped\n");
    }

    # if processing PDBs using all available instances, delete an entry 
    # mapping this process to the instance for which it was created from 
    # %$ProcInstMap_ref
    if ((%$ProcInstMap_ref) && 
        (exists $ProcInstMap_ref->{$ProcIds->[$ps]})) {
      delete $ProcInstMap_ref->{$ProcIds->[$ps]};

      if ($DebugOn) {
        log_msg("end_processes: deleted entry for process ".
                $ProcIds->[$ps]." from ProcInstMap_ref\n");
      }
    }
  }

  if ($DebugOn) {
    if ($DeadProc == -1) {
      log_msg("end_processes: done with processes $FirstProcIdx to ".
              "$LastProcIdx\n");
    } else {
      log_msg("end_processes: done with dead process $DeadProc ".
              "(id=$ProcIds->[$DeadProc])\n");
    }
  }

  # clean up completion files
  clean_up_compl_files($LogFilePathBase, $ProcIds, $FirstProcIdx, $LastProcIdx,
                       $DebugOn);
  
  # delete @$ProcIds entries corresponding to processes which were ended, 
  # unless we are handling a dead process which will be replaced with a new 
  # process, in which case we want to preserve its spot (since that's where 
  # we will place id of the process that will be started in its place)
  if ($DeadProc == -1) {
    splice @$ProcIds, $FirstProcIdx, $LastProcIdx - $FirstProcIdx + 1;

    # if processing PDBs using all available instances, update 
    # $InstProcMap_ref to accurately describe offset in @$ProcIds of the 
    # first process created for a given instance
    if (%$InstProcMap_ref) {
      my @instances = keys %$InstProcMap_ref;

      if ($#instances < 0) {        
        log_msg("end_processes: instance-to-process map contains no ".
                "entries\n");
        return 1;
      }

      # first, we delete FIRST_PROC entries for every instance
      foreach my $inst (@instances) {
        if (exists $InstProcMap_ref->{$inst}->{FIRST_PROC}) {
          delete $InstProcMap_ref->{$inst}->{FIRST_PROC};

          if ($DebugOn) {
            log_msg("end_processes: deleted FIRST_PROC entry for instance ".
                    $inst." from InstProcMap_ref\n");
          }
        }
      }

      # then we traverse @$ProcIds and determine offsets of the first process 
      # (if any) which was created for every instance
      my $currInst;

      for (my $pidIdx = 0; $pidIdx <= $#$ProcIds; $pidIdx++) {
        my $pid = $ProcIds->[$pidIdx];

        if (! exists $ProcInstMap_ref->{$pid}) {
          log_msg("end_processes: ProcInstMap_ref does not contain an ".
                  "entry for process $pid\n".
                  "which is represented by an element of ProcIds\n");
          return 1;
        }

        my $inst = $ProcInstMap_ref->{$pid};

        if ((! defined $currInst) || $inst ne $currInst) {
          # first process creted for instance $inst
          $InstProcMap_ref->{$inst}->{FIRST_PROC} = $pidIdx;
          $currInst = $inst;

          if ($DebugOn) {
            log_msg("end_proceses: id ($pid) of the first process started ".
                    "for instance $inst\n".
                    "was found in ProcIds at offset $pidIdx\n");
          }
        }
      }
    }
  } else {
    # we did not delete the @$ProcIds entry corresponding to the dead process, 
    # so reset it to -1 to make sure we do not assume that it represents a 
    # valid process id
    $ProcIds->[$DeadProc] = -1;
  }


  if ($DebugOn) {
    log_msg("end_processes: ended processes $FirstProcIdx to $LastProcIdx\n");
  }

  return 0;
}

#
# clean_up_compl_files - purge files indicating completion of processes.
#                        Purging will start from the first process given
#                        and end at the last process given.
#
# parameters:
#   - base of process completion file name
#   - reference to an array of process ids
#   - First process whose completion files are to be purged
#   - Last processes whose completion files are to be purged
#   - an indicator of whether to print debugging info
#
sub clean_up_compl_files ($$$$$) {
  my ($FileNameBase, $ProcIds, $FirstProc, $LastProc, $DebugOn) =  @_;

  my $ps;

  if ($DebugOn) {
    log_msg(qq#clean_up_compl_files: FileNameBase = $FileNameBase, FirstProc = $FirstProc, LastProc = $LastProc\n#);
  }

  for ($ps=$FirstProc; $ps <= $LastProc; $ps++) {
    my $DoneFile = done_file_name($FileNameBase, $ProcIds->[$ps]);
    
    if (-e $DoneFile && -f $DoneFile) {
      if ($DebugOn) {
        log_msg(qq#clean_up_compl_files: call sureunlink to remove $DoneFile\n#);
      }

      sureunlink($DoneFile, $DebugOn);
      
      if ($DebugOn) {
        log_msg(qq#clean_up_compl_files: removed $DoneFile\n#);
      }
    }
  }
}

#
# get_log_file_base_path - determine base path for log files
#
# Parameters:
#   - log directory, as specified by the user
#   - base for log file names
#   - an indicator of whether to produce debugging info
#
sub get_log_file_base_path ($$$) {
  my ($LogDir, $LogBase, $DebugOn) = @_;

  if ($LogDir) {
    if ($DebugOn) {
      log_msg <<msg;
get_log_file_base_path: log file directory = $LogDir
msg
    } 
  } else {
    # if LogDir was not specified, use current directory; returning unqualified
    # $LogBase (like we used to do when LogDir was not supplied) sometimes 
    # caused catcon.pl to hang waiting for the SQL*Plus process to complete, 
    # presumably because it could not write the .done file
    $LogDir = cwd();

    if ($DebugOn) {
      log_msg <<msg;
get_log_file_base_path: no log file directory was specified - using current dirctory ($LogDir)
msg
    }
  }

  return ($LogDir."/".$LogBase);
}

#
# send_sig_to_procs - given an array of process ids, send specified signal to 
#                     these processes 
#
# Parameters:
#   - reference to an array of process ids
#   - signal to send
#
# Returns:
#   number of processes whose ids were passed to us which got the signal
#
sub send_sig_to_procs (\@$) {
  my ($ProcIds, $Sig)  =  @_;
  
  return  kill($Sig, @$ProcIds);
}

#
# next_proc - determine next process which has finished work assigned to it
#             (and is available to execute a script or an SQL statement)
#
# Description:
#   This subroutine will look for a file ("done" file) indicating that the 
#   process has finished running a script or a statement which was sent to it.
#   Once such file is located, it will be deleted and a number of a process 
#   to which that file corresponded will be returned to the caller
#
# Parameters:
#   - number of processes from which we may choose the next available process
#   - total number of processes which were started (used when we try to
#     determine if some processes may have died)
#   - number of the process starting with which to start our search; if the 
#     supplied number is greater than $ProcsUsed, it will be reset to 0
#   - reference to an array of booleans indicating status of which processes 
#     should be ignored; may be undefined
#   - base for names of files whose presense will indicate that a process has 
#     completed
#   - reference to an array of process ids
#   - an indicator of whether to recover from death of a child process
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - name of the "kill session script"
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - an indicator of whether to display diagnostic info
#
sub next_proc ($$$$$$$\@$$$) {
  my ($ProcsUsed, $NumProcs, $StartingProc, $ProcsToIgnore, 
      $FilePathBase, $ProcIds, $RecoverFromChildDeath, 
      $IntConnectString, $KillSessScript, $DoneCmd, $DebugOn) = @_;

  if ($DebugOn) {
    my $msg = <<next_proc_DEBUG;
  running next_proc(ProcsUsed    = $ProcsUsed, 
                    NumProcs     = $NumProcs,
                    StartingProc = $StartingProc,
                    FilePathBase = $FilePathBase,
                    RecoverFromChildDeath = $RecoverFromChildDeath,
                    KillSessScript = $KillSessScript,
                    DoneCmd      = $DoneCmd,
                    DebugOn      = $DebugOn);

next_proc_DEBUG
    log_msg($msg);
  }

  # process number; will be used to construct names of "done" files
  # before executing the while loop for the first time, it will be set to 
  # $StartingProc.  If that number is outside of [0, $ProcsUsed-1], 
  # it [$CurProc] will be reset to 0; for subsequent iterations through the 
  # while loop, $CurProc will start with 0
  my $CurProc = ($StartingProc >= 0 && $StartingProc <= $ProcsUsed-1) 
    ? $StartingProc : 0;
    
  # look for *.done files which will indicate which processes have 
  # completed their work

  # we may end up waiting a while before finding an available process and if 
  # debugging is turned on, user's screen may be flooded with 
  #     next_proc: Skip checking process ...
  # and
  #     next_proc: Checking if process ... is available
  # messages.  
  #
  # To avoid this, we will print this message every 10 second or so.  Since 
  # we check for processes becoming available every 0.01 of a second (or so), 
  # we will report generate debugging messages every 1000-th time through the 
  # loop
  my $itersBetweenMsgs = 1000;

  for (my $numIters = 0; ; $numIters++) {
    #
    # Bug 22887047: now that we no longer try to catch SIGCHLD, we will check 
    #   if the next process is still alive, and if it is not, we will start a 
    #   new process in its place and proceed as if nothing happened
    # 

    for (; $CurProc < $ProcsUsed; $CurProc++) {
        
      if (   $ProcsToIgnore && @$ProcsToIgnore 
          && $ProcsToIgnore->[$CurProc]) {
        if ($DebugOn && $numIters % $itersBetweenMsgs == 0) {
          log_msg("next_proc: Skip checking process $CurProc\n");
        }

        next;
      }

      # Bug 22887047: check if the next process is still alive;  if it is no 
      #   longer with us, start a new process in its place
      my $procAlive = kill 0, $ProcIds->[$CurProc];

      # catconBounceDeadProcess will return non-zero only if 
      # catconBounceProcesses was unable to bounce it, in which case we will 
      # report an error to the caller
      if (!$procAlive) {
        if ($DebugOn) {
          log_msg("next_proc: process $CurProc (PID=$ProcIds->[$CurProc]) ".
                  "is dead\n");
        }

        # LRG 19633516: check whether the caller indicated that death of a 
        # child process should result in death of catcon
        if (!$RecoverFromChildDeath) {
          if ($DebugOn) {
            my $msg = <<msg;
next_proc: caller has not requested recovery from death of SQL*Plus processes -
           will release resources and exit
msg
            log_msg($msg);
          }

          # LRG 19650060: need to ignore SIGPIPE. Even though I am avoiding 
          #   sending any new commands to the process once I know that it is 
          #   dead, it's likely that the command to generate "done" file which
          #   was sent before the process has encountered a statement that 
          #   caused it to die is causing SIGPIPE
          if ($DebugOn) {
            log_msg("next_proc: SIGPIPE will be ignored\n");
          }

          $SIG{PIPE} = 'IGNORE';

          # release resources (including sending EXIT to each SQL*Plus 
          # process still running
          if ($DebugOn) {
            log_msg("next_proc: invoking catconWrapUp\n");
          }
          
          catconWrapUp();

          # in case some of the processes did not react to EXIT 
          # (e.g. because they were running some script), use 
          # ALTER SYSTEM KILL SESSION to terminate them
          if ($KillSessScript && (-e $KillSessScript)) {
            if ($DebugOn) {
              log_msg("next_proc: invoking kill_sqlplus_sessions\n");
            }

            kill_sqlplus_sessions(@$IntConnectString, 
                                  $KillSessScript, $DoneCmd, 
                                  done_file_name_prefix($FilePathBase, $$),
                                  $DebugOn);
          }

          # if any of them are still around, send SIGKILL to them
          if ($DebugOn) {
            log_msg("next_proc: sending SIGKILL to SQL*Plus processes\n");
          }
          
          send_sig_to_procs(@$ProcIds, 9);

          if ($DebugOn) {
            log_msg("next_proc: returning -1 to the caller\n");
          }

          return -1;
        }

        if ($DebugOn) {
          my $msg = <<msg;
next_proc: caller has requested recovery from death of SQL*Plus processes
msg
          log_msg($msg);
        }

        # id of the apparently dead process
        my $deadProcId = $ProcIds->[$CurProc];

        if ($DebugOn) {
          my $msg = <<msg;
next_proc: SQL*Plus process $deadProcId (ProcIds[$CurProc]) appears 
           to have died. Will try to start another process in its place
msg
          log_msg($msg);
        }

        if (catconBounceDeadProcess($CurProc)) {
          my $msg = <<msg;
next_proc: Attempt to start a new SQL*Process to take place of process 
           $deadProcId (ProcIds[$CurProc]) failed
msg
          log_msg($msg);
          
          return -1;
        }

        if ($DebugOn) {
          my $msg = <<msg;
next_proc: new SQL*Plus process $ProcIds->[$CurProc] was successfully started 
           to take place of process $deadProcId (ProcIds[$CurProc]) which 
           appears to have died
msg
          log_msg($msg);
        }
      }

      # file which will indicate that process $CurProc finished its work
      #
      # Bug 18011217: append _catcon_$ProcIds->[$CurProc] to $FilePathBase 
      # to avoid conflicts with other catcon processes running on the same 
      # host
      my $DoneFile = done_file_name($FilePathBase, $ProcIds->[$CurProc]);

      if ($DebugOn && $numIters % $itersBetweenMsgs == 0) {
        log_msg("next_proc: Checking if process $CurProc (id = $ProcIds->[$CurProc]) is available\n");
      }

      #
      # Is file is present, remove the "done" file (thus making this process 
      # appear "busy") and return process number to the caller.
      #
      if (-e $DoneFile) {
        if ($DebugOn) {
          log_msg("next_proc: call sureunlink to remove $DoneFile\n");
        }

        sureunlink($DoneFile, $DebugOn);

        if ($DebugOn) {
          log_msg("next_proc: process $CurProc is available\n");
        }

        return $CurProc;
      }
    }
    select (undef, undef, undef, 0.01);

    $CurProc = 0;
  }
  
  return -1;  # this statement will never be reached
}

#
# wait_for_completion - wait for completion of processes
#
# Description:
#   This subroutine will wait for processes to indicate that they've 
#   completed their work by creating a "done" file.  Since it will use 
#   next_proc() (which deleted "done" files of processes whose ids it returns) 
#   to find processes which are done running, this subroutine will generate 
#   "done" files once all processes are done to indicate that they are 
#   available to take on more work.
#
# Parameters:
#   - number of processes which were used (and whose completion we need to 
#     confirm)
#   - total number of processes which were started
#   - base for generating names of files whose presense will indicate that a 
#     process has completed
#   - references to an array of process file handles
#   - reference to a array of process ids
#   - command which will be sent to a process to cause it to generate a 
#     "done" file
#   - reference to an array of statements, if any, to be issued when a process 
#     completes
#   - if connected to a CDB, name of the Root Container into which we should 
#     switch (Bug 17898118: ensures that no process is connected to a PDB 
#     which may get closed by some script (which would run only if there are 
#     no processes running scripts in that PDB))
#   - internal connect strings - [0] used to connect to a DB, 
#                                [1] can be used to produce diagnostic message
#   - an indicator of whether to recover from death of a child process
#   - name of the "kill session script"
#   - an indicator of whether the instance may be shut down in which case we 
#     do not want to echo "done" statement to the sql log file
#   - an indicator of whether to display diagnostic info
#
sub wait_for_completion ($$$\@\@$\@$\@$$$$) {
  my ($ProcsUsed, $NumProcs, $FilePathBase, $ProcFileHandles, 
      $ProcIds, $DoneCmd, $EndStmts, $Root, $InternalConnectString, 
      $RecoverFromChildDeath, $KillSessScript, $InstShutDown, $DebugOn) = @_;

  if ($DebugOn) {
    my $RootVal = (defined $Root) ? $Root : "undefined";
    my $msg = <<wait_for_completion_DEBUG;
  running wait_for_completion(ProcsUsed             = $ProcsUsed, 
                              FilePathBase          = $FilePathBase,
                              DoneCmd               = $DoneCmd,
                              Root                  = $RootVal,
                              InternalConnectString = $InternalConnectString->[1],
                              RecoverFromChildDeath = $RecoverFromChildDeath,
                              KillSessScript        = $KillSessScript,
                              DebugOn               = $DebugOn);

wait_for_completion_DEBUG
      log_msg($msg);
  }

  # process number
  my $CurProc = 0;
    
  if ($DebugOn) {
    log_msg("wait_for_completion: waiting for $ProcsUsed processes to complete\n");
  }

  # look for *.done files which will indicate which processes have 
  # completed their work

  my $NumProcsCompleted = 0;  # how many processes have completed

  # this array will be used to keep track of processes which have completed 
  # so as to avoid checking for existence of files which have already been 
  # seen and removed
  my @ProcsCompleted = (0) x $ProcsUsed;

  while ($NumProcsCompleted < $ProcsUsed) {
    $CurProc = next_proc($ProcsUsed, $NumProcs, $CurProc + 1, \@ProcsCompleted,
                         $FilePathBase, $ProcIds, $RecoverFromChildDeath,
                         @$InternalConnectString, $KillSessScript, $DoneCmd, 
                         $DebugOn);

    if ($CurProc < 0) {
      log_msg(qq#wait_for_completion: unexpected error in next_proc()\n#);
      return 1;
    }

    # if the caller has supplied us with statements to run after a 
    # process finished its work, do it now
    if ($EndStmts && $#$EndStmts >= 0) {

      if ($DebugOn) {
        log_msg("wait_for_completion: sending completion statements to process $CurProc:\n");
      }

      foreach my $Stmt (@$EndStmts) {

        # $Stmt may contain %proc strings which need to be replaced with 
        # process number, but we don't want to modify the statement in 
        # @$EndStmts, so we modify a copy of it
        my $Stmt1;

        ($Stmt1 = $Stmt) =~ s/%proc/$CurProc/g;
        
        if ($DebugOn) {
          log_msg("\t$Stmt1\n");
        }

        printToSqlplus("wait_for_completion", $ProcFileHandles->[$CurProc],
                       $Stmt1, "\n", $DebugOn);
      }

      $ProcFileHandles->[$CurProc]->flush; # flush the buffer
    }

    # Bug 17898118: if connected to a CDB, switch the process into the Root
    # to avoid getting caught connected to a PDB that gets closed. 
    #
    # Before switching, however, we need to check to make sure the CDB is 
    # open (some scripts shut down the CDB, and trying to switch into the 
    # Root while the CDB is closed results in errors
    #
    # Bug 18011217: append _catcon_$ProcIds->[0] to $FilePathBase to avoid 
    # conflicts with other catcon processes running on the same host
    if ($Root) {
      my $instanceStatus = 
        get_instance_status(@$InternalConnectString, $DoneCmd, 
                            done_file_name_prefix($FilePathBase, 
                                                  $ProcIds->[0]), 
                            $DebugOn);
      if ((defined $instanceStatus)) {
        if ($instanceStatus =~ /^OPEN/) {
          if ($DebugOn) {
            log_msg("wait_for_completion: switching process $CurProc into $Root\n");
          }

          print {$ProcFileHandles->[$CurProc]} 
            qq#ALTER SESSION SET CONTAINER = "$Root"\n/\n#;
          $ProcFileHandles->[$CurProc]->flush; # flush the buffer
        } elsif ($DebugOn) {
          log_msg("wait_for_completion: process $CurProc not switched into $Root because the CDB is not open\n");
        }
      } else {
        # instance status could not be obtained
        log_msg("wait_for_completion: unexpected error in get_instance_status\n");
        return 1;
      }
    }

    if ($DebugOn) {
      log_msg("wait_for_completion: process $CurProc is done\n");
    }

    $NumProcsCompleted++; # one more process has comleted

    # remember that this process has completed so next_proc does not try to 
    # check its status
    $ProcsCompleted[$CurProc] = 1;
  }

  if ($DebugOn) {
    log_msg("wait_for_completion: All $NumProcsCompleted processes have completed\n");
  }
  
  # issue statements to cause "done" files to be created to indicate
  # that all $ProcsUsed processes are ready to take on more work
  for ($CurProc = 0; $CurProc < $ProcsUsed; $CurProc++) {

    # file which will indicate that process $CurProc finished its work
    #
    # Bug 18011217: append _catcon_$ProcIds->[$CurProc] to $FilePathBase to 
    # avoid conflicts with other catcon processes running on the same host
    my $DoneFile = done_file_name($FilePathBase, $ProcIds->[$CurProc]);

    printToSqlplus("wait_for_completion", $ProcFileHandles->[$CurProc],
                   qq/$DoneCmd $DoneFile/,
                   "\n", $DebugOn && !$InstShutDown);

    # flush the file so a subsequent test for file existence does 
    # not fail due to buffering
    $ProcFileHandles->[$CurProc]->flush;

    if ($DebugOn) {
      my $msg = <<msg;
wait_for_completion: sent "$DoneCmd $DoneFile" 
    to process $CurProc (id = $ProcIds->[$CurProc]) to indicate that it is 
    available to take on more work
msg
      log_msg($msg);
    }
  }

  return 0;
}

#
# return a timestamp in yyyy-mm-dd h24:mi:sec format
#
sub TimeStamp {
  my ($sec,$min,$hour,$mday,$mon,$year)=localtime(time);

  return sprintf "%4d-%02d-%02d %02d:%02d:%02d",
                 $year+1900,$mon+1,$mday,$hour,$min,$sec;
}

#
# getSpoolFileNameSuffix - generate suffix for a spool file names using 
#   container name supplied by the caller
#
# Parameters:
# - container name (IN)
#
sub getSpoolFileNameSuffix ($) {
 my ($SpoolFileNameSuffix) = @_;

 # $SpoolFileNameSuffix may contain characters which may be 
 # illegal in file name - replace them all with _
 $SpoolFileNameSuffix =~ s/\W/_/g;

 return lc($SpoolFileNameSuffix);
}

#
# pickNextProc - pick a process to run next statement or script
#
# Parameters:
#   (IN) unless indicated otherwise
#   - number of processes from which we may choose the next available process 
#   - total number of processes which were started (used when we try to
#     determine if some processes may have died)
#   - number of the process starting with which to start our search; if the 
#     supplied number is greater than $ProcsUsed, it will be reset to 0
#   - base for names of files whose presense will indicate that a process has 
#     completed
#   - reference to an array of process ids
#   - an indicator of whether to recover from death of a child process
#   - connect strings - [0] used to connect to a DB, 
#                       [1] can be used to produce diagnostic message
#   - name of the "kill session script"
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - an indicator of whether to display diagnostic info
# 
sub pickNextProc ($$$$$$\@$$$) {
  my ($ProcsUsed, $NumProcs, $StartingProc, $FilePathBase, 
      $ProcIds, $RecoverFromChildDeath, $IntConnectString, $KillSessScript, 
      $DoneCmd, $DebugOn) = @_;

  # find next available process
  my $CurProc = next_proc($ProcsUsed, $NumProcs, $StartingProc, undef,
                          $FilePathBase, $ProcIds, $RecoverFromChildDeath, 
                          @$IntConnectString, $KillSessScript, $DoneCmd, 
                          $DebugOn);
  if ($CurProc < 0) {
    # some unexpected error was encountered
    log_msg("pickNextProc: unexpected error in next_proc\n");

    return -1;
  }

  return $CurProc;
}

#
# firstProcUseStmts - issue statements to a process that is being used for 
#                     the first time
# Parameters:
#   - value of IDENTIFIER if ERRORLOGGING is enabled (IN)
#   - value of custom Errorlogging Identifier; if defined, $ErrLoggingIdent 
#     will be used as is, without appending "InitStmts" (IN)
#   - name of error logging table, if defined (IN)
#   - reference to an array of file handle references (IN)
#   - number of the process about to be used (IN)
#   - reference to an array of statements which will be executed in every 
#     process before it is asked to run the first script (IN)
#   - array used to keep track of processes to which statements contained in 
#     @$PerProcInitStmts_REF need to be sent because they [processes]
#     have not been used in the course of this invocation of catconExec()
#   - an indicator of whether we are running a user script (meaning that 
#     _ORACLE_SCRIPT was not set and so err_logging_tbl_stmt does not need to 
#     temporarily reset it) (IN)
#   - name of a Federation Root if it was supplied by the caller of catcon to 
#     indicate that scripts are to be run against Containers comprising a 
#     Federation rooted in that Container; if set, _federation_script may be 
#     set and so needs to be temporarily reset in err_logging_tbl_stmt (IN)
#   - an indicator of whether debugging is enabled (IN)
#
sub firstProcUseStmts ($$$$$$$$$$) {
  my ($ErrLoggingIdent, $CustomErrLoggingIdent, $ErrLogging, $FileHandles_REF, 
      $CurProc, $PerProcInitStmts_REF, $NeedInitStmts_REF, $UserScript, 
      $FedRoot, $DebugOn) = @_;
  
  if ($ErrLogging) {
    # construct and issue SET ERRORLOGGING ON [TABLE...] statement
    err_logging_tbl_stmt($ErrLogging, $FileHandles_REF, $CurProc, $UserScript, 
                         $FedRoot, $DebugOn);

    if ($ErrLoggingIdent) {
      # send SET ERRORLOGGING ON IDENTIFIER ... statement
      my $Stmt;
      if ($CustomErrLoggingIdent) {
        # NOTE: if the caller of catconExec() has supplied a custom 
        #       Errorlogging Identifier, it has already been copied into 
        #       $ErrLoggingIdent which was then normalized, so our use of 
        #       $ErrLoggingIdent rather than $CustomErrLoggingIdent below is 
        #       intentional
        $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent, 0, 256)."'";
      } else {
        $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent."InitStmts", 0, 256)."'";
      }

      if ($DebugOn) {
        log_msg("firstProcUseStmts: sending $Stmt to process $CurProc\n");
      }

      printToSqlplus("firstProcUseStmts", $FileHandles_REF->[$CurProc],
                     $Stmt, "\n", $DebugOn);
    }
  }

  if ($DebugOn) {
    log_msg("firstProcUseStmts: sending init statements to process $CurProc:\n");
  }

  foreach my $Stmt (@$PerProcInitStmts_REF) {

    # $Stmt may contain %proc strings which need to be replaced with 
    # process number, but we don't want to modify the statement in 
    # @$PerProcInitStmts_REF, so we modify a copy of its element
    my $Stmt1;

    ($Stmt1 = $Stmt) =~ s/%proc/$CurProc/g;

    if ($DebugOn) {
      log_msg("\t$Stmt1\n");
    }

    printToSqlplus("firstProcUseStmts", $FileHandles_REF->[$CurProc],
                   $Stmt1, "\n", $DebugOn);
  }

  # remember that "initialization" statements have been run for this 
  # process
  $NeedInitStmts_REF->[$CurProc] = 0;

  return;
}

sub printToSqlplus ($$$$$) {
  my ($func, $fh, $stmt, $tail, $DebugOn) = @_;

  if ($DebugOn) {
    my $printableStmt = $stmt;
    $printableStmt  =~ s/['"]/#/g; # mask quotes to avoid confusion
    $printableStmt  =~ s/\n/#LF#/g; # mask \n
    my $debugQry = qq{select '$func(): $printableStmt' as catcon_statement from dual\n/\n};
    print {$fh} $debugQry;
  }

  print {$fh} qq#$stmt$tail#;
}

#
# additionalInitStmts - issue additional initialization statements after 
#   picking a new process to run a script in a given Container
#
# Parameters: (all parameters are IN unless noted otherwise)
#   - reference to an array of file handles used to communicate to processes
#   - process number of the process which was picked to run the next script
#   - name of the container against which the next script will be run
#     (can be null if running against ROOT or in a non-CDB
#   - query which will identify current container in the log
#     (can be null if running against ROOT or in a non-consolidated DB)
#   - process id of the process hich was picked to run the next script
#   - an inidcator of whether to produce debugging info
#   - an indicator of whether called by cdb_sqlexec.pl
#   - name of a Federation Root if called to run scripts against all 
#     Containers comprising a Federation rooted in the specified Container
#
sub additionalInitStmts ($$$$$$$$$) {
  my ($FileHandles_REF, $CurProc, $CurConName,
      $CurrentContainerQuery, $CurProcId, $EchoOn, $DebugOn, $UserScript,
      $FedRoot) = @_;

  # if not running a user script, 
  #   _ORACLE_SCRIPT needs to be set
  # else if Federation Root name was supplied
  #   _FEDERATION_SCRIPT needs to be set
  if (!$UserScript) {
    printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                   qq#ALTER SESSION SET "_ORACLE_SCRIPT"=TRUE#, "\n/\n", 
                   $DebugOn);
  } elsif ($FedRoot) {
    printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                   qq#ALTER SESSION SET "_FEDERATION_SCRIPT"=TRUE#, "\n/\n", 
                   $DebugOn);
  }

  # set tab and trimspool to on, so that spooled log files start 
  # with a consistent setting
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#SET TAB ON#, "\n", 
                 $DebugOn);

  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#SET TRIMSPOOL ON#, "\n", 
                 $DebugOn);

  # ensure that common SQL*PLus vars that affect appearance of log files are 
  # set to consistent values.  This will ensure that vars modified in one 
  # script do not affect appearance of log files produced by running another 
  # script or running the same script in a different Container
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set colsep ' '#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set escape off#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set feedback 6#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set heading on#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set linesize 80#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set long 80#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set newpage 1#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set numwidth 10#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set pagesize 14#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set recsep wrapped#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set showmode off#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set sqlprompt "SQL> "#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set termout on#, "\n", 
                 $DebugOn);
  #@@@@ Don't overide these values needed for performance work
  #printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
  #               qq#set time off#, "\n", 
  #               $DebugOn);
  #printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
  #               qq#set timing off#, "\n", 
  #               $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set trimout on#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set underline on#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set verify on#, "\n", 
                 $DebugOn);
  printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                 qq#set wrap on#, "\n", 
                 $DebugOn);
          
  if ($CurConName) {
    printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                   qq#ALTER SESSION SET CONTAINER = "$CurConName"#, "\n/\n", 
                   $DebugOn);

    if ($DebugOn) {
      log_msg("additionalInitStmts: process $CurProc (id = $CurProcId) connected to Container $CurConName\n");
    }
  }

  if ($CurrentContainerQuery) {
    print {$FileHandles_REF->[$CurProc]} $CurrentContainerQuery;
  }

  if ($EchoOn) {
    printToSqlplus("additionalInitStmts", $FileHandles_REF->[$CurProc], 
                   qq#SET ECHO ON#, "\n", 
                   $DebugOn);
  }

  return;
}

sub find_in_path($$) {
  my ($exec, $windows) = @_;
  my @pathext = ('');

  # for Windows, look for $exec.<one of extensions (e.g. .exe or .bat)>
  if ($windows) {
    if ( $ENV{PATHEXT} ) {
      push @pathext, split ';', $ENV{PATHEXT};
    } else {
      # if PATHEXT is not set use one of .com, .exe or .bat
      push @pathext, qw{.com .exe .bat};
    }
  }

  my @path = File::Spec->path;
  if ($windows) {
    unshift @path, File::Spec->curdir;
  }
 
  foreach my $base ( map { File::Spec->catfile($_, $exec) } @path ) {
    for my $ext ( @pathext ) {
      my $file = $base.$ext;
 
      # We don't want dirs (as they are -x)
      next if -d $file;
      
      if (
          # Executable, normal case
          -x _
          or (
              (
               $windows
               and
               grep {$file =~ /$_\z/i} @pathext[1..$#pathext])
              # DOSish systems don't pass -x on
              # non-exe/bat/com files. so we check -e.
              # However, we don't want to pass -e on files
              # that aren't in PATHEXT, like README.
              and -e _)
         ) {
        return 1;
      }
    }
  }

  return undef;
}

#
# log_script_execution - produce a message indicating script or statement about
#                        to be executed as well as the name of a Container 
#                        (if applicable) in which it is being executed
#
sub log_script_execution (\$$$) {
  my ($FilePath, $ConName, $ProcIdx) = @_;

  my $msg;
  
  $msg = "executing".(($$FilePath =~ /^@/) ? " " : " statement ");
  $msg .= "\"".$$FilePath."\" in ".($ConName ? "container ".$ConName : "non-CDB");
  
  log_msg("catconExec_int: ".$msg." using process $ProcIdx\n");
}

# 
# os_dependent_stuff: set OS-dependent variables
#
# Parameters:
#   - reference to a variable storing an indication of whether we are 
#     running on Windows
#   - reference to a variable storing a command which must be set to SQL*Plus 
#     to cause a "done" file to be created
#
sub os_dependent_stuff(\$\$) {
  my ($isWindows, $doneCmd) = @_;

  my $UnixDoneCmd = "\nhost sqlplus -v >";
  my $WindowsDoneCmd = "\nhost sqlplus/nolog -v >";

  # figure out if we are running under Windows and set $catcon_DoneCmd
  # accordingly

  $$doneCmd = ($$isWindows = ($OSNAME =~ /^MSWin/)) ? $WindowsDoneCmd : $UnixDoneCmd;

  return;
}

# compute number of processes which will be used to run 
# script/statements specified by the caller in all remaining PDBs; 
# value returned by this subroutine will be used to determine when all 
# processes finished their work
sub compute_procs_used_for_pdbs($$$$) {
  my ($singleThreaded, $numPDBs, $numProcs, $scriptPaths) = @_;

  my $ProcsUsed;
  
  if ($singleThreaded) {
    $ProcsUsed = ($numPDBs > $numProcs) ? $numProcs : $numPDBs;
  } else {
    $ProcsUsed = ($scriptPaths * $numPDBs > $numProcs) 
      ? $numProcs : $scriptPaths * $numPDBs;
  }
  
  return $ProcsUsed;
}

# subroutines which may be invoked by callers outside this file and variables 
# that need to persist across such invocations
{
  # have all the vars that need to persist across calls been initialized?
  # computed
  my $catcon_InitDone;
  
  # name of the directory for sqlplus script(s), as supplied by the caller
  my $catcon_SrcDir;

  # name of the directory for log file(s), as supplied by the caller
  my $catcon_LogDir;

  # base for paths of log files
  my $catcon_LogFilePathBase;

  # number of processes which will be used to run sqlplus script(s) or SQL 
  # statement(s) using this instance of catcon.pl, as supplied by the caller; 
  my $catcon_NumProcesses;

  # indicator of whether echo should be turned on while running sqlplus 
  # script(s) or SQL statement(s), as supplied by the caller
  my $catcon_EchoOn;

  # indicator of whether output of running scripts should be spooled into 
  # Container- and script-specific spool files, as supplied by the caller
  my $catcon_SpoolOn;

  # indicator of whether debugging info should be generated while executing 
  # various subroutines in this block
  my $catcon_DebugOn;

  # indicator of whether verbose output should be generated while executing 
  # various subroutines in this block; verbose output is not quite as chatty 
  # detailed as debugging
  my $catcon_Verbose;

  # an indicator of whether we are being invoked from a GUI tool which on 
  # Windows means that the passwords and hidden parameters need not be hidden
  my $catcon_GUI;

  # indicator of whether the caller wants us to check whether scripts end 
  # without committing the current transaction
  my $catcon_UncommittedXactCheck;

  # indicator of whether we were instructed to run scripts in ALL PDBs and 
  # then in Root
  my $catcon_PdbsThenRoot;

  # indicator of whether non-existent and closed PDBs should be ignored when 
  # constructing a list of Containers against which to run scripts
  my $catcon_IgnoreInaccessiblePDBs;

  # indicator to ignored all validation of Pdb's when 
  # constructing a list of Containers against which to run scripts
  my $catcon_IgnoreAllInaccessiblePDBs = 0;

  my $catcon_EZConnect;

  # errors that a caller can instruct us to ignore; for example, 
  # if a caller indicates that we can ignore script_path errors, 
  # $catcon_ErrorsToIgnore{script_path} will get set to 1 and if 
  # validate_script_path cannot validate the specified script path, the 
  # errors will be reported but ignored
  my %catcon_ErrorsToIgnore = (
      script_path => 0
    );

  # name of a Root of a Federation against members of which script(s) are 
  # to be run; if defined and $catcon_UserScript is not set, _federation_script
  # will be set
  my $catcon_FedRoot;

  # an indicator of whether we were told that the DB has been shut down
  my $catcon_DbClosed;

  # Array of 2 elements: 
  # - connect string used when running SQL for internal statements: this
  #   should be done as sys
  # - same as above, but with password redacted - will be used when producing 
  #   diagnostic output (bug 21202842)
  # computed
  my @catcon_InternalConnectString;        

  # Array of 2 elements: 
  # - connect string used when running sqlplus script(s) or SQL statement(s) in
  #   the context of the user passed in.
  # - same as above, but with password redacted - will be used when producing 
  #   diagnostic output (bug 21202842)
  # computed
  my @catcon_UserConnectString;

  # password used when connecting to the database to run sqlplus script(s) or 
  # SQL statement(s)
  my $catcon_UserPass;

  # names of ALL containers of a CDB if connected to one; empty if connected 
  # to a non-CDB;
  # computed
  my @catcon_AllContainers;

  # indicators (Y or N) of whether a Container in @catcon_AllContainers is open
  my %catcon_IsConOpen;

  # empty if connected to a non-CDB; same as @catcon_AllContainers if the 
  # user did not specify -c or -C flag, otherwise, consists of 
  # names of Containers explicitly included (-c) or not explicitly 
  # excluded (-C);
  #
  # NOTE:
  #   by default, catconExec will run scripts/statements in all Containers 
  #   whose names are found in catcon_Containers; however, this can be 
  #   modified by passing to catconExec a list of Containers to include or 
  #   exclude during the current invocation 
  # computed
  my @catcon_Containers;

  # environment variable containing EZConnect Strings for instances to be 
  # used to run scripts
  my $EZCONNECT_ENV_TAG = "CATCONEZCONNECT";

  # name of the Root if operating on a Consolidated DB
  my $catcon_Root;

  # array of file handle references; 
  my @catcon_FileHandles;

  # array of process ids
  my @catcon_ProcIds;

  # are we running on Windows?
  my $catcon_Windows;
  my $catcon_DoneCmd;

  # string introducing an argument to SQL*PLus scripts which will be supplied 
  # using clear text
  my $catcon_RegularArgDelim;

  # string introducing an argument to SQL*PLus scripts which will have to be 
  # specified by the user at run-time
  my $catcon_SecretArgDelim;

  # if strings used to introduce "regular" as well as secret arguments are 
  # specified and one is a prefix of the other and we encounter a string 
  # that could be either a regular or a secret argument, we will resolve in 
  # favor of the longer string.  
  # $catcon_ArgToPick will be used to determine how to resolve such conflicts, 
  # should they occur; it will remain set to 0 if such conflicts cannot occur 
  # (i.e. if both strings are not specified or if neither is a prefix of the 
  # other)

  # pick regular argument in the event of conflict
  my $catcon_PickRegularArg;
  # pick secret argument in the event of conflict
  my $catcon_PickSecretArg;

  my $catcon_ArgToPick = 0;

  # reference to an array of statements which will be executed in every 
  # process before it is asked to run the first script
  my $catcon_PerProcInitStmts;

  # reference to an array of statements which will be executed in every 
  # process after it finishes runing the last script
  my $catcon_PerProcEndStmts;

  # if defined, will contain name of error logging table
  my $catcon_ErrLogging;

  # if set, SET ERRORLOGGING ON IDENTIFIER statement will NOT be issued
  my $catcon_NoErrLogIdent;

  # force_pdb_modes() gets handed an array of names of PDBs which need to be 
  # open in a specified mode (see catcon_AllPdbMode) before running scripts 
  # against them.  
  #
  # Some of these PDBs may already be open in the desired mode, while others 
  # may need to be closed and reopened in the desired mode. Since catconExec 
  # (which invokes force_pdb_modes() if $catcon_AllPdbMode indicates that the 
  # caller has requested that all relevant PDBs be open in a specified mode) 
  # may be invoked repeatedly with different lists of Containers on which to 
  # operate, we need to be able to compute a list of PDBs whose mode needs to 
  # be checked and possibly changed without checking PDBs which have 
  # previously been estanblished to be open in the correct mode.
  #
  # This hash will be used to keep track of PDBs which we know to be open 
  # in requested mode
  my %catcon_PdbsOpenInReqMode;

  # mode in which all PDBs (including PDB$SEED, so if one insists on setting 
  # both $catcon_SeedPdbMode and $catcon_AllPdbMode, the latter will take 
  # precedence) to be operated upon should be open
  #
  # This variable does not get set in catconInit (to avoid changing its 
  # signature), so we initialize it in such a way that if the caller does 
  # not invoke the subroutine that sets it, it will end up being set to the 
  # same value as if the caller were to invoke it and tell us that the user 
  # has not specified mode in which all PDBs should be opened
  my $catcon_AllPdbMode = CATCON_PDB_MODE_NA;

  # $catcon_SeedPdbMode serves the same purpose as $catcon_AllPdbMode, but 
  # only for PDB$SEED 
  #
  # NOTE: if a caller specifies a value for $catcon_AllPdbMode, it will 
  #       override whatever value was specified for $catcon_SeedPdbMode, so 
  #       there is never a reason to set them both
  my $catcon_SeedPdbMode = CATCON_PDB_MODE_NA;

  # $catcon_RevertUserPdbModes will be used to store 
  #   (mode.' '. restricted) => ref {instance-name => ref @pdb-names}
  # records for user PDBs on which we need to operate and which were not open 
  # in the mode specified by the caller. Contents of this hash will act as a 
  # reminder that these PDBs need to be reopened in their erstwhile modes as 
  # a part of catconRevertPdbModes().
  # The reason for choosing this hash layout is to minimize number of
  #  ALTER PDB OPEN <mode> [RESTRICTDED] INSTANCES=...
  # statements that need to be issued to restore user PDB modes in 
  # catconRevertPdbModes.
  #
  # NOTE: if a PDB was not opened in desired mode (as indicated by 
  #       $catcon_AllPdbMode) when catconExec() discovers for the first time 
  #       that it needs to run some scripts or statements against it, it will 
  #       call force_pdb_modes() which will call reset_pdb_modes() which will 
  #       close it and then reopen in desired mode on all instances.  From 
  #       that point on, that PDB will remain in that mode until it gets 
  #       reset to its original mode in catconRevertPdbModes().  
  #
  #       It is IMPORTANT that restoring PDB's mode happens AFTER all 
  #       processes opened in catconInit() have been killed because otherwise 
  #       ALTER PDB CLOSE IMMEDIATE issued by reset_pdb_modes() will 
  #       cause any of the processes which happen to be connected to that PDB
  #       to become unusable (they will be disconnected from the database 
  #       and any statements sent to them will fail.)  
  #       As a side note, before we fixed bug 13072385, reset_pdb_modes() 
  #       used to issue ALTER PDB CLOSE (without IMMEDIATE), which hung if 
  #       any process was connected to PDB$SEED (which was the only PDB on 
  #       which it was operating (and at that time the subroutine was called 
  #       reset_seed_pdb_mode))
  #
  my %catcon_RevertUserPdbModes;

  # %catcon_RevertSeedPdbMode serves the same purpose as 
  # %catcon_RevertUserPdbModes but only for PDB$SEED
  my %catcon_RevertSeedPdbMode;

  # environment variable containing user password
  my $USER_PASSWD_ENV_TAG = "CATCONUSERPASSWD";

  # environment variable containing internal password
  my $INTERNAL_PASSWD_ENV_TAG = "CATCONINTERNALPASSWD";

  # a global variable indicates whether catcon is called by cdb_sqlexec
  # If true then the _oracle_script will not be set and it will not execute 
  # in Root or PDB$SEED
  my $catcon_UserScript = 0;

  # this hash will be used to save signal handlers in effect before catconInit
  # was invoked so that they can be restored at the end of catconWrapUp
  my %catcon_SaveSig;

  my $CATCONOUT;

  # Bug 23020062: variable indicating whether pdb_lockdown should be disabled
  # while running script.
  my $catcon_DisableLockdown = 0;

  # LRG 19633516: variable indicating whether death of a spawned SQL*Plus 
  # process should result in catcon dying
  my $catcon_RecoverFromChildDeath = 0;

  # Bug 22887047: an index of an entry in @catcon_ProcIds that contains id of
  #   a process that died unexpectedly; will be set to -1 unless it gets set 
  #   by catconBounceDeadProcess when it gets called from next_proc upon 
  #   encountering a dead process
  my $catcon_DeadProc = -1;

  # Bug 22887047: id of a process that died unexpectedly; should not be 
  #   referenced unless $catcon_DeadProc != -1
  my $catcon_DeadProcId;

  # 
  # Bug 21871308: every SQL*Plus process that we start (in start_processes), 
  #   will add a line to the "kill all sessions script" which will be run from 
  #   the END block if SIGINT/SIGTERM/SIGQUIT is caught causing catcon to die. 
  #   Running the "kill all sessions script" will ensure that all SQL*Plus 
  #   processes started by us end gracefully, releasing their resources.
  #
  my $catcon_KillAllSessScript;

  # Bug 21871308: name of a script that will be invoked to generate a
  #   "kill session script" for a new SQL*Plus process
  my $catcon_GenKillSessScript;

  # Oracle DBMS version. Used to ensure that SQL statements generated by this
  # script are compatible with the version of the RDBMS to which we connect
  my $catcon_DbmsVersion;

  # an indicator of whether catcon should open PDBs in the mode indicated by 
  # $catcon_AllPdbMode (as long as that mode is not set to 
  # CATCON_PDB_MODE_UNCHANGED) on all available instances.
  my $catcon_AllInst;

  # hash mapping names of instances to 
  # - number of SQL*Plus processes that can be allocated for that instance
  #   ($catcon_InstProcMap{$inst}->{NUM_PROCS})
  # - offset of the id of the first such process in @catcon_ProcIds
  #   ($catcon_InstProcMap{$inst}->{FIRST_PROC})
  # if we are processing PDBs using all available instances
  my %catcon_InstProcMap;

  # hash mapping ids of processes (found in @catcon_ProcIds) to names of 
  # instances to which they are connected; currently will be used when we are
  # trying to recover from a death of a SQL*Plus process if it is posible 
  # that # not all of them are connected to the same instance, i.e. if 
  # %catcon_InstProcMap is defined
  my %catcon_ProcInstMap;

  # Bug 20193612: these variables will be used to save references to
  # instance->pdbs hash returned by force_pdb_modes.
  # The reason for saving them in these variables instead of just using values 
  # returned by force_pdb_modes is because force_pdb_modes will not compute 
  # them if catconExec gets invoked repeatedly with the same set of PDBs 
  # (or with a subsequent set of PDBs being a subset of an earlier set), but 
  # we still may need this information for forking of child processes to 
  # handle processing PDBs open in different instances 
  my $catcon_ForcePdbModeInstUserPdbMap;
  my $catcon_ForcePdbModeInstSeedPdbMap;

  # will be set to 1 in a forked child process; 
  # will be use by log_msg to deduce that it should not be writing output to 
  # STDERR to avoid confusion which would arise from multiple child processes 
  # dumping output to the same STDERR
  my $catcon_ChildProc;

  # Bug 25315864: hash mapping instance name to a connect string:
  # - if connected to a DB open on a single instance, instance name will be 
  #   mapped to an empty string 
  #
  #    NOTE: in some cases (bug 25366291), gen_inst_conn_strings which is 
  #          responsible for populating this hash may fetch 0 rows from the 
  #          RDBMS and leave this hash undefined.  Rather than treating it as 
  #          an error, we will force all processing to occur on the default 
  #          instance
  # - otherwise, instance name(s) will be mapped to <connect-string>(s)
  my %catcon_InstConnStr;

  # Bug 25363697: this variable will be used to remember whether the last 
  #  script run by catconExec in a given container was catshutdown.sql, which 
  #  will tell us to not send diagnostic statements to sqlplus processes 
  #  until the next script is run in that container (the assumption being 
  #  that that script will start the Container making it safe to send 
  #  diagnostic statements to a sqlplus process connected to the container.  
  #
  #  Prior to this change, code in catconExec tried to handle this by keeping 
  #  track of scripts executed during its current invocation, but that failed 
  #  to account for the script invoked during the preceeding invocation
  my %catcon_CatShutdown;

  #
  # log_msg - add supplied string to both STDERR and $CATCONOUT
  #
  sub log_msg($) {
    my ($LogMsg) = @_;
    
    # forked child processes write their output only to CATCONOUT
    if (!$catcon_ChildProc) {
      print STDERR $LogMsg;
    }

    # this subroutine may get called before $CATCONOUT has been initialized
    if (defined $CATCONOUT) {
      print $CATCONOUT $LogMsg;
    }
  }

  #
  # set_log_file_base_path
  #
  # Parameters:
  #   - log directory, as specified by the user
  #   - base for log file names
  #   - an indicator of whether to produce debugging info
  #
  # Returns
  #   - 0  ERROR 
  #   - Log Base File Name
  #
  sub set_log_file_base_path ($$$) {
    my ($LogDir, $LogBase, $DebugOn) = @_;

    my $RetLogFilePathBase = 0;

    # NOTE:
    #   log directory and base for log file names is handled first since we 
    #    need them to open the catcon output file

    # if directory for log file(s) has been specified, verify that it exists 
    # and is writable
    if (!valid_log_dir($LogDir)) {
      log_msg <<msg;
set_log_file_base_path: Unexpected error returned by valid_log_dir
msg
      return 0;
    }

    # determine base for log file names
    # the same base will be used for spool file names if $SpoolOn is true
    $RetLogFilePathBase = 
      get_log_file_base_path($LogDir, $LogBase, $DebugOn);

    # open $CATCONOUT.  If opened successfully, all output generated by catcon 
    # should be written to $CATCONOUT to ensure that we can find it after 
    # running an lrg involving catcon on the farm or through some other Perl 
    # script
    if (!open($CATCONOUT, ">>", $RetLogFilePathBase."_catcon_".$$.".lst")) {
      print STDERR "set_log_file_base_path: unable to open ".$RetLogFilePathBase."_catcon_".$$.".lst as CATCONOUT\n";
      return 0;
    }

    # last message which will be seen only on the console
    print STDERR "catcon: ALL catcon-related output will be written to [".$RetLogFilePathBase."_catcon_".$$.".lst]\n";

    # make $CATCONOUT "hot" so diagnostic and error message output does not get
    # buffered
    select((select($CATCONOUT), $|=1)[0]);

    log_msg("catcon: See [${RetLogFilePathBase}*.log] files for output generated by scripts\n");
    log_msg("catcon: See [${RetLogFilePathBase}_*.lst] files for spool files, if any\n");

    my $ts = TimeStamp();

    # do not dump the banner to STDERR
    print $CATCONOUT <<msg;
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

$VERSION
\tcatconInit: start logging catcon output at $ts

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

msg

    return $RetLogFilePathBase;

  }


  #
  # catconInit - initialize and validate catcon static vars and start 
  #              SQL*Plus processes
  #
  # Parameters:
  #   - user name, optionally with password, supplied by the caller; may be 
  #     undefined (default / AS SYSDBA)
  #   - internal user name, optionally with password; may be 
  #     undefined (default / AS SYSDBA)
  #   - directory containing sqlplus script(s) to be run; may be undefined
  #   - directory to use for spool and log files; may be undefined
  #   - base for spool log file name; must be specified
  #
  #   - container(s) (names separated by one or more spaces) in which to run 
  #     sqlplus script(s) and SQL statement(s) (i.e. skip containers not 
  #     referenced in this list)
  #   - container(s) (names separated by one or more spaces) in which NOT to 
  #     run sqlplus script(s) and SQL statement(s) (i.e. skip containers
  #     referenced in this list)
  #     NOTE: the above 2 args are mutually exclusive; if neither is defined, 
  #       sqlplus script(s) and SQL statement(s) will be run in the 
  #       non-Consolidated Database or all Containers of a Consolidated 
  #       Database
  #
  #   - number of processes to be used to run sqlplus script(s) and SQL 
  #     statement(s);  may be undefined (default value will be computed based 
  #     on host's hardware characteristics, number of concurrent sessions, 
  #     and whether the subroutine is getting invoked interactively)
  #   - external degree of parallelism, i.e. if more than one script using 
  #     catcon will be invoked simultaneously on a given host, this parameter 
  #     should contain a number of such simultaneous invocations; 
  #     may be undefined;
  #     will be used to determine number of processes to start;
  #     this parameter MUST be undefined or set to 0 if the preceding 
  #     parameter (number of processes) is non-zero
  #   - indicator of whether echo should be set ON while running script(s) 
  #     and SQL statement(s); defaults to FALSE
  #   - indicator of whether output of running scripts should be spooled in 
  #     files stored in the same directory as that used for log files and 
  #     whose name will be constructed as follows:
  #       <log-file-name-base>_<script_name_without_extension>_[<container_name_if_any>].<default_extension>
  #   - reference to a string used to introduce an argument to SQL*Plus 
  #     scripts which will be supplied using clear text (i.e. not require 
  #     user to input them at run-time)
  #     NOTE: variable referenced by this parameter will default to "--p" 
  #           if not supplied by the caller
  #   - reference to a string used to introduce an argument (such as a 
  #     password) to SQL*Plus scripts which will require user to input them 
  #     at run-time
  #     NOTE: variable referenced by this parameter will default to "--P" 
  #           if not supplied by the caller
  #   - indicator of whether ERRORLOGGING should be set ON while running 
  #     script(s) and SQL statement(s); defaults to FALSE; if value is ON, 
  #     default error logging table (SPERRORLOG) will be used, otherwise 
  #     specified value will be treated as the name of the error logging 
  #     table which should have been pre-created in every Container
  #   - indicator of whether the caller has instructed us to NOT set 
  #     Errorlogging Identifier
  #   - reference to an array of statements which will be executed in every 
  #     process before it is asked to run the first script; if operating on a 
  #     CDB, statements will be executed against the Root; statements may 
  #     contain 0 or more instances of string %proc which will be replaced 
  #     with process number
  #   - reference to an array of statements which will be executed in every 
  #     process after it finishes runing the last script; if operating on a 
  #     CDB, statements will be executed against the Root; statements may 
  #     contain 0 or more instances of string %proc which will be replaced 
  #     with process number
  #   - indicator of the mode in which PDB$SEED needs to be open before 
  #     running scripts (see description of catcon_SeedPdbMode for details)
  #   - indicator of whether to produce debugging messages; defaults to FALSE
  #   - an indicator of whether we are being called from a GUI tool which on 
  #     Windows means that the passwords and hidden parameters need not be 
  #     hidden
  #   - an indicator of whether it is called from cdb_sqlexec.pl. Will use
  #     this parameter to set $catcon_UserScript
  #


  sub catconInit ($$$$$$$$$$$\$\$$$\@\@$$$$) {
    my ($User, $InternalUser, $SrcDir, $LogDir, $LogBase, 
        $ConNamesIncl, $ConNamesExcl, 
        $NumProcesses, $ExtParallelDegree,
        $EchoOn, $SpoolOn, $RegularArgDelim_ref, $SecretArgDelim_ref, 
        $ErrLogging, $NoErrLogIdent, $PerProcInitStmts, 
        $PerProcEndStmts, $SeedMode, $DebugOn, $GUI, $UserScript) = @_;

    if ($catcon_InitDone) {
      # if catconInit has already been called, $CATCONOUT must be available
      # send it to STDERR as well to increase likelihood of it being noticed
      log_msg <<msg;
catconInit: script execution state has already been initialized; call 
    catconWrapUp before invoking catconInit
msg
      return 1;
    }

    # save DebugOn indicator in a variable that will persist across calls
    $catcon_DebugOn = $DebugOn;

    # make STDERR "hot" so diagnostic and error message output does not get 
    # buffered
    select((select(STDERR), $|=1)[0]);

    # set Log File Path Base
    $catcon_LogFilePathBase = 
        set_log_file_base_path ($LogDir,$LogBase, $catcon_DebugOn);

    # check to make sure we have a Log File Path Base
    if (!$catcon_LogFilePathBase) {
      log_msg <<msg;
catconInit: Unexpected error returned by set_log_file_base_path
msg
      return 1;
    }
    
    # Set Log directory
    $catcon_LogDir = $LogDir;

    my $ts = TimeStamp();

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconInit: start initializing catcon\n");
    }

    if ($catcon_DebugOn) {
      log_msg("catconInit: base for log and spool file names = ".
              "$catcon_LogFilePathBase\n");
    }

    if ($catcon_DebugOn) {

      log_msg <<catconInit_DEBUG;
  running catconInit(User              = $User, 
                     InternalUser      = $InternalUser,
                     SrcDir            = $SrcDir, 
                     LogDir            = $LogDir, 
                     LogBase           = $LogBase,
                     ConNamesIncl      = $ConNamesIncl, 
                     ConNamesExcl      = $ConNamesExcl, 
                     NumProcesses      = $NumProcesses,
                     ExtParallelDegree = $ExtParallelDegree,
                     EchoOn            = $EchoOn, 
                     SpoolOn           = $SpoolOn,
                     RegularArgDelim   = $$RegularArgDelim_ref,
                     SecretArgDelim    = $$SecretArgDelim_ref,
                     ErrLogging        = $ErrLogging,
                     NoErrLogIdent     = $NoErrLogIdent,
                     SeedMode          = $SeedMode,
                     Debug             = $DebugOn,
                     GUI               = $GUI,
                     UserScript        = $UserScript)\t\t($ts)

catconInit_DEBUG
    }

    # remember whether catconInit is called by cdb_sqlexec.pl
    $catcon_UserScript = $UserScript;

    # will contain status of the instance (v$instance.status)
    my $instanceStatus;

    # will contain an indicator of whether a DB is a CDB (v$database.cdb)
    my $IsCDB;

    # base for log file names must be supplied
    if (!$LogBase) {
      log_msg("catconInit: Base for log file names must be supplied");
      return 1;
    }

    # if caller indicated that we are to start a negative number of processes,
    # it is meaningless, so replace it with 0
    if ($NumProcesses < 0) {
      $NumProcesses = 0;
    }

    if ($NumProcesses && $ExtParallelDegree) {
      log_msg <<msg;
catconInit: you may not specify both the number of processes ($NumProcesses) 
    and the external degree of parallelism ($ExtParallelDegree)
msg
      return 1;
    }

    # save EchoOn indicator in a variable that will persist across calls
    $catcon_EchoOn = $EchoOn;

    # save SpoolOn indicator in a variable that will persist across calls
    $catcon_SpoolOn = $SpoolOn;

    # initialize indicators used to determine how to resolve conflicts 
    # between regular and secret argument delimiters
    $catcon_PickRegularArg = 1; 
    $catcon_PickSecretArg = 2;  

    # 
    # set up signal handler in case SQL process crashes
    # before it completes its work
    #
    # Bug 18488530: add a handler for SIGINT
    # Bug 18548396: add handlkers for SIGTERM and SIGQUIT
    # Bug 22887047: we will no longer try to handle SIGCHLD; instead we will 
    #   ignore it and deal with any dead child processes when we come across 
    #   them in the course of trying to pick a next process to do perform some 
    #   task (next_proc())
    #
    # save original handlers for SIGCHLD, SIGINT, SIGTERM, and SIGQUIT before 
    # resetting them
    $catcon_SaveSig{CHLD} = $SIG{CHLD} if (exists $SIG{CHLD});

    $catcon_SaveSig{INT}  = $SIG{INT}  if (exists $SIG{INT});

    $catcon_SaveSig{TERM} = $SIG{TERM} if (exists $SIG{TERM});

    $catcon_SaveSig{QUIT} = $SIG{QUIT} if (exists $SIG{QUIT});

    $SIG{CHLD} = 'IGNORE';
    $SIG{INT} = \&catcon_HandleSigINT;
    $SIG{TERM} = \&catcon_HandleSigTERM;
    $SIG{QUIT} = \&catcon_HandleSigQUIT;

    # figure out if we are running under Windows and set $catcon_DoneCmd
    # accordingly
    os_dependent_stuff($catcon_Windows, $catcon_DoneCmd);

    if ($catcon_DebugOn) {
      log_msg("catconInit: running on $OSNAME; DoneCmd = $catcon_DoneCmd\n");
    }

    #
    # unset TWO_TASK or LOCAL if it happens to be set to ensure that CONNECT
    # which does not specify a service results in a connection to the Root
    #

    if($catcon_Windows) {
      if ($ENV{LOCAL}) {
         if ($catcon_DebugOn) {
            log_msg("catconInit: LOCAL was set to $ENV{LOCAL} - unsetting ".
                    "it\n");
         }

         delete $ENV{LOCAL};
      } else {
        if ($catcon_DebugOn) {
            log_msg("catconInit: LOCAL was not set, so there is not need ".
                    "to unset it\n");
        }
      }
    }

    if ($ENV{TWO_TASK}) {
      if ($catcon_DebugOn) {
        log_msg("catconInit: TWO_TASK was set to $ENV{TWO_TASK} - unsetting ".
                "it\n");
      }

      delete $ENV{TWO_TASK};
    } else {
      if ($catcon_DebugOn) {
        log_msg("catconInit: TWO_TASK was not set, so there is no need to ".
                "unset it\n");
      }
    }

    if (!$$RegularArgDelim_ref) {
      $$RegularArgDelim_ref = '--p';

      if ($catcon_DebugOn) {
        log_msg("catconInit: regular argument delimiter was not specified - ".
                "defaulting to $$RegularArgDelim_ref\n");
      }
    }

    if (!$$SecretArgDelim_ref) {
      $$SecretArgDelim_ref = '--P';

      if ($catcon_DebugOn) {
        log_msg("catconInit: secret argument delimiter was not specified - ".
                "defaulting to $$SecretArgDelim_ref\n");
      }
    }

    # save strings used to introduce arguments to SQL*Plus scripts
    if ($$RegularArgDelim_ref ne "--p" || $$SecretArgDelim_ref ne "--P") {

      if ($catcon_DebugOn) {
        log_msg("catconInit: either regular ($$RegularArgDelim_ref) or secret ($$SecretArgDelim_ref) argument delimters were explicitly specified\n");
      }

      # if both are specified, they must not be the same
      if ($$RegularArgDelim_ref eq $$SecretArgDelim_ref) {
        log_msg <<msg;
catconInit: string introducing regular script argument ($$RegularArgDelim_ref) 
    may not be \nthe same as a string introducing a secret script 
    argument ($$SecretArgDelim_ref)
msg
        return 1;
      }

      # if one of the strings is a prefix of the other, remember which one to 
      # pick if an argument supplied by the user may be either
      if ($$RegularArgDelim_ref =~ /^$$SecretArgDelim_ref/) {
        $catcon_ArgToPick = $catcon_PickRegularArg;

        if ($catcon_DebugOn) {
          log_msg <<msg;
catconInit: secret argument delimiter ($$SecretArgDelim_ref) is a prefix of 
    regular argument delimiter ($$RegularArgDelim_ref); if in doubt, 
    will pick regular
msg
        }
      } elsif ($$SecretArgDelim_ref =~ /^$$RegularArgDelim_ref/) {
        $catcon_ArgToPick = $catcon_PickSecretArg;

        if ($catcon_DebugOn) {
          log_msg <<msg;
catconInit: regular argument delimiter ($$RegularArgDelim_ref) is a prefix of 
    secret argument delimiter ($$SecretArgDelim_ref); if in doubt, will 
    pick secret
msg
        }
      }
    }

    # argument delimiters may not start with @
    if ($$RegularArgDelim_ref =~ /^@/) {
      log_msg("catconInit: regular argument delimiter ($$RegularArgDelim_ref) begins with @ which is not allowed\n");
      return 1;
    }

    if ($$SecretArgDelim_ref =~ /^@/) {
      log_msg("catconInit: secret argument delimiter ($$SecretArgDelim_ref) begins with @ which is not allowed\n");
      return 1;
    }

    $catcon_RegularArgDelim = $$RegularArgDelim_ref;
    $catcon_SecretArgDelim = $$SecretArgDelim_ref;

    # remember whether ERRORLOGGING should be set 
    $catcon_ErrLogging = $ErrLogging;

    # remember whether the caller has instructed us to NOT issue 
    # SET ERRORLOGGING ON IDENTIFIER ...
    $catcon_NoErrLogIdent = $NoErrLogIdent;

    # saves references to arrays of statements which should be executed 
    # before using sending a script to a process for the first time and 
    # after a process finishes executing the last script assigned to it
    # in the course of running catconExec()
    $catcon_PerProcInitStmts = $PerProcInitStmts;
    $catcon_PerProcEndStmts = $PerProcEndStmts;

    # process EZConnect strings, if any

    my @ezConnStrings;

    if ($catcon_EZConnect || $ENV{$EZCONNECT_ENV_TAG}) {
      # store a list of EZConnect strings in @ezConnStrings for use when 
      # determining which instance will be picked for running scriptrs
      if ($catcon_EZConnect) {
        # use a list of EZConnect strings supplied by the caller
        @ezConnStrings = split(/  */,$catcon_EZConnect);
      
        # $catcon_EZConnect should have contained at least 1 string
        if ($#ezConnStrings < 0) {
          log_msg("catconInit: EZConnect string list must contain at least 1 string\n");
          return 1;
        }

        if ($catcon_DebugOn) {
          log_msg("catconInit: EZConnect strings supplied by the user:\n\t");
          log_msg(join("\n\t", @ezConnStrings)."\n");
        }
      } else {
        @ezConnStrings = split(/  */, $ENV{$EZCONNECT_ENV_TAG});
          
        # $ENV{$EZCONNECT_ENV_TAG} should have contained at least 1 string
        if ($#ezConnStrings < 0) {
          log_msg("catconInit: $EZCONNECT_ENV_TAG environment variable must contain at least 1 string\n");
          return 1;
        }

        if ($catcon_DebugOn) {
          log_msg("catconInit: EZConnect strings found in $EZCONNECT_ENV_TAG env var");
          log_msg(join("\n\t", @ezConnStrings));
        }
      }

      if ($catcon_DebugOn || $catcon_Verbose) {
        log_msg("catconInit: finished processing EZConnect strings\n");
      }
    } else {
      # having $ezConnStrings[0] eq "" serves as an indicator that we will
      # use the default instance
      push @ezConnStrings, "";

      if ($catcon_DebugOn) {
        log_msg("catconInit: no EZConnect strings supplied - using default instance\n");
      }
    }

    # construct a user connect string and store it as well as the password 
    # (so catconUserPass() can return it), if any, in variables which will 
    # persist across calls.  
    #
    # NOTE: If ezConnStrings does not represent the default instance, 
    #       $catcon_UserConnectString will contain a placeholder (@%s) for 
    #       the EZConnect string corresponding to the instance which we will 
    #       eventually pick.
    ($catcon_UserConnectString[0], $catcon_UserConnectString[1], 
     $catcon_UserPass) = 
      get_connect_string($User, !($catcon_Windows && $GUI), 
                         $ENV{$USER_PASSWD_ENV_TAG}, 
                         $ezConnStrings[0] ne "");

    if (!$catcon_UserConnectString[0]) {
      # NOTE: $catcon_UserConnectString may be set to undef
      #       if the caller has not supplied a value 
      #       for $USER (which would lead get_connect_string() to return 
      #       "/ AS SYSDBA" as the connect string while specifying instances 
      #       on which scripts are to be run (by means of passing one or more 
      #       EZConnect strings)
      log_msg("catconInit: Empty user connect string returned by ".
              "get_connect_string\n");
      return 1;
    }

    if ($catcon_UserPass && length($catcon_UserPass) < 1) {
      # unless user will be connecting using "/ as sysdba" (in which case 
      # $catcon_UserPass would end up being undefined), password consisting 
      # of at least 1 character must have been specified
      log_msg("catconInit: Empty user password was specified\n");
      return 1;
    }

    if ($catcon_DebugOn) {
      log_msg("catconInit: User Connect String = ".
              "$catcon_UserConnectString[1]\n");

      # bug 18591655: if user password was not supplied, display 
      #   appropriate message
      log_msg("catconInit: User password ".
              ($catcon_UserPass ? "specified" : "not specified")."\n");
    }

    # Bug 18020158: generating internal connect string is complicated by the 
    # fact that catctl does not accept it as a parameter.  There are two 
    # distinct cases to consider:
    # - the caller supplied <internal user> and possibly <internal password> 
    #   inside $InternalUser:
    #   - if both internal user and password were supplied, use them
    #   - otherwise, if <internal password> was not supplied, we will try to 
    #     obtain it from $ENV{$INTERNAL_PASSWD_ENV_TAG}, and failing that, 
    #     prompt the user to enter it
    # - otherwise:
    #   - recycle user connect string as the internal connect string
    if ($InternalUser) {
      my $InternalPass; # used to accept a value returned by get_connect_string

      # NOTE: If ezConnStrings does not represent the default instance, 
      #       $catcon_InternalConnectString will contain a placeholder (@%s) 
      #       for the EZConnect string corresponding to the instance which 
      #       we will eventually pick.
      ($catcon_InternalConnectString[0], $catcon_InternalConnectString[1],
       $InternalPass) = 
        get_connect_string($InternalUser, !($catcon_Windows && $GUI), 
                           $ENV{$INTERNAL_PASSWD_ENV_TAG}, 
                           $ezConnStrings[0] ne "");

      if (!$catcon_InternalConnectString[0]) {
        log_msg("catconInit: Empty internal connect string returned by get_connect_string\n");
        return 1;
      }

      if ($InternalPass && length($InternalPass) < 1) {
        # unless internal user will be connecting using "/ as sysdba" (in 
        # which case $InternalPass would end up being undefined), password 
        # consisting of at least 1 character must have been specified
        log_msg("catconInit: Empty internal password was specified\n");
        return 1;
      }

      if ($catcon_DebugOn) {
        log_msg("catconInit: Internal Connect String = $catcon_InternalConnectString[1]\n");

        # bug 18591655: if internal password was not supplied, display 
        #   appropriate message
        log_msg("catconInit: Internal password ".($InternalPass ? "specified" : "not specified")."\n");
      }
    } else {
      @catcon_InternalConnectString = @catcon_UserConnectString;

      if ($catcon_DebugOn) {
        log_msg("catconInit: setting Internal Connect String to User Connect String\n");
      }
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconInit: finished constructing connect strings\n");
    }

    if (!valid_src_dir($catcon_SrcDir = $SrcDir)) {
      log_msg("catconInit: Unexpected error returned by valid_src_dir\n");
      return 1;
    }

    if ($catcon_DebugOn) {
      if ($catcon_SrcDir) {
        log_msg("catconInit: source file directory = $catcon_SrcDir\n");
      } else {
        log_msg("catconInit: no source file directory was specified\n");
      }
    }

    # Bug 17810688: make sure sqlplus is in $PATH
    if (!find_in_path("sqlplus", $catcon_Windows)) {
      log_msg("catconInit: sqlplus not in PATH.\n");
      return 1;
    }

    # if the caller specified mode in which PDB$SEED should be opened, 
    # validate it
    if ($SeedMode != CATCON_PDB_MODE_NA && !valid_pdb_mode($SeedMode)) {
      log_msg("catconInit: Unexpected value ($SeedMode) for SeedMode\n");
      return 1;
    }

    # finalize modes in which PDB$SEED and all other PDBs that we will be 
    # operating upon should be opened.
    #
    # NOTE: after the next 2 statements are executed, $catcon_SeedPdbMode 
    #       and $catcon_AllPdbMode are guaranteed to NOT be set to 
    #       CATCON_PDB_MODE_NA
    $catcon_SeedPdbMode = seed_pdb_mode_to_use($SeedMode, $catcon_AllPdbMode);

    $catcon_AllPdbMode = all_pdb_mode_to_use($catcon_AllPdbMode);

    # For each EZConnect String (or "" for default instance) stored in 
    # @ezConnStrings, connect to the instances represented by the string and 
    # - verify that the database is open on that instance, 
    # - if processing the first EZConnect string 
    #   - determine whether the database is a CDB 
    # - if @ezConnStrings contains more than one string, 
    #   - verify that the current EZConnect string takes us to the same DB 
    #     as the preceding one
    # - if the EZConnect string takes us to a CDB, 
    #   - verify that it takes us to the CDB's Root
    #   - if we are yet to determine which Instance to use
    #     - determine which Containers are open on this Instance
    #     - if there is only one EZConnect string (which may or may not  
    #       represent the default instance)
    #       - determine the set of Containers against which scripts will be 
    #         run, honoring the caller's directive regarding reporting of 
    #         errors due to some PDBs not being accessible
    #       - remember that this is the instance which we will be using
    #     - else
    #       - if processing the first EZConnect string,
    #         - determine the set of Containers against which scripts will be 
    #           run WITHOUT taking into consideration whether these Containers 
    #           are open on the instance corresponding to this EZConnect string
    #  
    #           the reason we are choosing to ignore open mode of Containers on
    #           this Instance is because we do not want to report an error if 
    #           some Container is not open on this instance - all desired 
    #           Containers may very well be open on some other instance - we 
    #           just need a list of Containers after taking into account the 
    #           inclusive or exclusive Container list (if any) + we DO want to 
    #           report an error if either of these lists included an unknown 
    #           Container and the caller has NOT instructed us to ignore such 
    #           Containers
    #       - determine if ALL Containers against which scripts need to be run 
    #         are open on this Instance
    #       - if all desired Containers are open on this Instance
    #         - remember that this is the instance which we will be using
    #     - if we finally determined the instance which will be using, 
    #       determine how many processes may be started on it
    # - else (i.e. not connecting to a CDB)
    #   - determine the number of processes that can be started on this 
    #     instance, and if it is greater than that for all preceding 
    #     instances (if any), remember that this is the instance we will 
    #     want to use (unless we find an instance on which an even greater 
    #     number of processes can be started)
    # - endif

    # set to 0 in case we were called to run scripts against a non-CDB and need
    # to look for an instance that can support the greatest number of processes
    $catcon_NumProcesses = 0;

    # If considering more than one instance, $dbid will be used to verify 
    # that all instances are connected to the same database
    my $dbid;

    # prefix of "done" file name
    #
    # Bug 18011217: append _catcon_$$ (process id) to $catcon_LogFilePathBase
    # to avoid conflicts with other catcon processes running on the same host
    my $doneFileNamePrefix = 
      done_file_name_prefix($catcon_LogFilePathBase, $$);

    # EZConnect string corresponding to the instance which we picked to run 
    # scripts. If still not set when we exit the loop, report an error
    my $ezConnToUse;

    for (my $currInst = 0; $currInst <= $#ezConnStrings; $currInst++) {
      my $EZConnect = $ezConnStrings[$currInst];

      if ($catcon_DebugOn) {
        log_msg("catconInit: considering EZConnect string ".$EZConnect."\n");
      }

      # construct a connect string which will be used to obtain information 
      # about the current instance, the database which is running on it, etc.
      my @connectString;

      # Bug 21202842: construct connect string with/without redacted password
      $connectString[0] = 
        build_connect_string($catcon_InternalConnectString[0], $EZConnect, 0);
      $connectString[1] = 
        build_connect_string($catcon_InternalConnectString[1], $EZConnect,
                             $catcon_DebugOn);

      # build_connect_string may return undef, in which case we will return 1 
      # (to indicate that an error was encountered)
      if (!$connectString[0]) {
        log_msg("catconInit: unexpected error encountered in ".
                "build_connect_string\n");
        return 1;
      }

      # (13704981) verify that the instance is open

      if ($catcon_DebugOn) {
        log_msg("catconInit: call get_instance_status_and_name\n");
      }

      # status and name of the instance (v$instance.status and 
      # v$instance.instance_name)
      my ($instanceStatus, $instanceName) = 
        get_instance_status_and_name(@connectString, $catcon_DoneCmd, 
                                     $doneFileNamePrefix, $catcon_DebugOn);

      if (!$instanceStatus) {
        log_msg("catconInit: unexpected error in ".
                "get_instance_status_and_name\n");
        return 1;
      } elsif ($instanceStatus !~ /^OPEN/) {
        log_msg("catconInit: database is not open");
        if ($EZConnect eq "") {
          # default instance
          log_msg(" on the default instance (".$instanceName.")\n");
        } else {
          log_msg(" on instance ".$instanceName." with EZConnect string = ".
                  $EZConnect."\n");
        }
        return 1;
      }

      if ($catcon_DebugOn) {
        log_msg("catconInit: instance $instanceName (EZConnect = ".
                $EZConnect.") has status $instanceStatus\n");
      }

      # determine whether a DB is a CDB.  Do it only once (before obtaining 
      # database' DBID)
      if (!$dbid) {
        if ($catcon_DebugOn) {
          log_msg("catconInit: processing first instance - call ".
                  "get_CDB_indicator\n");
        }

        $IsCDB = 
          get_CDB_indicator(@connectString, $catcon_DoneCmd, 
                            $doneFileNamePrefix, $catcon_DebugOn);

        if (!(defined $IsCDB)) {
          log_msg("catconInit: unexpected error in get_CDB_indicator\n");
          return 1;
        }

        if ($catcon_DebugOn) {
          log_msg("catconInit: database is ".
                  (($IsCDB eq 'NO') ? "non-" : "")."Consolidated\n");
        }
      }

      if ($catcon_DebugOn) {
        log_msg("catconInit: call get_dbid\n");
      }

      # obtain DBID of the database to which the current instance is 
      # connected and compare id to DBID of preceding instances, if any
      my $currDBID = 
        get_dbid(@connectString, $catcon_DoneCmd, 
                 $doneFileNamePrefix, $catcon_DebugOn);

      if (!(defined $currDBID)) {
        # DBID could not be obtained
        log_msg("catconInit: unexpected error in get_dbid\n");
        return 1;
      }

      if (!$dbid) {
        # first EZ Connect string (or "", representing the default instance)
        $dbid = $currDBID;

        if ($catcon_DebugOn) {
          log_msg("catconInit: DBID = $dbid\n");
        }
      } elsif ($dbid ne $currDBID) {
        # at least 2 EZConnect strings were specified and, not all of them 
        # correspond to the same database
        log_msg <<msg;
catconInit: instance $instanceName (EZConnect string = $EZConnect)
    is connected to a database with DBID = $currDBID which is different
    from DBID ($dbid) of the database to which preceding instances are 
    connected
msg
        return 1;
      }

      # CDB-specific stuff (see above)
      if ($IsCDB eq "YES") {
        if ($catcon_DebugOn || $catcon_Verbose) {
          log_msg("catconInit: start CDB-specific processing\n");
        }

        # make sure that the current EZConnect string will take us to the Root.
        if ($catcon_DebugOn) {
          log_msg("catconInit: call get_con_id to verify that the current ".
                  "EZConnect string will take us to the Root\n");
        }

        my $conId = get_con_id(@connectString, $catcon_DoneCmd, 
                               $doneFileNamePrefix, $catcon_DebugOn);

        if (!(defined $conId)) {
          # con_id could not be obtained
          log_msg("catconInit: unexpected error in get_con_id\n");
          return 1;
        }

        if ($conId != 1) {
          # EZConnect string is not taking us to a CDB's Root
          log_msg <<msg;
catconInit: Instance $instanceName (EZConnect string = $EZConnect)
    points to a Container with CON_ID of $conId instead of the Root
msg
          return 1;
        }

        # the rest of this block has to do with finding an Instance on which 
        # all desired Containers are open.  If such Instance has already been 
        # located, process the next EZConnect string (if any)
        if (defined $ezConnToUse) {
          if ($catcon_DebugOn) {
            log_msg <<msg;
catconInit: skip to the next EZConnect string since we have already determined 
    which EZConnect string to use
msg
          }

          next;
        }

        # indicators of which Containers are open on this instance
        my %isConOpen;

        if (!@catcon_AllContainers) {

          # we are processing the first EZConnect string, so obtain names of 
          # ALL Containers in the CDB along with indicators of which of these 
          # Containers are open on this instance
          if ($catcon_DebugOn) {
            log_msg("catconInit: call get_con_info to obtain container ".
                    "names and open indicators\n");
          }

          if (get_con_info(@connectString, $catcon_DoneCmd, 
                           $doneFileNamePrefix, \@catcon_AllContainers, 
                           \%isConOpen, 0, $catcon_DebugOn)) {
            log_msg("catconInit: unexpected error in get_con_info(1)\n");
            return 1;
          }

          # save name of the Root (it will always be in the 0-th element of 
          # @catcon_AllContainers because its contents are sorted by 
          # v$containers.con_id)
          $catcon_Root = $catcon_AllContainers[0];

          if ($catcon_UserScript) {
            # if running user- (rather than Oracle-) supplied scripts, discard 
            # the first 2 elements of @catcon_AllContainers (CDB$ROOT and 
            # PDB$SEED) and corresponding elements of %isConOpen
            #
            # if the CDB does not contain any other Containers, report an error
            # because there are no PDBs against which to run user scripts
            if ($#catcon_AllContainers < 2) {
              log_msg("catconInit: cannot run user scripts against a CDB ".
                      "which contains no user PDBs\n");
              return 1;
            }

            delete $isConOpen{shift @catcon_AllContainers};
            delete $isConOpen{shift @catcon_AllContainers};

            if ($catcon_DebugOn) {
              log_msg <<msg;
catconInit: running customer scripts, so purged the first 2 entries from 
    catcon_AllContainers and deleted corresponding entries from isConOpen
msg
            }
          }

          # - if a list of names of Containers in which to run (or not to run) 
          #   scripts has been specified, 
          #   - make sure both of them have NOT been specified (since they are 
          #     mutually exclusive) and
          # - if there is only one EZConnect string to consider
          #   - if an inclusive or exclusive Container list was specified,
          #     - verify that the desired Containers actually exist and are 
          #       open on this instance (taking into consideration caller's 
          #       request (if any) to ignore non-existent Containers mentioned 
          #       in the inclusive/exclusive list and Containers which are not 
          #       open on this instance)
          #   - else
          #     - verify that all Containers are open on this instance (taking 
          #       into consideration caller's request (if any) to ignore 
          #       Containers which are not open on this instance)
          if ($ConNamesIncl && $ConNamesExcl) {
            log_msg <<msg;
catconInit: both a list of Containers in which to run scripts and a list of 
    Containers in which NOT to run scripts were specified, which is disallowed
msg
            return 1;
          } elsif ($catcon_FedRoot) {
            # if Federation Root name was supplied, verify that neither 
            #   inclusive nor exclusive list of Container name has been 
            #   supplied
            # - confirm that the specified Container exists and is, indeed, 
            #   a Federation Root
            # - construct a string consisting of the name of the specified 
            #   Federation Root and all Federation PDBs belonging to it and 
            #   set $ConNamesIncl to that string (as if it were specified by 
            #   the caller, so we can take advantage of existing code handling 
            #   caller-supplied inclusive Container name list)
            if ($ConNamesIncl || $ConNamesExcl) {
              log_msg <<msg;
catconInit: Federation Root name and either a list of Containers in which to 
    run scripts or a list of Containers in which NOT to run scripts were 
    specified, which is disallowed
msg
              return 1;
            }

            if ($catcon_DebugOn) {
              log_msg("catconInit: confirm that $catcon_FedRoot is an ".
                      "App Root\n");
            }
           
            for (my $curCon = 0; $curCon <= $#catcon_AllContainers; $curCon++) 
            {
              if ($catcon_AllContainers[$curCon] eq $catcon_FedRoot) {
                # specified Container appears to exist; next we will 
                # - verify that the specified Container name refers to a 
                #   Federation Root and obtain its CON ID
                # - construct $ConNamesIncl by appending to the name of the 
                #   Federation Root names of all Federation PDBs 
                #   belonging to this Federation Root (so we can take 
                #   advantage of existing code handling caller-supplied 
                #   inclusive Container name list)
                if ($catcon_DebugOn) {
                  log_msg <<msg;
catconInit: Container specified as a Federation Root exists
    verify that it is a Federation Root
msg
                }
                
                my $IsFedRoot;
                my $FedRootConId;

                if (get_fed_root_info(@connectString, $catcon_DoneCmd,
                                      $doneFileNamePrefix, 
                                      $catcon_FedRoot, $IsFedRoot, 
                                      $FedRootConId, $catcon_DebugOn)) {
                  log_msg("catconInit: unexpected error in ".
                          "get_fed_root_info\n");
                  return 1;
                }

                if (!$IsFedRoot) {
                  log_msg <<msg;
catconInit: Purported Federation Root $catcon_FedRoot has disappeared
msg
                  return 1;
                }

                if ($IsFedRoot ne "YES") {
                  log_msg <<msg;
catconInit: Purported Federation Root $catcon_FedRoot is not
msg
                  return 1;
                }

                if ($catcon_DebugOn) {
                  log_msg <<msg;
catconInit: Container $catcon_FedRoot is indeed a Federation Root;
    calling get_fed_pdb_names
msg
                }

                my @FedPdbNames;

                if (get_fed_pdb_names(@connectString, $catcon_DoneCmd,
                                      $doneFileNamePrefix, 
                                      $FedRootConId, @FedPdbNames,
                                      $catcon_DebugOn)) {
                  log_msg("catconInit: unexpected error in ".
                          "get_fed_pdb_names\n");
                  return 1;
                }

                # place Federation Root name at the beginning of $ConNamesIncl
                $ConNamesIncl = $catcon_FedRoot;

                # append Federation PDB names to $ConNamesIncl
                for my $FedPdbName ( @FedPdbNames ) {
                  $ConNamesIncl .= " ".$FedPdbName;
                }

                if ($catcon_DebugOn) {
                  log_msg <<msg;
catconInit: ConNamesIncl reset to names of Containers comprising 
    Federation rooted in $catcon_FedRoot:
        $ConNamesIncl
msg
                }
              }
            }

            # if the specified Container was a Federation Root, $ConNamesIncl
            # should be set to a list consisting of the name of the 
            # Federation Root followed by the names of its Federation PDBs, 
            # but if that Container did not exist, then $ConNamesIncl will be 
            # still undefined
            if (!$ConNamesIncl) {
              log_msg <<msg;
catconInit: Purported Federation Root $catcon_FedRoot does not exist
msg
                  return 1;
            }
          }

          if ($#ezConnStrings == 0) {
            if ($catcon_DebugOn) {
              log_msg("catconInit: only one instance to choose from; ".
                      "checking if all desired \n\tContainers are open on ".
                      "it\n");
            }

            if ($ConNamesExcl) {
              if (!(@catcon_Containers = 
                    validate_con_names($ConNamesExcl, 1, @catcon_AllContainers,
                        $catcon_AllPdbMode == CATCON_PDB_MODE_UNCHANGED
                          ? \%isConOpen : undef, 
                        $catcon_IgnoreInaccessiblePDBs,
                        $catcon_IgnoreAllInaccessiblePDBs, 
                        $catcon_DebugOn))) {
                log_msg("catconInit: Unexpected error returned by ".
                        "validate_con_names for exclusive Container list\n");
                return 1;
              }
            } elsif ($ConNamesIncl) {
              if (!(@catcon_Containers = 
                    validate_con_names($ConNamesIncl, 0, @catcon_AllContainers,
                        $catcon_AllPdbMode == CATCON_PDB_MODE_UNCHANGED
                          ? \%isConOpen : undef, 
                        $catcon_IgnoreInaccessiblePDBs,
                        $catcon_IgnoreAllInaccessiblePDBs, 
                        $catcon_DebugOn))) {
                log_msg("catconInit: Unexpected error returned by ".
                        "validate_con_names for inclusive Container list\n");
                return 1;
              } 
            } else {
              # we know that all desired Containers (i.e. Containers in 
              # @catcon_AllContainers) exist, so we just need to check if 
              # they are open

              my $AllConStr = join(" ", @catcon_AllContainers);

              if (!(@catcon_Containers = 
                    validate_con_names($AllConStr, 0, @catcon_AllContainers, 
                        $catcon_AllPdbMode == CATCON_PDB_MODE_UNCHANGED
                          ? \%isConOpen : undef, 
                        $catcon_IgnoreInaccessiblePDBs,
                        $catcon_IgnoreAllInaccessiblePDBs, 
                        $catcon_DebugOn))) {
                log_msg("catconInit: Unexpected error returned by ".
                        "validate_con_names\n");
                return 1;
              } 
            }

            # there are no apparent obstacles to using this Instance. 
            # Remember that we found a satisfactory instance and copy a 
            # list of open Container indicators for future use
            $ezConnToUse = $EZConnect;
            %catcon_IsConOpen = %isConOpen;
            
            if ($catcon_DebugOn) {
              log_msg <<msg;
catconInit: instance $instanceName represented by the only EZConnect
    string has all desired Containers open on it
msg
            }
          }
        } else {
          # at least one EZConnect string has been processed, so we no longer
          # need to fetch names of all Containers.  We do, however, need to 
          # determine which of the Containers are open on this instance and 
          # populate %isConOpen accordingly

          if ($catcon_DebugOn) {
            log_msg("catconInit: Obtain open container indicators\n");
          }

          if (get_con_info(@connectString, $catcon_DoneCmd, 
                           $doneFileNamePrefix, undef, \%isConOpen, 
                           $catcon_UserScript, $catcon_DebugOn)) {
            log_msg("catconInit: unexpected error in get_con_info(2)\n");
            return 1;
          }
        }
        
        if ($catcon_DebugOn) {
          log_msg("catconInit: on instance ".$instanceName.
                  ", Container names and open indicators are {\n");

          for (my $curCon = 0; $curCon <= $#catcon_AllContainers; $curCon++) {
            log_msg("\t\t$catcon_AllContainers[$curCon] -- ".
                    "$isConOpen{$catcon_AllContainers[$curCon]}\n");
          }
          log_msg("catconInit: }\n");
        }

        if (defined $ezConnToUse) {
          # there is nothing more to do for this EZConnect string, and no 
          # more string to process (I could have called last earlier, but I 
          # did not want to duplicate code dumping Container names and modes)
          if ($catcon_DebugOn) {
            if ($ezConnToUse eq "") {
              log_msg <<msg;
catconInit: all desired Containers are open on the default instance 
    ($instanceName) - skip the remainder of EZConnect string processing
msg
            } else {
              log_msg <<msg;
catconInit: all desired Containers are open on instance ($instanceName)
    corresponding to the only eZConnect string - skip the remainder of 
    EZConnect string processing
msg
            }
          }

          last;
        }

        # if this is the first (of many) EZConnect strings, determine the list 
        # of desired Containers
        if ($currInst == 0) {
          if ($catcon_DebugOn) {
            log_msg <<msg;
catconInit: processing the first of many EZConnect strings - determine names 
    of desired Containers
msg
          }

          if ($ConNamesExcl) {
            if (!(@catcon_Containers = 
                  validate_con_names($ConNamesExcl, 1, 
                                     @catcon_AllContainers, undef,
                                     $catcon_IgnoreInaccessiblePDBs,
                                     $catcon_IgnoreAllInaccessiblePDBs, 
                                     $catcon_DebugOn))) {
              log_msg("catconInit: Unexpected error returned by ".
                      "validate_con_names for exclusive Container list\n");
              return 1;
            }
          } elsif ($ConNamesIncl) {
            if (!(@catcon_Containers = 
                  validate_con_names($ConNamesIncl, 0, 
                                     @catcon_AllContainers, undef,
                                     $catcon_IgnoreInaccessiblePDBs,
                                     $catcon_IgnoreAllInaccessiblePDBs, 
                                     $catcon_DebugOn))) {
              log_msg("catconInit: Unexpected error returned by ".
                      "validate_con_names for inclusive Container list\n");
              return 1;
            } 
          } else {
            # we know that all desired Containers (i.e. Containers in 
            # @catcon_AllContainers) exist, and we don't care whether they 
            # are open, so just copy @catcon__AllContainers into 
            # @catcon_Containers
            @catcon_Containers = @catcon_AllContainers;
          }

          if ($catcon_DebugOn) {
            log_msg("catconInit: scripts need to be run in the following Containers:\n\t");
            log_msg(join("\n\t", @catcon_Containers)."\n");
          }
        }

        if ($catcon_DebugOn) {
          log_msg("catconInit: check if desired Containers are open on this instance:\n");
        }

        # unless the caller indicated that all relevant PDBs must be opened 
        # prior to running scripts against them, determine if all desired 
        # Containers (whose names are stored in @catcon_Containers) are open 
        # on this Instance
        if ($catcon_AllPdbMode != CATCON_PDB_MODE_UNCHANGED) {
          $ezConnToUse = $EZConnect;
        } elsif ($ConNamesExcl || $ConNamesIncl) {
          # Containers whose names are stored in @catcon_Containers need to be 
          # open

          my $curCon;             #index into @catcon_Containers

          for ($curCon = 0; $curCon <= $#catcon_Containers; $curCon++) {
            if ($isConOpen{$catcon_Containers[$curCon]} eq "N") {
              # this Container is not open - no need to look further
              if ($catcon_DebugOn) {
                log_msg("\tContainer ".$catcon_Containers[$curCon].
                        " is not open - skip the rest\n");
              }
              last;
            } elsif ($catcon_DebugOn) {
              log_msg("\tContainer ".$catcon_Containers[$curCon]." is open\n");
            }
          }
          
          if ($curCon > $#catcon_Containers) {
            # all desired Containers are open on this Instance, so use it
            $ezConnToUse = $EZConnect;
          }
        } else {
          # ALL Containers need to be open, so we simply need to run down 
          # @isConOpen and hope not to find a single indicator containing "N"
          my $curCon;

          for ($curCon = 0; 
               $curCon <= $#catcon_AllContainers && 
               $isConOpen{$catcon_AllContainers[$curCon]} eq "Y"; 
               $curCon++) {
            if ($catcon_DebugOn) {
                log_msg("\tContainer ".$catcon_AllContainers[$curCon].
                        " is open\n");
            }
          }

          if ($curCon > $#catcon_AllContainers) {
            # all desired Containers are open on this Instance, so use it
            $ezConnToUse = $EZConnect;
          } elsif ($catcon_DebugOn) {
            log_msg("\tContainer ".$catcon_AllContainers[$curCon].
                    " is not open - skip the rest\n");
          }
        }
        
        if (defined $ezConnToUse) {
          # we found our instance - copy a list of open Container indicators 
          # for future use
          %catcon_IsConOpen = %isConOpen;

          if ($catcon_DebugOn) {
            log_msg <<msg;
catconInit: all desired Containers are open on instance ($instanceName) 
    which will be picked to run all scripts
msg
          }
        }

        if ($catcon_DebugOn || $catcon_Verbose) {
          log_msg("catconInit: end CDB-specific processing\n");
        }
      } else {
        if ($catcon_DebugOn || $catcon_Verbose) {
          log_msg("catconInit: start non-CDB-specific processing\n");
        }

        # will run scripts against a non-CDB

        # compute number of processes that can be started on this instance if 
        # this is the first instance or if the caller has not specified a 
        # number of processes to start (which may result in us discovering 
        # that this instance may support a larger number of processes than 
        # previously processed instances) 
        if ($currInst == 0 || $NumProcesses == 0) {
          if ($catcon_DebugOn) {
            log_msg("catconInit: call get_num_procs to determine number of ".
                    "processes that may be started on this Instance ".
                    "because\n");
            if ($currInst == 0) {
              log_msg("    this is the first instance being processed\n");
            } else {
              log_msg("    user did not specify a number of processes\n");
            }
          }

          # NOTE: in the non-CDB case, we will not ask get_num_procs to 
          #       determine how many processes can be allocated to various 
          #       instances (because in case of upgrade (the initial target 
          #       of changes to enable scripts to be run on multiple 
          #       instances), a non-CDB can only be open on one instance 
          #       and we will leave it open where it is)
          my $numProcs = 
            get_num_procs($NumProcesses, $ExtParallelDegree, \@connectString, 
                          $catcon_DoneCmd, $doneFileNamePrefix, 
                          0, undef, $catcon_DebugOn);

          if ($numProcs == -1) {
            log_msg("catconInit: unexpected error in get_num_procs\n");
            return 1;
          } elsif ($numProcs < 1) {
            log_msg("catconInit: invalid number of processes ($numProcs) returned by get_num_procs\n");
            return 1;
          } elsif ($catcon_DebugOn) {
            log_msg("catconInit: get_num_procs determined that $numProcs processes can be started on this instance\n");
          }
            
          if ($numProcs > $catcon_NumProcesses) {
            # this instance will support a greater number of processes than
            # preceding instance (if any) - remember the number of 
            # processes as well as the EZConnect string of the instance we 
            # will want to use
            $catcon_NumProcesses = $numProcs;
            $ezConnToUse = $EZConnect;
            
            if ($catcon_DebugOn) {
              log_msg("catconInit: remember this instance as supporting the greatest number of processes so far\n");
            }
          }
        }

        if ($catcon_DebugOn || $catcon_Verbose) {
          log_msg("catconInit: end non-CDB-specific processing\n");
        }
      }
    }

    # make sure we found a suitable Instance
    if (!(defined $ezConnToUse)) {
      # none of the instances had all of the desired Containers open
      log_msg("catconInit: multiple instances were specified, but none of them had all of the desired Containers open\n");
      return 2;
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconInit: finished examining instances which can be used to run scripts/SQL statements.\n");
    }

    # if EZConnect strings were specified, $catcon_InternalConnectString and 
    # $catcon_UserConnectString contain placeholders for the EZConnect string 
    # corresponding to the instance that we pick.  Now that we have 
    # settled on a specific instance, replace these placeholders with the 
    # actual EZConnect string
    if (@ezConnStrings) {
      # Bug 21202842: do not produce diagnostic output when generating real 
      #               connect strings to avoid displaying passwords
      $catcon_InternalConnectString[0] = 
        build_connect_string($catcon_InternalConnectString[0], $ezConnToUse, 
                             0);
      $catcon_UserConnectString[0] = 
        build_connect_string($catcon_UserConnectString[0], $ezConnToUse, 0);

      # Bug 21202842: pass the real debug setting since here we are using 
      #               connect strings with redacted password
      $catcon_InternalConnectString[1] = 
        build_connect_string($catcon_InternalConnectString[1], $ezConnToUse, 
                             $catcon_DebugOn);
      $catcon_UserConnectString[1] = 
        build_connect_string($catcon_UserConnectString[1], $ezConnToUse, 
                             $catcon_DebugOn);
      
      if ($catcon_DebugOn) {
        log_msg <<msg;
catconInit: finalized connect strings:\n\tInternal Connect String = $catcon_InternalConnectString[1]
User Connect String = $catcon_UserConnectString[1]
msg
      }
    }

    # if we haven't determined the number of processes which may be started 
    # on the chosen Instance (which would imply that we will be running 
    # against a CDB since for non-CDBs we select the instance to use based on 
    # the number of processes that can be started on it), do it now
    if (!$catcon_NumProcesses) {
      if ($catcon_DebugOn) {
        log_msg <<msg;
catconInit: call get_num_procs to determine number of processes that will
    be started
msg
      }

      # Bug 20193612: if 
      #   - connected to a CDB,
      #   - the caller told us to use all available instances, 
      #   - the caller has specified the mode in which PDBs should be open, and
      #   - a specific instance was not specified, 
      # we will ignore number of processes supplied by the caller and let 
      # get_num_procs determine number of processes that can be opened on 
      # all of CDB's instances
      my $getProcsOnAllInstances = 
        $IsCDB eq "YES" && 
        $catcon_AllInst && 
        $catcon_AllPdbMode != CATCON_PDB_MODE_UNCHANGED && 
        index($catcon_InternalConnectString[0], "@") == -1;

      $catcon_NumProcesses = 
        get_num_procs($NumProcesses, $ExtParallelDegree, 
                      \@catcon_InternalConnectString, $catcon_DoneCmd, 
                      $doneFileNamePrefix, $getProcsOnAllInstances, 
                      \%catcon_InstProcMap, $catcon_DebugOn);
      
      if ($catcon_NumProcesses == -1) {
        log_msg("catconInit: unexpected error in get_num_procs\n");
        return 1;
      } elsif ($catcon_NumProcesses < 1) {
        log_msg("catconInit: invalid number of processes ($catcon_NumProcesses) returned by get_num_procs\n");
        return 1;
      } elsif ($catcon_DebugOn) {
        log_msg("catconInit: get_num_procs determined that $catcon_NumProcesses processes can be started on this instance\n");
      }
    }

    # generate "kill all sessions script" name
    $catcon_KillAllSessScript = 
      kill_session_script_name($catcon_LogFilePathBase, "ALL");

    # save name of a script that will be used to generate a 
    # "kill session script"
    $catcon_GenKillSessScript = 
      $ENV{ORACLE_HOME}."/rdbms/admin/catcon_kill_sess_gen.sql";

    # look for catcon_kill_sess_gen.sql in the current catcon.pm file directory
    # if ORACLE_HOME has not yet been properly setup
    if (!-e $catcon_GenKillSessScript) {
      my $catcon_dir = dirname(File::Spec->rel2abs(__FILE__));
      $catcon_GenKillSessScript = $catcon_dir."/catcon_kill_sess_gen.sql";
      log_msg("catconInit: catcon_kill_sess_gen.sql not found at ".
              "ORACLE_HOME, will instead use $catcon_GenKillSessScript\n");
    }

    # bug 25315864: determine connect strings which need to be used to ensure 
    # connection to a given instance name

    if ($catcon_DebugOn) {
      log_msg <<msg;
catconInit: call gen_inst_conn_strings to obtain instance connect strings
msg
    }

    # Bug 25366291: in some cases, gen_inst_conn_strings may fetch no rows, 
    #   causing @catcon_InstConnStr to remain undefined and 
    #   gen_inst_conn_strings to return 0. 
    #
    #   Rather than treating it as an error, we will force all processing to 
    #   occur on the default instance
    if (! defined gen_inst_conn_strings(\@catcon_InternalConnectString, 
                                        $catcon_DoneCmd, $doneFileNamePrefix, 
                                        \%catcon_InstConnStr, 
                                        $catcon_DebugOn)) {
      log_msg("catconInit: unexpected error in gen_inst_conn_strings\n");
      return 1;
    }

    # it is possible that we may start more processes than there are PDBs (or 
    # more than 1 process when operating on a non-CDB.)  It's OK since a user 
    # can tell catconExec() that some scripts may be executed concurrently, 
    # so we may have more than one process working on the same Container (or 
    # non-CDB.)  If a user does not want us to start too many processes, he 
    # can tell us how many processes to start (using -n option to catcon.pl, 
    # for instance)

    # start processes
    if (start_processes($catcon_NumProcesses, $catcon_LogFilePathBase, 
                        @catcon_FileHandles, @catcon_ProcIds,
                        @catcon_Containers, $catcon_Root,
                        @catcon_UserConnectString, 
                        $catcon_EchoOn, $catcon_ErrLogging, $catcon_DebugOn, 1,
                        $catcon_UserScript, $catcon_FedRoot, 
                        $catcon_DoneCmd, $catcon_DisableLockdown,
                        -1, undef, $catcon_GenKillSessScript,
                        $catcon_KillAllSessScript, \%catcon_InstProcMap, 
                        \%catcon_ProcInstMap, \%catcon_InstConnStr)) {
      return 1;
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconInit: started SQL*Plus processes.\n");
    }

    # remember whether we are being invoked from a GUI tool which on Windows 
    # means that the passwords and hidden parameters need not be hidden
    $catcon_GUI = $GUI;

    # check for uncommitted transaction after running every script?
    # defaults to FALSE; caller can modify by calling catconXact
    $catcon_UncommittedXactCheck = 0;

    # run scripts against all PDBs and then against the Root
    # defaults to FALSE; caller can modify by calling catconReverse
    $catcon_PdbsThenRoot = 0;

    # should non-existent and closed PDBs be ignored when constructing a list 
    # of Containers against which to run scripts?
    # defaults to FALSE; caller can modify by calling catconForce
    $catcon_IgnoreInaccessiblePDBs = 0;

    # remember that the database is open
    $catcon_DbClosed = 0;

    # remember that initialization has completed successfully
    $catcon_InitDone = 1;

    # Bug 25061922: obtain DBMS version to help catcon to use 
    # backward-compatible syntax if working on earlier version DBMS
    $catcon_DbmsVersion = 
      get_dbms_version(@catcon_InternalConnectString, 
                       $catcon_DoneCmd, $doneFileNamePrefix, $catcon_DebugOn);

    if (!(defined $catcon_DbmsVersion)) {
      log_msg("catconInit: unexpected error in get_dbms_version\n");
      return 1;
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconInit: DBMS version: $catcon_DbmsVersion.\n");
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconInit: initialization completed successfully (".
              TimeStamp().")\n");
    }

    #
    # Workaround for bug 18969473
    #
    if ($catcon_Windows) {
      open(STDIN, "<", "NUL") || die "open NUL: $!";
    }

    # success
    return 0;
  }

  #
  # catconXact - set a flag indicating whether we should check for scripts 
  #     ending with uncommitted transaction
  #
  # Parameters:
  #   - value to assign to $catcon_UncommittedXactCheck
  #
  sub catconXact ($) {
    my ($UncommittedXactCheck) = @_;

    # remember whether the caller wants us to check for uncommitted 
    # transactions at the end of scripts
    $catcon_UncommittedXactCheck = $UncommittedXactCheck;
  }

  #
  # catconReverse - set a flag indicating whether we should run the script 
  # against all PDBs and then against the Root
  #
  # Parameters:
  #   - value to assign to $catcon_PdbsThenRoot
  #
  sub catconReverse ($) {
    my ($f) = @_;

    # remember whether the caller wants us to check for uncommitted 
    # transactions at the end of scripts
    $catcon_PdbsThenRoot = $f;
  }

  #
  # catconPdbMode - remember the mode in which all PDBs being operated upon 
  #                 must be opened
  #
  # Parameters:
  #   - value to assign to $catcon_AllPdbMode
  #
  sub catconPdbMode ($) {
    my ($mode) = @_;
   
    # verify that the specified mode is valid
    if (!valid_pdb_mode($mode)) {
      log_msg("catconPdbMode: Unexpected value ($mode) for mode\n");
      return 1;
    }
    
    # we don't want to change the mode in which PDBs are to be opened once it 
    # has been set so as to avoid resetting mode of PDBs after it has been 
    # already changed
    if ($catcon_AllPdbMode != CATCON_PDB_MODE_NA) {
      log_msg("catconPdbMode: may not be called more than once ".
              "(current mode = $catcon_AllPdbMode)\n");
      return 1;
    }

    $catcon_AllPdbMode = $mode;
  }

  #
  # catconAllInst - remember whether the caller has informed us that if 
  #                 running against a CDB, PDBs should be open using all 
  #                 available instances
  #
  # Parameters:
  #   - value to assign to $catcon_AllInst
  #
  sub catconAllInst ($) {
    my ($allInst) = @_;
   
    $catcon_AllInst = 0;
  }

  #
  # catconForce - set a flag indicating whether we should ignore closed or 
  # non-existent PDBs when constructing a list of Containers against which to 
  # run scripts
  #
  # Parameters:
  #   - value to assign to $catcon_IgnoreInaccessiblePDBs
  #
  sub catconForce ($) {
    my ($Ignore) = @_;

    # remember whether the caller wants us to ignore closed and 
    # non-existent PDBs 
    $catcon_IgnoreInaccessiblePDBs = $Ignore;
  }


  #
  # catconUpgForce - set a flag indicating whether we should ignore all PDBs
  # when constructing a list of Containers against which to 
  # run scripts
  #
  # Parameters:
  #   - value to assign to $catcon_IgnoreAllInaccessiblePDBs
  #
  sub catconUpgForce ($) {
    my ($Ignore) = @_;

    # remember whether the caller wants us to ignore all PDB's 
    $catcon_IgnoreAllInaccessiblePDBs = $Ignore;
  }

  #
  # catconUpgSetPdbOpen - Sets the array value for the PDB to open
  #                       which is opened by upgrade after we called
  #                       catconInit.
  # Parameters:
  #   - PDB to set open
  #
  # Returns:
  #   1 - Failure
  #   0 - Success
  #
  sub catconUpgSetPdbOpen ($$) {
    my ($pPdb,$DebugOn) = @_;

    my $curCon; # index for catcon_AllContainers array

    #
    # Check to make sure catconInit has been called
    #
    if (!$catcon_InitDone) {
      log_msg("catconUpgSetPdbOpen: catconInit has not been run\n");
      return 1;
    }

    #
    # Display Debug Message
    #
    if ($DebugOn) {
      my $msg = <<catconUpgSetPdbOpen_DEBUG;
  running catconUpgSetPdbOpen(pPdb   = $pPdb,
                              Debug  = $DebugOn);
catconUpgSetPdbOpen_DEBUG
      log_msg($msg);
    }

    #
    # If the specified PDB exists, set open flag
    # 
    if (exists $catcon_IsConOpen{$pPdb}) {
      $catcon_IsConOpen{$pPdb} = 'Y';
      return 0;
    }

    #
    # Pdb not found
    #
    log_msg("catconUpgSetPdbOpen: Cannot set Pdb to open. ".
            "[$pPdb] is not found\n");
    return 1;

  }

  #
  # catconEZConnect - store a string consisting of EZConnect strings 
  # corresponding to RAC instances to be used to run scripts
  #
  # Parameters:
  #   - value to assign to $catcon_EZConnect
  #
  sub catconEZConnect ($) {
    my ($EZConnect) = @_;

    $catcon_EZConnect = $EZConnect;
  }

  #
  # catconVerbose - store the flag indicating whether verbose output is desired
  #
  # Parameters:
  #   - value to assign to $catcon_Verbose
  #
  sub catconVerbose ($) {
    my ($Verbose) = @_;

    $catcon_Verbose = $Verbose;
  }

  #
  # catconIgnoreErr - remember which errors should be ignored
  #
  # Parameters:
  #   - space-separated list of indicators of which errors should be ignored
  #
  sub catconIgnoreErr($) {
    my ($Indicators) = @_;

    # if no indicators were supplied, we are done
    if (!$Indicators) {
      return 0;
    }

    my @IndArr = split(/  */, $Indicators);

    # iterate over indicators, marking corresponding %catcon_ErrorsToIgnore 
    # elements
    for my $Ind ( @IndArr ) {
      if (!(defined $catcon_ErrorsToIgnore{$Ind})) {
        # unrecognized indicator
        print STDERR "catconIgnoreErr: unrecognized indicator - $Ind\n";
        print STDERR "\t The following indicators are supported:\n";
        for my $ValidInd (keys %catcon_ErrorsToIgnore) {
          print STDERR "\t\t$ValidInd \n";
        }
        return 1;
      }

      $catcon_ErrorsToIgnore{$Ind} = 1;
    }

    return 0;
  }

  #
  # catconFedRoot - store name of a Root of a Federation against members of 
  #                 which script(s) are to be run
  #
  # Parameters:
  #   - value to assign to $catcon_FedRoot
  #
  sub catconFedRoot ($) {
    my ($FedRoot) = @_;

    $catcon_FedRoot = $FedRoot;
  }

  #
  # catconDisableLockdown - store boolean indicating whether lockdown needs to
  #                         be disabled.
  #
  # Parameters:
  #   - value to assign to $catcon_DisableLockdown
  #
  sub catconDisableLockdown ($) {
    my ($DisableLockdown) = @_;

    $catcon_DisableLockdown = $DisableLockdown;
  }

  #
  # catconRecoverFromChildDeath - store boolean indicating whether death of a 
  #                               spawned sqlplus process should cause catcon
  #                               to die.
  #
  # Parameters:
  #   - value to assign to $catcon_RecoverFromChildDeath
  #
  sub catconRecoverFromChildDeath ($) {
    my ($recoverFromChildDeath) = @_;

    $catcon_RecoverFromChildDeath = $recoverFromChildDeath;
  }

  # catconExec - run specified sqlplus script(s) or SQL statements
  #
  # If connected to a non-Consolidated DB, each script will be executed 
  # using one of the processes connected to the DB.
  #
  # If connected to a Consolidated DB and the caller requested that all 
  # scripts and SQL statements be run against the Root (possibly in addition 
  # to other Containers), each script and statement will be executed in the 
  # Root using one of the processes connected to the Root.
  #
  # If connected to a Consolidated DB and were asked to run scripts and SQL 
  # statements in one or more Containers besides the Root, all scripts and 
  # statements will be run against those PDBs in parallel.
  #
  # Bug 18085409: if connected to a CDB and $catcon_PdbsThenRoot is set, the 
  #   order will be reversed and we will run scripts against all PDBs and 
  #   then against the Root.  
  #   To accomplish this without substantially rewriting catconExec, I 
  #   - moved the whole catconExec into catconExec_int() and
  #   - changed catconExec to call catconExec_int as follows:
  #     - if not connected to a CDB or if $catcon_PdbsThenRoot is not set,
  #       - just call catconExec_int
  #     - else (i.e. connected to a CDB and $catcon_PdbsThenRoot is set)
  #       - call catconExec_int passing it name of the Root as a Container 
  #         in which not to run scripts
  #       - if the preceding call does not report an error, call 
  #         catconExec_int telling it to run scripts only in the Root
  #
  # Parameters:
  #   - a reference to an array of sqlplus script name(s) or SQL statement(s); 
  #     script names are expected to be prefixed with @
  #   - an indicator of whether scripts need to be run in order
  #       TRUE => run in order
  #   - an indicator of whether scripts or SQL statements need to be run only 
  #     in the Root if operating on a CDB (temporarily overriding whatever 
  #     was set by catconInit) 
  #       TRUE => if operating on a CDB, run in Root only
  #   - an indicator of whether per process initialization/completion 
  #     statements need to be issued
  #     TRUE => init/comletion statements, if specified, will be issued
  #   - a reference to a list of names of Containers in which to run scripts 
  #     and statements during this invocation of catconExec; does not 
  #     overwrite @catcon_Containers so if no value is supplied for this or 
  #     the next parameter during subsequent invocations of catconExec, 
  #     @catcon_Containers will be used
  #
  #     NOTE: This parameter should not be defined if
  #           - connected to a non-CDB
  #           - caller told us to run statements/scripts only in the Root
  #           - caller has supplied us with a list of names of Containers in 
  #             which NOT to run scripts
  #
  #   - a reference to a list of names of Containers in which NOT to run 
  #     scripts and statements during this invocation of catconExec; does not 
  #     overwrite @catcon_Containers so if no value is supplied for this or 
  #     the next parameter during subsequent invocations of catconExec, 
  #     @catcon_Containers will be used
  #
  #     NOTE: This parameter should not be defined if
  #           - connected to a non-CDB
  #           - caller told us to run statements/scripts only in the Root
  #           - caller has supplied us with a list of names of Containers in 
  #             which NOT to run scripts
  #
  #   - a reference to a custom Errorlogging Identifier; if specified, we will
  #     use the supplied Identifier, ignoring $catcon_NoErrLogIdent
  #   - a query to run within a container to determine if the array should
  #     be executed in that container.  A NULL return value causes the
  #     script array to NOT be run.
  #
  sub catconExec(\@$$$\$\$\$\$) {

    my ($StuffToRun, $SingleThreaded, $RootOnly, $IssuePerProcStmts, 
        $ConNamesIncl, $ConNamesExcl, $CustomErrLoggingIdent, $skipProcQuery) = @_;

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconExec: start executing scripts/SQL statements\n")
    }

    if ($catcon_DebugOn) {
      log_msg("catconExec\n\tScript names/SQL statements:\n");
      foreach (@$StuffToRun) {
        log_msg("\t\t$_\n");
      }

      log_msg("\tSingleThreaded        = $SingleThreaded\n");
      log_msg("\tRootOnly              = $RootOnly\n");
      log_msg("\tIssuePerProcStmts     = $IssuePerProcStmts\n");
      if ($$ConNamesIncl) {
        log_msg("\tConNamesIncl          = $$ConNamesIncl\n");
      } else {
        log_msg("\tConNamesIncl            undefined\n");
      }
      if ($$ConNamesExcl) {
        log_msg("\tConNamesExcl          = $$ConNamesExcl\n");
      } else {
        log_msg("\tConNamesExcl            undefined\n");
      }
      if ($$CustomErrLoggingIdent) {
        log_msg("\tCustomErrLoggingIdent = $$CustomErrLoggingIdent\n");
      } else {
        log_msg("\tCustomErrLoggingIdent   undefined\n");
      }
      if ($$skipProcQuery) {
        log_msg("\tskipProcQuery = $$skipProcQuery\n");
      }  else {
        log_msg("\tskipProcQuery   undefined\n");
      }

      log_msg("\t(".TimeStamp().")\n");
    }

    # there must be at least one script or statement to run
    if (!@$StuffToRun || $#$StuffToRun == -1) {
      log_msg("catconExec: At least one sqlplus script name or SQL statement must be supplied\n");
      return 1;
    }

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconExec: catconInit has not been run\n");
      return 1;
    }

    # a number of checks if either a list of Containers in which to run 
    # scripts or a list of Containers in which NOT to tun scripts was specified
    if ($$ConNamesIncl || $$ConNamesExcl) {
      if (!@catcon_Containers) {
        # Container names specified even though we are running against a 
        # non-CDB
        log_msg("catconExec: Container names specified for a non-CDB\n");

        return 1;
      }

      if ($RootOnly) {
        # Container names specified even though we were told to run ONLY 
        # against the Root
        log_msg("catconExec: Container names specified while told to run only against the Root\n");

        return 1;
      }

      if ($catcon_PdbsThenRoot) {
        # Container names specified even though we were told to run in 
        # ALL PDBs and then in the Root
        log_msg("catconExec: Container names specified while told to run in ALL PDBs and then in Root\n");

        return 1;
      }

      # only one of the lists may be defined
      if ($$ConNamesIncl && $$ConNamesExcl) {
        log_msg <<msg;
catconExec: both inclusive
        $$ConNamesIncl
    and exclusive
        $$ConNamesExcl
    Container name lists are defined
msg
        return 1;
      }
    }

    if ($RootOnly && $catcon_PdbsThenRoot) {
      # RootOnly may not be specified if were told to run in ALL PDBs and 
      # then in the Root
        log_msg <<msg;
catconExec: Caller asked to run only in the Root while also indicating that 
    scripts must be run in ALL PDBs and then in Root
msg

        return 1;
    }

    my $RetVal;

    if (!@catcon_Containers || !$catcon_PdbsThenRoot) {
      # non-CDB or not told to reverse the order of Containers - just pass 
      # parameters on to catconExec_int
      $RetVal = catconExec_int($StuffToRun, $SingleThreaded, $RootOnly, 
                               $IssuePerProcStmts, $ConNamesIncl, 
                               $ConNamesExcl, $CustomErrLoggingIdent,
                               $skipProcQuery);
    } else {
      # connected to a CDB and need to run scripts in ALL PDBs and then in
      # the Root

      # bug 23292787: it is possible that the caller of catcon.pl has supplied
      #   both -c/-C and -r, in which case we need to determine whether the 
      #   set of Containers produced by processing -c/-C (and contained in 
      #   @catcon_Containers) includes the Root and one or more PDBs.  
      #   If we discover that it contained at least one PDB, we will process 
      #   PDBs. If we discover that it also contained Root, we will then 
      #   process the Root
      if ($catcon_DebugOn) {
        log_msg("catconExec: caller requested that we run scripts in PDBs ".
                "and then in Root\n");
        log_msg("\twill determine if the set of Containers in which we ".
                "need to run includes Root and PDBs\n");
      }

      my ($root, $pdbs) = split_root_and_pdbs(@catcon_Containers, 
                                              $catcon_Root);

      if (@$pdbs) {
        my $pdb_list = join(' ', @$pdbs);

        if ($catcon_DebugOn) {
          log_msg("catconExec: set of Containers included at least one PDB, ".
                  "so run scripts in ($pdb_list) first\n");
        }

        # first call catconExec_int to run scripts in all Containers other than
        # the Root
        $RetVal = catconExec_int($StuffToRun, $SingleThreaded, $RootOnly, 
                                 $IssuePerProcStmts, \$pdb_list, 
                                 $ConNamesExcl, $CustomErrLoggingIdent,
                                 $skipProcQuery);
      }

      # skip processing the Root if running in a child process forked in 
      # catconExec to process PDBs open on a single instance
      if (defined $root && !$catcon_ChildProc) {
        if ($catcon_DebugOn) {
          log_msg("catconExec: set of Containers included the Root, ".
                  "so run it in $catcon_Root\n");
        }

        # if catconExec_int reported an error, no further processing is needed
        if (!$RetVal) {
          # finally, call catconExec_int to run scirpts only in the Root
          $RetVal = catconExec_int($StuffToRun, $SingleThreaded, 1,
                                   $IssuePerProcStmts, $ConNamesIncl, 
                                   $ConNamesExcl, $CustomErrLoggingIdent,
                                   $skipProcQuery);
        }
      }
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      if ($RetVal) {
        log_msg("catconExec: an error (".$RetVal.") was encountered\n");
      } else {
        log_msg("catconExec: finished executing scripts/SQL statements\n")
      }
    }

    # if running in a child process, exit
    # 
    # NOTE: unfortunately, (at least as far as I can tell) on Linux, exit does 
    #       not seem to cause the forked process to exit (as can be seen by 
    #       examining its log file), so I force it to commit suicide by 
    #       sending SIGKILL to itself
    if ($catcon_ChildProc) {
      if ($catcon_DebugOn) {
        log_msg("catconExec: child process $$ exiting with $RetVal\n");
      }

      # commit suicide
      my @me;
      push @me, $$;

      # construct a done file informing the parent that this child process 
      # finished its work
      my $childDoneFile = child_done_file_name($catcon_LogFilePathBase, $$);
      my $childDoneFileFH;
      
      if (sysopen($childDoneFileFH, $childDoneFile, 
                  O_CREAT | O_WRONLY | O_TRUNC, 0600)) {
        print $childDoneFileFH $RetVal."\n";
        close($childDoneFileFH);
      } else {
        log_msg("catconExec: child process finished with return value = ".
                $RetVal.", but done file could not be created\n");
      }

      send_sig_to_procs(@me, 9);
    }

    return $RetVal;
  }

  # NOTE: catconExec_int() should ONLY be called via catconExec which 
  #       performs error checking to ensure that catconExec_int is not 
  #       presented with a combination of arguments which it is not prepared 
  #       to handle
  sub catconExec_int (\@$$$\$\$\$\$) {

    my ($StuffToRun, $SingleThreaded, $RootOnly, $IssuePerProcStmts, 
        $ConNamesIncl, $ConNamesExcl, $CustomErrLoggingIdent, $skipProcQuery) 
      = @_;

     #Hack add con_id
    my $CurrentContainerQuery = qq#select '==== Current Container = ' || SYS_CONTEXT('USERENV','CON_NAME') || ' Id = ' || SYS_CONTEXT('USERENV','CON_ID') || ' ====' AS now_connected_to from sys.dual;\n/\n#;

    # this array will be used to keep track of processes to which statements 
    # contained in @$PerProcInitStmts need to be sent because they [processes]
    # have not been used in the course of this invocation of catconExec()
    my @NeedInitStmts = ();
    my $bInitStmts = 0;

    # script invocations (together with parameters) and/or SQL statements 
    # which will be executed
    my @ScriptPaths;

    # same as @ScriptPaths but secret parameters will be replaced with prompts
    # so as to avoid storing sensitive data in log files (begin_running and 
    # end_running queries) and error logging tables (in identifier column)
    my @ScriptPathsToDisplay;

    # if $catcon_SpoolOn is set, we will spool output produced by running all 
    # scripts into spool files whose names will be stored in this array
    my @SpoolFileNames;

    my $NextItem;

    # validate script paths and add them, along with parameters and 
    # SQL statements, if any, to @ScriptPaths

    if ($catcon_DebugOn) {
      log_msg("catconExec: validating scripts/statements supplied by the caller\n");
    }

    # if the user supplied regular and/or secret argument delimiters, this 
    # variable will be set to true after we encounter a script name to remind 
    # us to check for possible arguments
    my $LookForArgs = 0;
    
    # indicators of whether a string may be a regular or a secret script 
    # argument or an argument whose value is contained in an env var
    my $RegularScriptArg;
    my $SecretScriptArg;
    my $EnvScriptArg;

    foreach $NextItem (@$StuffToRun) {

      if ($catcon_DebugOn) {
        log_msg("catconExec: going over StuffToRun: NextItem = $NextItem\n");
      }

      if ($NextItem =~ /^@/) {
        # leading @ implies that $NextItem contains a script name
        
        # name of SQL*Plus script
        my $FileName;

        if ($catcon_DebugOn) {
          log_msg("catconExec: next script name = $NextItem\n");
        }

        # strip off the leading @ before prepending source directory name to 
        # script name
        ($FileName = $NextItem) =~ s/^@//;

        # validate path of the sqlplus script and add it to @ScriptPaths
        my $Path = 
          validate_script_path($FileName, $catcon_SrcDir, $catcon_Windows, 
                               $catcon_ErrorsToIgnore{script_path}, 
                               $catcon_DebugOn);

        if (!$Path) {
          log_msg <<msg;
catconExec: empty Path returned by validate_script_path for 
    SrcDir = $catcon_SrcDir, FileName = $FileName
msg
          return 1;
        }

        push @ScriptPaths, "@".$Path;

        # before pushing a new element onto @ScriptPathsToDisplay, replace 
        # single and double quotes in the previous element (if any) with # 
        # to avoid errors in begin/end_running queries and in 
        # SET ERRORLOGGING ON IDENTIFIER ... statements
        if ($#ScriptPathsToDisplay >= 0) {
          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] =~ s/['"]/#/g;
        }

        # assuming it's OK to show script paths
        push @ScriptPathsToDisplay, "@".$Path; 

        if ($catcon_DebugOn) {
          log_msg("catconExec: full path = $Path\n");
        }
        
        # if caller requested that output of running scripts be spooled, 
        # construct prefix of a spool file name and add it to @SpoolFileNames 
        if ($catcon_SpoolOn) {
          # spool files will get stored in the same directory as log files, 
          # and their names will start with "log file base" followed by '_'
          my $SpoolFileNamePrefix = $catcon_LogFilePathBase .'_';

          if ($catcon_DebugOn) {
            log_msg("catconExec: constructing spool file name prefix: log file path base +' _' = $SpoolFileNamePrefix\n");
          }
          
          # script file name specified by the caller may contain directories;
          # since we want to store spool files in the same directory as log
          # files, we will replace slashes with _
          my $Temp = $FileName;

          $Temp =~ s/[\/\\]/_/g;

          if ($catcon_DebugOn) {
            log_msg("catconExec: constructing spool file name prefix: script name without slashes = $Temp\n");
          }

          # we also want to get rid of script file name extension, so we look 
          # for the last occurrence of '.' and strip off '.' along with 
          # whatever follows it
          my $DotPos = rindex($Temp, '.');

          if ($DotPos > 0) {
            # script name should not start with . (if it does, I will let it 
            # be, I guess
            $Temp = substr($Temp, 0, $DotPos);

            if ($catcon_DebugOn) {
              log_msg <<msg
catconExec: constructing spool file name prefix: 
    after stripping off script file name extension, script name = $Temp
msg
            }
          } else {
            if ($catcon_DebugOn) {
              log_msg("catconExec: constructing spool file name prefix: script file name contained no extension\n");
            }
          }

          push @SpoolFileNames, $SpoolFileNamePrefix.$Temp.'_';

          if ($catcon_DebugOn) {
            log_msg <<msg;
catconExec: constructing spool file name prefix: 
    added $SpoolFileNames[$#SpoolFileNames] to the list
msg
          }
        }

        if ($catcon_RegularArgDelim || $catcon_SecretArgDelim) {
          # remember that script name may be followed by argument(s)
          $LookForArgs = 1;

          if ($catcon_DebugOn) {
            log_msg("catconExec: prepare to handle arguments\n");
          }
        }
      } elsif (   ($RegularScriptArg = 
                     (   $catcon_RegularArgDelim 
                      && $NextItem =~ /^$catcon_RegularArgDelim/))
               || ($SecretScriptArg = 
                     (   $catcon_SecretArgDelim 
                      && $NextItem =~ /^$catcon_SecretArgDelim/))
               || ($EnvScriptArg = ($NextItem =~ /^--e/))) {
        # looks like an argument to a script; make sure we are actually 
        # looking for script arguments
        if (!$LookForArgs) {
          log_msg("catconExec: unexpected script argument ($NextItem) encountered\n");
          return 1;
        }

        if ($catcon_DebugOn) {
          log_msg("catconExec: processing script argument string ($NextItem)\n");
        }

        # because of short-circuiting of logical operators, if 
        # $RegularScriptArg got set to TRUE, we will never even try to 
        # determine if this string could also represent a secret argument, so 
        # if code in catconInit has determined that a regular argument 
        # delimiter is a prefix of secret argument delimiter, we need to 
        # determine whether this string may also represent a secret argument
        # (and if a regular argument delimiter is NOT a prefix of secret 
        # argument delimiter, reset $SecretScriptArg in case the preceding 
        # item represented a secret argument)
        if ($RegularScriptArg) {
          $SecretScriptArg = 
               ($catcon_ArgToPick == $catcon_PickSecretArg) 
            && ($NextItem =~ /^$catcon_SecretArgDelim/);
        }

        # If this argument could be either (i.e. one of the 
        # argument-introducing strings is a prefix of the other), use 
        # $catcon_ArgToPick to decide how to treat this argument
        # argument stripped off the string marking it as such
        if ($RegularScriptArg && $SecretScriptArg) {

          if ($catcon_ArgToPick == $catcon_PickRegularArg) {
            $SecretScriptArg = undef;  # treat this arg as a regular arg

            if ($catcon_DebugOn) {
              log_msg <<msg;
catconExec: (catcon_ArgToPick = $catcon_ArgToPick, catcon_PickRegularArg = $catcon_PickRegularArg) 
    argument string ($NextItem) could be represent either regular or secret 
    argument - treat as regular
msg
            }
          } else {
            $RegularScriptArg = undef; # treat this arg as a secret arg

            if ($catcon_DebugOn) {
              log_msg <<msg;
catconExec: argument string ($NextItem) could be represent either
    regular or secret argument - treat as secret
msg
            }
          }
        }

        # - if it is a secret argument, prompt the user to enter its value 
        #   and append the value entered by the user to the most recently 
        #   added element of @ScriptPaths
        # - if it is a "regular" argument, append this argument to the most 
        #   recently added element of @ScriptPaths
        # - if it is an "env" argument, obtain value of the specified 
        #   environment variable and append it to the most recently added 
        #   element of @ScriptPaths
        # In all cases, we will strip off the string introducing the 
        # argument and quote the string before adding it to 
        # $ScriptPaths[$#@ScriptPaths] in case it contains embedded blanks.
        # 
        my $Arg;
    
        if ($SecretScriptArg) {
          ($Arg = $NextItem) =~ s/^$catcon_SecretArgDelim//;

          if ($catcon_DebugOn) {
            log_msg("catconExec: prepare to obtain value for secret argument after issuing ($Arg)\n");
          }

          # get the user to enter value for this argument
          log_msg("$Arg: ");

          # do not set ReadMode to noecho if invoked on Windows from a GUI 
          # tool which requires that we do not hide passwords and hidden 
          # parameters
          if (!($catcon_Windows && $catcon_GUI)) {
            ReadMode 'noecho';
          }

          my $ArgVal = ReadLine 0;
          chomp $ArgVal;

          # do not restore ReadMode if invoked on Windows from a GUI tool 
          # which requires that we do not hide passwords and hidden parameters
          if (!($catcon_Windows && $catcon_GUI)) {
            ReadMode 'normal';
          }

          print "\n";

          # quote value entered by the user and append it to 
          # $ScriptPaths[$#ScriptPaths]
          $ScriptPaths[$#ScriptPaths] .= " '" .$ArgVal."'";

          # instead of showing secret parameter value supplied by the user, 
          # we will show the prompt in response to which the value was 
          # supplied
          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] .= " '" .$Arg."'";

          if ($catcon_DebugOn) {
            log_msg("catconExec: added secret argument to script invocation string\n");
          }
        } elsif ($RegularScriptArg) {
          ($Arg = $NextItem) =~ s/^$catcon_RegularArgDelim//;

          # quote regular argument and append it to 
          # $ScriptPaths[$#ScriptPaths]
          $ScriptPaths[$#ScriptPaths] .= " '" .$Arg."'";

          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] .= " '" .$Arg."'";

          if ($catcon_DebugOn) {
            log_msg("catconExec: added regular argument to script invocation string\n");
          }
        } else {
          # bug 22181521 - argument whose value is contained in the 
          # specified environment variable

          # name of the environment variable
          my $envVar = substr($NextItem, 3);

          # make sure the env variable was set
          if ((! exists $ENV{$envVar}) || (! defined $ENV{$envVar})) {
            log_msg("catconExec: user parameter environment variable ".$envVar." specified with --e was undefined\n");
            return 1;
          }

          # quote value of the environment variable specified by the user 
          # and append it to $ScriptPaths[$#ScriptPaths]
          $ScriptPaths[$#ScriptPaths] .= " '" .$ENV{$envVar}."'";

          # instead of showing value of the environment variable, we will 
          # display its name
          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] .= " '\$" .$envVar."'";

          if ($catcon_DebugOn) {
            log_msg("catconExec: added environmet argument to script invocation string\n");
          }
        }

        if ($catcon_DebugOn) {
          log_msg("catconExec: script invocation string constructed so far:\n\t$ScriptPathsToDisplay[$#ScriptPathsToDisplay]\n");
        }
      } else {
        # $NextItem must contain a SQL statement which we will copy into 
        # @ScriptPaths (a bit of misnomer in this case, sorry)

        if ($catcon_DebugOn) {
          log_msg("catconExec: next SQL statement = $NextItem\n");
        }

        push @ScriptPaths, $NextItem;

        # before pushing a new element onto @ScriptPathsToDisplay, replace 
        # single and double quotes in the previous element (if any) with # 
        # to avoid errors in begin/end_running queries and in 
        # SET ERRORLOGGING ON IDENTIFIER ... statements
        if ($#ScriptPathsToDisplay >= 0) {
          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] =~ s/['"]/#/g;
        }

        # assuming it's ok to display SQL statements; my excuse: SET 
        # ERRORLOGGING already does it for statements which result in errors
        push @ScriptPathsToDisplay, $NextItem;

        # we expect no arguments following a SQL statement
        $LookForArgs = 0;

        if ($catcon_DebugOn) {
          log_msg("catconExec: saw SQL statement - do not expect any arguments\n");
        }

        if ($catcon_SpoolOn) {
          push @SpoolFileNames, "";

          if ($catcon_DebugOn) {
            log_msg("catconExec: saw SQL statement - added empty spool file name prefix to the list\n");
          }
        }
      }
    }
    
    # replace single and double quotes in the last element of 
    # @ScriptPathsToDisplay with # to avoid errors in begin/end_running 
    # queries and in SET ERRORLOGGING ON IDENTIFIER ... statements
    $ScriptPathsToDisplay[$#ScriptPathsToDisplay] =~ s/['"]/#/g;

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconExec: finished examining scripts/SQL statements to be executed.\n");
    }

    # if running against a non-Consolidated Database
    #   - each script/statement will be run exactly once; 
    # else 
    #   - if the caller instructed us to run only in the Root, temporarily 
    #     overriding the list of Containers specified when calling catconInit, 
    #     or if the Root is among the list of Containers against which we are 
    #     to run
    #     - each script/statement will be run against the Root
    #   - then, if running against one or more PDBs
    #     - each script/statement will be run against all such PDBs in parallel

    # offset into the array of process file handles
    my $CurProc = 0;

    # compute number of processes which will be used to run script/statements 
    # specified by the caller; used to determine when all processes finished 
    # their work
    my $ProcsUsed;

    # if the caller requested that we generate spool files for output 
    # produced by running supplied scripts, an array of spool file name 
    # prefixes has already been constructed.  All that's left is to determine 
    # the suffix:
    #   - if running against a non-Consolidated DB, suffix will be an empty 
    #     string
    #   - otherwise, it needs to be set to the name of the Container against 
    #     which script(s) will be run
    my $SpoolFileNameSuffix;

    # initialize array used to keep track of whether we need to send 
    # initialization statements to a process used for the first time during 
    # this invocation of catcon
    if (   $IssuePerProcStmts 
        && $catcon_PerProcInitStmts && $#$catcon_PerProcInitStmts >= 0) {
      @NeedInitStmts = (1) x $catcon_NumProcesses;
    }
        
    # if ERORLOGGING is enabled, this will be used to store IDENTIFIER
    my $ErrLoggingIdent;

    # normally this will be set to @catcon_Containers, but if the caller
    # specified names of Containers in which to run or not to run scripts 
    # and/or statements during this invocation of catconExec, it will get 
    # modified to reflect this
    my @Containers;

    @Containers = @catcon_Containers;
    if ($$ConNamesIncl || $$ConNamesExcl) {
      if ($$ConNamesExcl) {
        if (!(@Containers = 
                validate_con_names($$ConNamesExcl, 1, @catcon_AllContainers, 
                    $catcon_AllPdbMode == CATCON_PDB_MODE_UNCHANGED
                      ? \%catcon_IsConOpen : undef, 
                    $catcon_IgnoreInaccessiblePDBs,
                    $catcon_IgnoreAllInaccessiblePDBs, 
                    $catcon_DebugOn))) {
          log_msg("catconExec: Unexpected error returned by ".
                  "validate_con_names\n");
          return 1;
        }
      } else {
        if (!(@Containers = 
                validate_con_names($$ConNamesIncl, 0, @catcon_AllContainers, 
                    $catcon_AllPdbMode == CATCON_PDB_MODE_UNCHANGED
                      ? \%catcon_IsConOpen : undef, 
                    $catcon_IgnoreInaccessiblePDBs,
                    $catcon_IgnoreAllInaccessiblePDBs, 
                    $catcon_DebugOn))) {
          log_msg("catconExec: Unexpected error returned by ".
                  "validate_con_names\n");
          return 1;
        } 
      }

      if ($catcon_DebugOn) {
        log_msg("catconExec: During this invocation ONLY, run only ".
                "against the following Containers {\n");
        foreach (@Containers) {
          log_msg("\t\t$_\n");
        }
        log_msg("catconExec: }\n");
      }
    }

    # if connected to a CDB, save name of a CDB Root or, if told to run 
    # scripts in a Federation Root and all Federation PDBs that belong to it,
    # of the Federation Root
    my $CDBorFedRoot;

    if (@Containers) {
      $CDBorFedRoot = ($catcon_FedRoot) ? $catcon_FedRoot : $catcon_Root;
    }

    # skip the portion of the code that deals with running all scripts in the 
    # Root, Federation Root or in a non-CDB if we need to run it in one or 
    # more PDBs but not in the Root or in explicitly specified Federation Root
    if (   @Containers && !$RootOnly 
        && ($Containers[0] ne $CDBorFedRoot)) {
      if ($catcon_DebugOn) {
        log_msg("catconExec: skipping single-Container portion of catconExec\n");
      }

      goto skipSingleContainerRun;
    }

    if ($SingleThreaded) {
      # use 1 process to run all script(s) against a non-Consolidated DB or in 
      # the Root or the specified Container of a Consolidated DB
      $ProcsUsed = 1;
    } else {
      # each script may be run against a non-Consolidated DB or in the Root or
      # the specified Container of a Consolidated DB using a separate process
      $ProcsUsed = (@ScriptPaths > $catcon_NumProcesses) 
        ? $catcon_NumProcesses : @ScriptPaths;
    }

    # if running against a CDB, we need to connect to the Root in every 
    # process, UNLESS we were called to invoke a script in a Federation Root 
    # and all of its Federation PDBs, in which case we need to connect to the 
    # Federation Root
    if (@Containers) {

      for ($CurProc = 0; $CurProc < $ProcsUsed; $CurProc++) {

      # @@@@
      if ((index($ScriptPathsToDisplay[0], "catupgrd") == -1) &&
          (index($ScriptPathsToDisplay[0], "catshutdown") == -1))
      {
        printToSqlplus("catconExec", $catcon_FileHandles[$CurProc], 
                       qq#ALTER SESSION SET CONTAINER = "$CDBorFedRoot"#, 
                       "\n/\n", $catcon_DebugOn);

        print {$catcon_FileHandles[$CurProc]} $CurrentContainerQuery;
        $catcon_FileHandles[$CurProc]->flush;
        $bInitStmts = 1;
      }


        if ($catcon_DebugOn) {
          if ($catcon_FedRoot) {
            log_msg("catconExec: process $CurProc (id = $catcon_ProcIds[$CurProc]) connected to Federation Root Container $catcon_FedRoot\n");
          } else {
            log_msg("catconExec: process $CurProc (id = $catcon_ProcIds[$CurProc]) connected to Root Container $catcon_Root\n");
          }
        }
      }
    }

    # run all scripts in 
    # - a non-Consolidated DB or
    # - the Root of a Consolidated DB, if caller specified that they be run 
    #   only in the Root OR if Root is among multiple Containers where scripts 
    #   need to be run or
    # - in Federation Root if the user asked us to run in Federation Root 
    #   and all Federation PDBs belonging to it

    if (!@Containers) {
      if ($catcon_DebugOn || $catcon_Verbose) {
        log_msg("catconExec: will run all scripts/statements against a non-CDB\n");
      }

      if ($catcon_SpoolOn) {
        $SpoolFileNameSuffix = "";
        if ($catcon_DebugOn) {
          log_msg("catconExec: non-CDB - set SpoolFileNameSuffix to empty string\n");
        }
      }
    } else {
      # it must be the case that 
      #         $RootOnly 
      #      || ($Containers[0] eq $CDBorFedRoot)
      # for otherwise we would have jumped to skipSingleContainerRun above

      if ($catcon_DebugOn || $catcon_Verbose) {
        if ($catcon_FedRoot) {
          log_msg("catconExec: will run all scripts/statements against Federation Root $catcon_FedRoot\n");
        } else {
          log_msg("catconExec: will run all scripts/statements against the Root (Container $catcon_Root) of a CDB\n");
        }
      }

      if ($catcon_SpoolOn) {
        # set spool file name suffix to the name of the Container in which 
        # we will be running
        $SpoolFileNameSuffix = getSpoolFileNameSuffix($CDBorFedRoot);

        if ($catcon_DebugOn) {
          log_msg("catconExec: CDB - set SpoolFileNameSuffix to $SpoolFileNameSuffix\n");
        }
      }
    }
    
    if ($catcon_ErrLogging) {
      # do not mess with $ErrLoggingIdent if $$CustomErrLoggingIdent is set 
      # since once the user provides a value for the Errorlogging Identifier, 
      # he is fully in charge of its value
      # 
      # Truth table describing how we determine whether to set the 
      # ERRORLOGGING Identifier, and if so, whether to use a default 
      # identifier or a custom identifier
      # N\C 0   1
      #   \--------
      #  0| d | c |
      #   ---------
      #  1| - | c |
      #   ---------
      #
      # N = is $catcon_NoErrLogIdent set?
      # C = is $$CustomErrLoggingIdent defined?
      # d = use default ERRORLOGGING Identifier
      # c = use custom ERRORLOGGING Identifier
      # - = do not set ERRORLOGGING Identifier
      #
      # NOTE: it's important that we we assign $$CustomErrLoggingIdent to 
      #       $ErrLoggingIdent and test the result BEFORE we test 
      #       $catcon_NoErrLogIdent because  if $$CustomErrLoggingIdent is 
      #       $defined, we will ignore catcon_NoErrLogIdent, so it's necessary 
      #       that $ErrLoggingIdent gets set to $$CustomErrLoggingIdent 
      #       regardless of whether $catcon_NoErrLogIdent is set
      if (!($ErrLoggingIdent = $$CustomErrLoggingIdent) && 
          !$catcon_NoErrLogIdent) {
        if (!@Containers) {
          $ErrLoggingIdent = "";
        } else {
          $ErrLoggingIdent = $CDBorFedRoot."::";
        }
      }

      # replace single and double quotes in $ErrLoggingIdent with # to avoid 
      # confusion
      if ($ErrLoggingIdent && $ErrLoggingIdent ne "") {
        $ErrLoggingIdent =~ s/['"]/#/g;
      }

      if ($catcon_DebugOn) {
        log_msg("catconExec: ErrLoggingIdent prefix = $ErrLoggingIdent\n");
      }
    }

    # $CurProc == -1 (any number < 0 would do, really, since such numbers 
    # could not be used to refer to a process) will indicate that we are yet 
    # to obtain a number of a process to which to send the first script (so we 
    # need to call pickNextProc() even if running single-threaded; once 
    # $CurProc >= 0, if running single-threaded, we will not be trying to 
    # obtain a new process number)
    $CurProc = -1; 

    # index into @ScriptPathsToDisplay array which will be kept in sync with 
    # the current element of @ScriptPaths
    # this index will also be used to access elements of @SpoolFileNames in 
    # sync with elements of the list of scripts and SQL statements to be 
    # executed
    my $ScriptPathDispIdx = -1;


    # check to see if the scripts should be run 
    if ($$skipProcQuery) {
        my @result = catconQuery($$skipProcQuery);
        if ($result[0] eq "") {
        	return 0;   # do not run any of the scripts
        }
    }

    # was the previous script that we ran catshutdown (meaning that the DB 
    # may be down before the current script finishes so we need to be careful 
    # about issuing statements that need to run in sqlplus)
    my $prevCatShutdown = 0;
  
    # Bug 25404458: key used to access %catcon_CatShutdown entry corresponding 
    #   to the Container being processed (CDB$ROOT or name of the App Root, if 
    #   operating on a CDB or non-CDB if operating on one)
    my $rootOrNonCdb = (defined $CDBorFedRoot) ? $CDBorFedRoot : "non-CDB";

    # processing catupgrd?
    my $currCatUpgrd = 0;
    
    foreach my $FilePath (@ScriptPaths) {
      # remember if the last script ran was catshutdown
      $prevCatShutdown = $catcon_CatShutdown{$rootOrNonCdb};

      # remember if the current script is catshutdown or catupgrd
      if (index($FilePath, "catshutdown") != -1) {
        $catcon_CatShutdown{$rootOrNonCdb} = 1;
        $currCatUpgrd = 0;
      } elsif (index($FilePath, "catupgrd") != -1) {
        $currCatUpgrd = 1;
        $catcon_CatShutdown{$rootOrNonCdb} = 0;
      } else {
        $currCatUpgrd = $catcon_CatShutdown{$rootOrNonCdb} = 0;
      }

      if (!$SingleThreaded || $CurProc < 0) {
        # find next available process
        $CurProc = pickNextProc($ProcsUsed, $catcon_NumProcesses, $CurProc + 1,
                                $catcon_LogFilePathBase, \@catcon_ProcIds, 
                                $catcon_RecoverFromChildDeath, 
                                @catcon_InternalConnectString, 
                                $catcon_KillAllSessScript,
                                $catcon_DoneCmd, $catcon_DebugOn);

        if ($CurProc < 0) {
          # some unexpected error was encountered
          log_msg("catconExec: unexpected error in pickNextProc\n");

          return 1;
        }

        # if this is the first time we are using this process and 
        # the caller indicated that one or more statements need to be executed 
        # before using a process for the first time, issue these statements now
        if ($#NeedInitStmts >= 0 && $NeedInitStmts[$CurProc]) {
          firstProcUseStmts($ErrLoggingIdent, $$CustomErrLoggingIdent,
                            $catcon_ErrLogging, 
                            \@catcon_FileHandles, $CurProc, 
                            $catcon_PerProcInitStmts, \@NeedInitStmts, 
                            $catcon_UserScript, $catcon_FedRoot, 
                            $catcon_DebugOn);
        }
      }

      # element of @ScriptPathsToDisplay which corresponds to $FilePath
      my $FilePathToDisplay = $ScriptPathsToDisplay[++$ScriptPathDispIdx];

      # @@@@
      if (!$currCatUpgrd)
      {

      # run additional init statements (set tab on, set _oracle_script, etc.)
      additionalInitStmts(\@catcon_FileHandles, $CurProc,
                          0, 0,
                          $catcon_ProcIds[$CurProc], $catcon_EchoOn,
                          $catcon_DebugOn, $catcon_UserScript, 
                          $catcon_FedRoot);

      if ($catcon_FedRoot) {
        print {$catcon_FileHandles[$CurProc]} qq#select '==== CATCON EXEC FEDERATION ROOT $catcon_FedRoot ====' AS catconsection from dual\n/\n#;
      } else {
        print {$catcon_FileHandles[$CurProc]} qq#select '==== CATCON EXEC ROOT ====' AS catconsection from dual\n/\n#;
      }

      # bracket script or statement with strings intended to make it easier 
      # to determine origin of output in the log file

      # @@@@ Added more info
      print {$catcon_FileHandles[$CurProc]} qq#select '==== $FilePathToDisplay' || ' Container:' || SYS_CONTEXT('USERENV','CON_NAME') || ' Id:' || SYS_CONTEXT('USERENV','CON_ID') || ' ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') || ' Proc:$CurProc ====' AS begin_running from sys.dual;\n/\n#;

     }

      if ($catcon_ErrLogging) {
        # construct and issue SET ERRORLOGGING ON [TABLE...] statement

        # @@@@
        if ($bInitStmts)
        {
            err_logging_tbl_stmt($catcon_ErrLogging, \@catcon_FileHandles, 
                                 $CurProc, $catcon_UserScript, $catcon_FedRoot,
                                 $catcon_DebugOn);
        }

        if ($ErrLoggingIdent) {
          # send SET ERRORLOGGING ON IDENTIFIER ... statement

          my $Stmt;

          if ($$CustomErrLoggingIdent) {
            # NOTE: if the caller of catconExec() has supplied a custom 
            #       Errorlogging Identifier, it has already been copied into 
            #       $ErrLoggingIdent which was then normalized, so our use of 
            #       $ErrLoggingIdent rather than $CustomErrLoggingIdent below 
            #       is intentional
            $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent, 0, 256)."'";
          } else { 
            $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent.$FilePathToDisplay, 0, 256)."'";
          }

          if ($catcon_DebugOn || $catcon_Verbose) {
            log_msg("catconExec: sending $Stmt to process $CurProc\n");
          }

          printToSqlplus("catconExec", $catcon_FileHandles[$CurProc], 
                 $Stmt, "\n", $catcon_DebugOn);
        }
      }

      # statement to start/end spooling output of a SQL*Plus script
      my $SpoolStmt;

      # if the caller requested that we generate a spool file and we are about 
      # to execute a script, issue SPOOL ... REPLACE
      if ($catcon_SpoolOn && $SpoolFileNames[$ScriptPathDispIdx] ne "") {
        $SpoolStmt = "SPOOL '" . $SpoolFileNames[$ScriptPathDispIdx] . $SpoolFileNameSuffix . "' REPLACE";

        if ($catcon_DebugOn) {
          log_msg("catconExec: sending $SpoolStmt to process $CurProc\n");
        }

        printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                 $SpoolStmt, "\n", $catcon_DebugOn);

        # remember that after we execute the script, we need to turn off 
        # spooling
        $SpoolStmt = "SPOOL OFF \n";
      }

      # value which will be used with ALTER SESSION SET APPLICATION ACTION
      my $AppInfoAction;

      my $LastSlash; # position of last / or \ in the script name
      
      if ($FilePath !~ /^@/) {
        $AppInfoAction = $FilePath;
      } elsif (($LastSlash = rindex($FilePath, '/')) >= 0 ||
               ($LastSlash = rindex($FilePath, '\\')) >= 0) {
        $AppInfoAction = substr($FilePath, $LastSlash + 1);
      } else {
        # FilePath contains neither backward nor forward slashes, so use it 
        # as is
        $AppInfoAction = $FilePath;
      }

      # $AppInfoAction may include parameters passed to the script. 
      # These parameters will be surrounded with single quotes, which will 
      # cause a problem since $AppInfoAction is used to construct a 
      # string parameter used with ALTER SESSION SET APPLICATION ACTION.
      # To prevent this from happening, we will replace single quotes found
      #in $AppInfoAction with #
      $AppInfoAction =~ s/[']/#/g;

      if (!@Containers) {
        $AppInfoAction = "non-CDB::".$AppInfoAction;
      } else {
        $AppInfoAction = $CDBorFedRoot."::".$AppInfoAction;
      }

      # use ALTER SESSION SET APPLICATION MODULE/ACTION to identify process, 
      # Container, if any, and script or statement being run

      if (!$currCatUpgrd && !$catcon_CatShutdown{$rootOrNonCdb}) {
        printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                       qq#ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'#,
                       "\n/\n", $catcon_DebugOn);
      }

      if ($catcon_DebugOn) {
        log_msg("catconExec: issued ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n");
      }

      if (!$currCatUpgrd && !$catcon_CatShutdown{$rootOrNonCdb}) {
        printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                       "ALTER SESSION SET APPLICATION ACTION = '$AppInfoAction'",
                       "\n/\n", $catcon_DebugOn);
      }

      if ($catcon_DebugOn) {
        log_msg("catconExec: issued ALTER SESSION SET APPLICATION ACTION = '$AppInfoAction'\n");
      }

      # execute next script or SQL statement
      if ($catcon_DebugOn || $catcon_Verbose) {
        log_script_execution($FilePathToDisplay, 
                             $CDBorFedRoot ? $CDBorFedRoot : undef, 
                             $CurProc);
      }

      # if catcon_DebugOn is set, printToSqlplus will normally issue 
      #   select "catconExec() ... as catcon_statement from dual"
      # which we need to avoid if we just finished running  catshutdown
      printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                     $FilePath, "\n", $catcon_DebugOn && !$prevCatShutdown);

      # if executing a statement, follow the statement with "/"
      if ($FilePath !~ /^@/) {
        print {$catcon_FileHandles[$CurProc]} "/\n";
      }

      # if we started spooling before running the script, turn it off after 
      # it is done
      if ($SpoolStmt) {
        if ($catcon_DebugOn) {
          log_msg("catconExec: sending $SpoolStmt to process $CurProc\n");
        }

        printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                       $SpoolStmt, "", $catcon_DebugOn);
      }

      if (!$catcon_CatShutdown{$rootOrNonCdb})
      {
        # @@@@ Added more info
        print {$catcon_FileHandles[$CurProc]} qq#select '==== $FilePathToDisplay' || ' Container:' || SYS_CONTEXT('USERENV','CON_NAME') || ' Id:' || SYS_CONTEXT('USERENV','CON_ID') || ' ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') || ' Proc:$CurProc ====' AS end_running from sys.dual;\n/\n#;

      }

      if ($FilePath =~ /^@/ && $catcon_UncommittedXactCheck) {
        # if asked to check whether a script has ended with uncommitted 
        # transaction, do so now
        printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
               qq#SELECT decode(COUNT(*), 0, 'OK', 'Script ended with uncommitted transaction') AS uncommitted_transaction_check FROM v\$transaction t, v\$session s, v\$mystat m WHERE t.ses_addr = s.saddr AND s.sid = m.sid AND ROWNUM = 1#, 
               "\n/\n", $catcon_DebugOn);
      }

      # unless we are running single-threaded, we need a "done" file to be 
      # created after the current script or statement completes so that 
      # next_proc() would recognize that this process is available and 
      # consider it when picking the next process to run a script or SQL 
      # statement
      if (!$SingleThreaded) {
        # file which will indicate that process $CurProc finished its work and 
        # is ready for more
        #
        # Bug 18011217: append _catcon_$catcon_ProcIds[$CurProc] to 
        # $catcon_LogFilePathBase to avoid conflicts with other catcon 
        # processes running on the same host
        my $DoneFile = 
          done_file_name($catcon_LogFilePathBase, $catcon_ProcIds[$CurProc]);

        # if catcon_DebugOn is set, printToSqlplus will normally issue 
        #   select "catconExec() ... as catcon_statement from dual"
        # which we need to avoid if we are running catshutdown
        printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                       qq/$catcon_DoneCmd $DoneFile/, "\n", 
                       $catcon_DebugOn && !$catcon_CatShutdown{$rootOrNonCdb});
      
        if ($catcon_DebugOn) {
          log_msg <<msg;
catconExec: sent "$catcon_DoneCmd $DoneFile" to process 
    $CurProc (id = $catcon_ProcIds[$CurProc]) to indicate its availability 
    after completing $FilePath
msg
        }
      }

      $catcon_FileHandles[$CurProc]->flush;
    }

    # if we are running single-threaded, we need a "done" file to be 
    # created after the last script or statement sent to the current Container 
    # completes so that next_proc() would recognize that this process is 
    # available and consider it when picking the next process to run a script 
    # or SQL statement
    if ($SingleThreaded) {
      # file which will indicate that process $CurProc finished its work and 
      # is ready for more
      #
      # Bug 18011217: append _catcon_$catcon_ProcIds[$CurProc] to 
      # $catcon_LogFilePathBase to avoid conflicts with other catcon 
      # processes running on the same host
      my $DoneFile = 
        done_file_name($catcon_LogFilePathBase, $catcon_ProcIds[$CurProc]);

      # if catcon_DebugOn is set, printToSqlplus will normally issue 
      #   "select 'catconExec() ... as catcon_statement from dual"
      # which we need to avoid if the last script we ran was catshutdown
      printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                     qq/$catcon_DoneCmd $DoneFile/, "\n", 
                     $catcon_DebugOn && 
                     index($ScriptPaths[$#ScriptPaths], "catshutdown") == -1);

      # flush the file so a subsequent test for file existence does 
      # not fail due to buffering
      $catcon_FileHandles[$CurProc]->flush;
      
      if ($catcon_DebugOn) {
        log_msg <<msg;
catconExec: sent "$catcon_DoneCmd $DoneFile" to process 
    $CurProc (id = $catcon_ProcIds[$CurProc]) to indicate its availability
msg
      }
    }
  
    # if 
    #   - there are no additional Containers in which we need to run scripts 
    #     and/or SQL statements and 
    #   - the user told us to issue per-process init and completion statements 
    #     and 
    #   - there are such statements to send, 
    # they need to be passed to wait_for_completion
    my $EndStmts;

    if (   ($RootOnly || !(@Containers && $#Containers > 0))
           && $IssuePerProcStmts) {
      $EndStmts = $catcon_PerProcEndStmts;
    } else {
      $EndStmts = undef;
    }

    # wait_for_completion uses calls printToSqlplus to send "done" command to 
    # processes and if catcon_DebugOn is set, printToSqlplus will normally 
    # issue 
    #   "select 'catconExec() ... as catcon_statement from dual"
    # which we need to avoid if the last script we ran was catshutdown
    if (wait_for_completion($ProcsUsed, $catcon_NumProcesses, 
          $catcon_LogFilePathBase, 
          @catcon_FileHandles, @catcon_ProcIds,
          $catcon_DoneCmd, @$EndStmts, 
          $CDBorFedRoot ? $CDBorFedRoot : undef,
          @catcon_InternalConnectString,
          $catcon_RecoverFromChildDeath,
          $catcon_KillAllSessScript, 
          index($ScriptPaths[$#ScriptPaths], "catshutdown") != -1,
          $catcon_DebugOn) == 1) {
      # unexpected error was encountered, return
      log_msg("catconExec: unexpected error in wait_for_completions\n");

      return 1;
    }

    # will jump to this label if scripts/stataments need to be run against one
    # or more PDBs, but not against Root or Federation Root
skipSingleContainerRun:

    # if 
    # - running against a Consolidated DB and 
    # - caller has not instructed us to run ONLY in the Root, 
    # - the list of Containers against which we need to run includes at 
    #   least one PDB (which can be establsihed by verifying that either 
    #   @Containers contains more than one element or the first
    #   element is not CDB$ROOT or the explicitly specified Federation Root 
    #   ($CDBorFedRoot would be set to $catcon_Root unless $catcon_FedRoot was 
    #   set, so comparing first element to $CDBorFedRoot with cover both cases)
    # run all scripts/statements against all remaining Containers (i.e. PDBs, 
    # since we already took care of the Root or Federation Root, unless the 
    # user instructed us to skip it)    
    if (   @Containers 
        && !$RootOnly 
        && (   $#Containers > 0 
            || ($Containers[0] ne $CDBorFedRoot)))
    {
      # A user may have excluded the Root (and possibly some other PDBs, but 
      # the important part is the Root) from the list of Containers against 
      # which to run scripts, in which case $Containers[0] would 
      # contain name of a PDB, and we would have skipped the single-container 
      # part of this subroutine.  
      #
      # It is important that we keep it in mind when deciding across how many 
      # Containers we need to run and whether to skip the Container whose 
      # name is stored in $Containers[0].

      # offset into @Containers of the first PDB name
      my $firstPDB = ($Containers[0] eq $CDBorFedRoot) ? 1 : 0;

      # number of PDB names in @Containers
      my $numPDBs = $#Containers + 1 - $firstPDB;

      # offset into @Containers of the first user (i.e. not PDB$SEED) PDB name;
      # subsequent code may test for equality of $firstPDB and $firstUserPDB to
      # determine if @Containers has PDB$SEED as it's first PDB name
      my $firstUserPDB = 
        ($Containers[$firstPDB] eq q/PDB$SEED/) ? $firstPDB + 1 : $firstPDB;
      
      if ($catcon_DebugOn || $catcon_Verbose) {
        log_msg("catconExec: run all scripts/statements against remaining ".
                "$numPDBs PDBs\n");
      }

      # compute number of processes which will be used to run 
      # script/statements specified by the caller in all remaining PDBs; 
      # used to determine when all processes finished their work
      $ProcsUsed = 
        compute_procs_used_for_pdbs($SingleThreaded, $numPDBs, 
                                    $catcon_NumProcesses, $#ScriptPaths+1);

      # one of the PDBs against which we will run scripts/SQL statements may be
      # PDB$SEED; before we attempt to run any statements against it,
      # we need to make sure it is open in the mode specified by the caller via
      # catconInit.SeedMode parameter.  
      #
      # Since PDB$SEED has con_id of 2, it will be the first PDB in the list.
      #
      # NOTE: as a part of fix for bug 13072385, I moved code to revert 
      #       PDB$SEED mode back to its original mode into catconWrapUp(), 
      #       but I am leaving code that changes its mode here in 
      #       catconExec() (rather than moving it into catconInit()) because 
      #       the caller may request that scripts/statements be run against 
      #       PDB$SEED even if it was not one of the Containers specified 
      #       when calling catconInit().  Because of that possibility, I have 
      #       to keep this code here, so I see no benefit from also adding it 
      #       to catconInit().

      # should _oracle_script be set when opening/closing PDBs (because one 
      # of the PDBs being affected is PDB$SEED)
      my $setOracleScript = 0;

      if ($firstPDB < $firstUserPDB) {
        if ($catcon_SeedPdbMode != CATCON_PDB_MODE_UNCHANGED) {
          if ($catcon_DebugOn) {
            log_msg("catconExec: invoke force_pdb_modes to reopen ".
                    "(if necessary) PDB\$SEED in ".
                    PDB_MODE_CONST_TO_STRING->{$catcon_SeedPdbMode}." mode\n");
          }

          # remember to set _oracle_script before opening/closing PDB$SEED
          $setOracleScript = 1;

          my ($ret, $instPdbHash_ref) = 
            force_pdb_modes(@Containers, $firstPDB, $firstPDB, 
                            $catcon_SeedPdbMode, \%catcon_RevertSeedPdbMode, 
                            $catcon_DebugOn, $catcon_Verbose, 
                            $catcon_ProcIds[0]."_seed_pdb", 
                            $catcon_LogFilePathBase, 
                            @catcon_InternalConnectString, 
                            $catcon_DoneCmd, $setOracleScript, 
                            %catcon_PdbsOpenInReqMode, 
                            %catcon_IsConOpen, $catcon_DbmsVersion,
                            $firstUserPDB <= $#Containers,
                            \%catcon_InstProcMap, \%catcon_InstConnStr);
          if ($catcon_DebugOn) {
            log_msg("catconExec: force_pdb_modes returned\n");
          }

          if ($ret == 1) {
            log_msg("catconExec: unexpected error in force_pdb_modes when ".
                    "trying to reopen PDB\$SEED\n");
            return 1;
          } elsif ($ret == 0 && $instPdbHash_ref) {
            # save reference to the hash mapping instance names to PDBs open 
            # on those instances
            $catcon_ForcePdbModeInstSeedPdbMap = $instPdbHash_ref;
            
            if ($catcon_DebugOn) {
              if (%$instPdbHash_ref) {
                log_msg("catconExec: PDB\$SEED was opened in ".
                        PDB_MODE_CONST_TO_STRING->{$catcon_SeedPdbMode}.
                        " mode\n");
              } else {
                log_msg("catconExec: PDB\$SEED was left closed and will be ".
                        "opened in ".
                        PDB_MODE_CONST_TO_STRING->{$catcon_SeedPdbMode}." ".
                        "mode together with user PDBs\n");
              }
            }

            # NOTE: If 
            #       - we are going to use all available instances, and 
            #       - force_pdb_modes discovered that the CDB is open on more 
            #         than one instance, and 
            #       - we told force_pdb_modes that there are PDBs besides 
            #         PDB$SEED that will need to be processed, 
            #       it closed PDB$SEED but did not reopen it (and communicated 
            #       this to us by undefining %$instPdbHash_ref), in which case 
            #       it will fall upon the invocation of force_pdb_modes that 
            #       will handle user PDBs (below) to open PDB$SEED; otherwise, 
            #       there is no reason to set it during that invocation, which 
            #       $is why we clear setOracleScript below if 
            #       %$instPdbHash_ref is defined
            #       (pardon the lengthy explanation)
            if (%$instPdbHash_ref) {
              $setOracleScript = 0;
            }
          } elsif ($ret == -1) {
            if ($catcon_DebugOn) {
              log_msg("catconExec: PDB\$SEED has been previously opened in ".
                      PDB_MODE_CONST_TO_STRING->{$catcon_SeedPdbMode}.
                      " mode\n");
            }

            # PDB$SEED will not need to be opened by the next invocation of 
            # force_pdb_modes, so it will not need to set _ORACLE_SCRIPT
            $setOracleScript = 0;
          }
        } elsif ($catcon_DebugOn) {
          log_msg("catconExec: PDB\$SEED mode will be left unchanged, per ".
                  "user directive\n");
        }
      }

      # do the same for any user PDBs that will be operated upon
      if ($firstUserPDB <= $#Containers) {
        if ($catcon_AllPdbMode != CATCON_PDB_MODE_UNCHANGED) {
          if ($catcon_DebugOn) {
            log_msg("catconExec: invoke force_pdb_modes to reopen ".
                    "(if necessary) user PDBs in ".
                    PDB_MODE_CONST_TO_STRING->{$catcon_AllPdbMode}." mode\n");
          }

          my ($ret, $instPdbHash_ref) = 
            force_pdb_modes(@Containers, $firstUserPDB, $#Containers, 
                            $catcon_AllPdbMode, \%catcon_RevertUserPdbModes, 
                            $catcon_DebugOn, $catcon_Verbose, 
                            $catcon_ProcIds[0]."_user_pdbs", 
                            $catcon_LogFilePathBase, 
                            @catcon_InternalConnectString, 
                            $catcon_DoneCmd, $setOracleScript, 
                            %catcon_PdbsOpenInReqMode, 
                            %catcon_IsConOpen, $catcon_DbmsVersion, 0,
                            \%catcon_InstProcMap, \%catcon_InstConnStr);
          if ($catcon_DebugOn) {
            log_msg("catconExec: force_pdb_modes returned\n");
          }

          if ($ret == 1) {
            log_msg("catconExec: unexpected error in force_pdb_modes when ".
                    "trying to reopen user PDBs\n");
            return 1;
          } elsif ($ret == 0 && $instPdbHash_ref) {
            # save references to the hash mapping instance names to PDBs open 
            # on those instances
            $catcon_ForcePdbModeInstUserPdbMap = $instPdbHash_ref;
            
            if ($catcon_DebugOn) {
              log_msg("catconExec: user PDBs were opened in ".
                      PDB_MODE_CONST_TO_STRING->{$catcon_AllPdbMode}.
                      " mode\n");
            }
          } elsif ($ret == -1 && $catcon_DebugOn) {
            log_msg("catconExec: user PDBs  have been previously opened in ".
                    PDB_MODE_CONST_TO_STRING->{$catcon_SeedPdbMode}." mode\n");
          }
        } elsif ($catcon_DebugOn) {
          log_msg("catconExec: user PDB(s) mode will be left unchanged, per ".
                  "user directive\n");
        }
      }

      # Bug 20193612: if the caller requested that we utilize all available 
      # instances and $catcon_ForcePdbModeInstSeedPdbMap OR 
      # $catcon_ForcePdbModeInstUserPdbMap (not both, mind you, since only one 
      # of them will be set if we were asked to use all available instances)
      # indicate that the PDBs on which we will be working are open on more 
      # than 1 instance, we will fork one process per instance, passing to 
      # them names of PDBs open on that instance and ids of processes which 
      # were allocated to that instance (based on value of cpu_count obtained 
      # from that instance)
      #
      # NOTE: the fact that only one of $catcon_ForcePdbModeInstSeedPdbMap and 
      #       $catcon_ForcePdbModeInstUserPdbMap is set is not enough to 
      #       conclude that we we were asked to use all available instances.  
      #       Such situation can also arise if the caller asked to leave user 
      #       PDBs alone, for instance, or if we were called to process only 
      #       PDB$SEED (or only user PDBs)
      
      if ($catcon_AllInst &&
          (   $catcon_ForcePdbModeInstSeedPdbMap
           || $catcon_ForcePdbModeInstUserPdbMap)) {
        # this hash will be used to store contents of a hash (either 
        # %$catcon_ForcePdbModeInstSeedPdbMap or 
        # %$catcon_ForcePdbModeInstUserPdbMap) describing PDBs and instances 
        # on which they are open and where they should be processed
        my %instPdbMap;

        # if only one of the two maps is defined, use it; if both are set, 
        # force_pdb_modes() must have decided that the PDBs should be all 
        # processed using the current catcon process (e.g. because that's the 
        # only instance on which the CDB is open)
        if (!$catcon_ForcePdbModeInstUserPdbMap) {
          if ($catcon_DebugOn) {
            log_msg("catconExec: processing only PDB\$SEED\n\n");
            inst_pdbs_hash_dump($catcon_ForcePdbModeInstSeedPdbMap, 
                                "catconExec", 
                                "catcon_ForcePdbModeInstSeedPdbMap");
          }

          %instPdbMap = %$catcon_ForcePdbModeInstSeedPdbMap;
        } elsif (!$catcon_ForcePdbModeInstSeedPdbMap ||
                 !%$catcon_ForcePdbModeInstSeedPdbMap) {
          # NOTE: if processing PDB$SEED and user PDBs using multiple 
          #   instances, %$catcon_ForcePdbModeInstSeedPdbMap will be undefined 
          #   while %$catcon_ForcePdbModeInstUserPdbMap will map PDB$SEED as 
          #   well as user PDBs to various instances
          if ($catcon_DebugOn) {
            if ($firstPDB < $firstUserPDB) {
              log_msg("catconExec: processing PDB\$SEED and user PDBs\n\n");
            } else {
              log_msg("catconExec: processing only user PDBs\n\n");
            }

            inst_pdbs_hash_dump($catcon_ForcePdbModeInstUserPdbMap, 
                                "catconExec", 
                                "catcon_ForcePdbModeInstUserPdbMap");
          }

          %instPdbMap = %$catcon_ForcePdbModeInstUserPdbMap;
        } else {
          # both are set; this means that we will be processing all PDBs using 
          # this process; there is no need to try and merge 
          # %$catcon_ForcePdbModeInstSeedPdbMap and 
          # %$catcon_ForcePdbModeInstUserPdbMap
          if ($catcon_DebugOn) {
            log_msg <<msg;
catconExec: the caller requested that we use all instances on which the CDB is
    open, but we will be processing PDBs using the current catcon process 
    because the CDB is open on only one instance
msg
          }
        }

        if ($catcon_DebugOn) {
          inst_pdbs_hash_dump(\%instPdbMap, "catconExec", "instPdbMap");
        }

        if (!%instPdbMap || keys %instPdbMap == 1) {
          # all PDBs will be processed using SQL*Plus processes connected to 
          # one instance, either because we were not told to use all available 
          # instances (!%instPdbMap) or because there is only 1 instance, so 
          # there is nothing to be gained from forking a child process
          if ($catcon_DebugOn) {
            log_msg("catconExec: all PDBs will be processed on the current ".
                    "instance and\n".
                    "will be handled by this process (".$$.")\n");
          }

          goto runScriptsInPDBs;
        }

        # proc ids of forked processes
        my %forkedProcIds;

        foreach my $pdbInst (keys %instPdbMap) {
          # store instance-specific info in variables which will be used to 
          # communicate it to forked processes and fork one process per 
          # instance on which one or more PDBs are open

          # PDBs open on $pdbInst (child's @Containers)
          my @childContainers = @{$instPdbMap{$pdbInst}};

          if ($catcon_DebugOn) {
            log_msg("catconExec: process ".$$." preparing to fork off a ".
                    "process to handle PDBs (".join(', ', @childContainers).
                    ") open on instance $pdbInst\n");
          }

          if ((! exists $catcon_InstProcMap{$pdbInst}->{NUM_PROCS}) ||
              !$catcon_InstProcMap{$pdbInst}->{NUM_PROCS}) {
            # this should never happen because force_pdb_modes should not be 
            # assigning PDBs to instances to which no sqlplus processes can 
            # be allocated
            log_msg("catconExec: PDBs ".join(', ', @childContainers)." ".
                    "were assigned to instance $pdbInst even though no ".
                    "sqlplus processes could be allocated to it\n");
            return 1;
          }

          my $childCatconNumProcesses = 
            $catcon_InstProcMap{$pdbInst}->{NUM_PROCS};

          if (! exists $catcon_InstProcMap{$pdbInst}->{FIRST_PROC}) {
            # this should never happen because force_pdb_modes should not be 
            # assigning PDBs to instances to which no sqlplus processes were 
            # allocated
            log_msg("catconExec: PDBs ".join(', ', @childContainers)." ".
                    "were assigned to instance $pdbInst even though ".
                    "the offset of the first sqlplus process allocated to it ".
                    "is not known\n");
            return 1;
          }

          # offset into @catcon_ProcIds of the first process allocated to 
          # processing PDBs open on this instance
          my $firstSqlplusProc = 
            $catcon_InstProcMap{$pdbInst}->{FIRST_PROC};

          # offset into @catcon_ProcIds of the last process allocated to 
          # processing PDBs open on this instance
          my $lastSqlplusProc = 
            $firstSqlplusProc + $childCatconNumProcesses - 1;

          my @childCatconProcIds = 
            @catcon_ProcIds[$firstSqlplusProc..$lastSqlplusProc];

          if ($catcon_DebugOn) {
            log_msg("catconExec: process to handle PDBs open on instance ".
                    "$pdbInst will use $childCatconNumProcesses processes, ".
                    "$firstSqlplusProc to $lastSqlplusProc (".
                    join(', ', @childCatconProcIds).")\n");
          }

          my @childCatconFileHandles = 
            @catcon_FileHandles[$firstSqlplusProc..$lastSqlplusProc];
          
          my $pid = fork();
          
          if (! defined $pid) {
            log_msg("catconExec: failed to fork a process to handle ".
                    "processing PDBs open on instance $pdbInst\n");
            return 1;
          }

          if ($pid) {
            # parent process

            # remember that we forked this process
            $forkedProcIds{$pid} = undef;
            
            log_msg("catconExec: process $pid was forked to process PDBs ".
                    "open on instance $pdbInst\n");
            log_msg("catconExec: output produced by this child process will ".
                    "be written to [".
                    $catcon_LogFilePathBase."_catcon_".$pid.".lst]\n");
          } else {
            # child process; use instance-specific data passed by the parent 
            # to overwrite certain variables and proceed with processing PDBs 
            # open on a given instance

            # remember that we are running in a child process
            $catcon_ChildProc = 1;

            # name of the file where output generated by this 
            # child process will be written
            my $childProcLog = 
              child_log_file_name($catcon_LogFilePathBase, $$);

            # open $CATCONOUT for this child process.  
            if (!open($CATCONOUT, ">>", $childProcLog)) {
              print STDERR "catconExec: unable to open ".$childProcLog." ".
                " as CATCONOUT for child process ".$$."\n";
              return 1;
            }

            # make $CATCONOUT "hot" so diagnostic and error message output 
            # does not get buffered
            select((select($CATCONOUT), $|=1)[0]);

            # PDBs opened on the instance assigned to this child process
            @Containers = @childContainers;
            $firstPDB = 0;

            # number of PDBs open on this instance
            $numPDBs = $#Containers + 1;

            # number of SQL*Plus processes allocated or PDBs open on this 
            # instance
            $catcon_NumProcesses = $childCatconNumProcesses;

            $ProcsUsed = 
              compute_procs_used_for_pdbs($SingleThreaded, $numPDBs, 
                                          $catcon_NumProcesses, 
                                          $#ScriptPaths+1);

            @catcon_ProcIds = @childCatconProcIds;

            @catcon_FileHandles = @childCatconFileHandles;

            goto runScriptsInPDBs;
          }
        }

        # in parent process

        if ($catcon_DebugOn) {
          log_msg("catconExec: having forked ".
                  (scalar keys %forkedProcIds)." ".
                  "child processes, wait for them to return\n");
        }

        # parent return value will be set to 0 if all child processes return 0
        # inside their done files; otherwise it will be set to 1 to indicate 
        # a potential problem
        my $parentRetVal = 0;

        while (keys %forkedProcIds) {
          foreach my $childPid (keys %forkedProcIds) {
            
            # check if the child process' "done" file exists and contains the 
            # return value
            my $childDoneFile = 
              child_done_file_name($catcon_LogFilePathBase, $childPid);

            if ((-e $childDoneFile) && (-s $childDoneFile)) {
              delete $forkedProcIds{$childPid};
              
              if ($catcon_DebugOn) {
                log_msg("catconExec: attempt to open $childDoneFile to ".
                        "determine value returned by child process ".
                        "$childPid\n");
              }
              
              # open the "done" file and extract the return value
              my $childFH;
              if (!open($childFH, '<', $childDoneFile)) {
                log_msg("catconExec: child process done file $childDoneFile ".
                        "could not be opened;\n". 
                        "child process appears to have completed, though\n");
      
                log_msg("catconExec: parent process will report error to the ".
                        "caller\n");
                $parentRetVal = 1;
              } else {
                my @lines = <$childFH>;

                if ($catcon_DebugOn) {
                  log_msg("catconExec: child process done file contents:\n");
                  for (@lines) {
                    log_msg("\t".$_."\n");
                  }
                }

                if ($#lines == 0) {
                  # get rid of \n
                  chomp($lines[0]);

                  log_msg("catconExec: child process $childPid exited with ".
                          "return value ". $lines[0]."\n");
      
                  my $childRetVal = $lines[0];

                  if ($childRetVal != 0) {
                    log_msg("catconExec: since child return value (".
                            $childRetVal.") did not indicate success, parent ".
                            "process will report error to the caller\n");
                    $parentRetVal = 1;
                  }
                } else {
                  log_msg("catconExec: done file has an unexpected number of ".
                          "lines: ".($#lines+1)."\n");
                  for (@lines) {
                    log_msg("\t".$_."\n");
                  }
                  
                  log_msg("catconExec: parent process will report error to ".
                          "the caller\n");
                  $parentRetVal = 1;
                }

                # since we were able to open and examine child process' done 
                # file, delete it
                sureunlink($childDoneFile, $catcon_DebugOn);
              }
            }
          }
        }

        # all processing has been done by child processes
        if ($catcon_DebugOn) {
          log_msg("catconExec: all child processes have completed.\n".
                  "\tReturning $parentRetVal...\n");
        }

        return $parentRetVal;
      } elsif ($catcon_DebugOn) {
        # either the user did not request that we use all available instances,
        # or he specified that all PDBs are to be left in the mode in which 
        # they are open (which causes us to ignore a request to use all 
        # available instances), and so we will process all PDBs using the 
        # current catcon process
        if (!$catcon_AllInst) {
          log_msg("catconExec: user did not request that we use all ".
                  "available instances\n");
        } else {
          log_msg("catconExec: neither catcon_ForcePdbModeInstSeedPdbMap ".
                  "nor\n\tcatcon_ForcePdbModeInstUserPdbMap is set\n");
        }
        
        log_msg("\twill run scripts using the current catcon process\n");
      }

runScriptsInPDBs:

      # offset into the array of remaining Container names
      my $CurCon;

      # set it to -1 so the first time next_proc is invoked, it will start by 
      # examining status of process 0
      $CurProc = -1;

      # NOTE: $firstPDB contains offset of the first PDB in the list, but 
      #       $#Containers still represents the offset of the last PDB
      for ($CurCon = $firstPDB; $CurCon <= $#Containers; $CurCon++) {

        # if there is a query to run to see if the scripts should be run
        # in this PDB, run the query and exit if it returns NULL

        if ($$skipProcQuery) {
            my @result = catconQuery($$skipProcQuery, $Containers[$CurCon]);
            if ($result[0] eq "") {
                next;   # do not run any of the scripts in this PDB
            }
        }

        # if we were told to run single-threaded, switch into the right 
        # Container ; all scripts and SQL statement to be executed in this
        # Container will be sent to the same process
        if ($SingleThreaded) {
          # as we are starting working on a new Container, find next 
          # available process
          $CurProc = pickNextProc($ProcsUsed, $catcon_NumProcesses, 
                                  $CurProc + 1,
                                  $catcon_LogFilePathBase, \@catcon_ProcIds, 
                                  $catcon_RecoverFromChildDeath,
                                  @catcon_InternalConnectString, 
                                  $catcon_KillAllSessScript,
                                  $catcon_DoneCmd, $catcon_DebugOn);
          
          if ($CurProc < 0) {
            # some unexpected error was encountered
            log_msg("catconExec: unexpected error in pickNextProc\n");

            return 1;
          }

          additionalInitStmts(\@catcon_FileHandles, $CurProc, 
                              $Containers[$CurCon], 
                              $CurrentContainerQuery, 
                              $catcon_ProcIds[$CurProc],
                              $catcon_EchoOn, $catcon_DebugOn, 
                              $catcon_UserScript, $catcon_FedRoot);
        }

        # determine prefix of ERRORLOGGING identifier, if needed
        if ($catcon_ErrLogging) {
          # do not mess with $ErrLoggingIdent if $$CustomErrLoggingIdent is set
          # since once the user provides a value for the Errorlogging 
          # Identifier, he is fully in charge of its value
          # do not mess with $ErrLoggingIdent if $$CustomErrLoggingIdent is set
          # since once the user provides a value for the Errorlogging 
          # Identifier, he is fully in charge of its value
          # 
          # Truth table describing how we determine whether to set the 
          # ERRORLOGGING Identifier, and if so, whether to use a default 
          # identifier or a custom identifier
          # N\C 0   1
          #   \--------
          #  0| d | c |
          #   ---------
          #  1| - | c |
          #   ---------
          #
          # N = is $catcon_NoErrLogIdent set?
          # C = is $$CustomErrLoggingIdent defined?
          # d = use default ERRORLOGGING Identifier
          # c = use custom ERRORLOGGING Identifier
          # - = do not setuse ERRORLOGGING Identifier
          #
          # NOTE: it's important that we we assign $$CustomErrLoggingIdent to 
          #       $ErrLoggingIdent and test the result BEFORE we test 
          #       $catcon_NoErrLogIdent because  if $$CustomErrLoggingIdent is 
          #       $defined, we will ignore catcon_NoErrLogIdent, so it's 
          #       necessary that $ErrLoggingIdent gets set to 
          #       $$CustomErrLoggingIdent regardless of whether 
          #       $catcon_NoErrLogIdent is set
          if (!($ErrLoggingIdent = $$CustomErrLoggingIdent) && 
              !$catcon_NoErrLogIdent) {
            $ErrLoggingIdent = "{$Containers[$CurCon]}::";
          }

          # replace single and double quotes in $ErrLoggingIdent with # to 
          # avoid confusion
          if ($ErrLoggingIdent && $ErrLoggingIdent ne "") {
            $ErrLoggingIdent =~ s/['"]/#/g;
          }

          if ($catcon_DebugOn) {
            log_msg("catconExec: ErrLoggingIdent prefix = $ErrLoggingIdent\n");
          }
        }

        my $ScriptPathDispIdx = -1;

        $currCatUpgrd = 0;
    
        foreach my $FilePath (@ScriptPaths) {
          # remember if the last script ran was catshutdown
          $prevCatShutdown = $catcon_CatShutdown{$Containers[$CurCon]};

          # remember if the current script is catshutdown or catupgrd
          if (index($FilePath, "catshutdown") != -1) {
            $catcon_CatShutdown{$Containers[$CurCon]} = 1;
            $currCatUpgrd = 0;
          } elsif (index($FilePath, "catupgrd") != -1) {
            $currCatUpgrd = 1;
            $catcon_CatShutdown{$Containers[$CurCon]} = 0;
          } else {
            $currCatUpgrd = $catcon_CatShutdown{$Containers[$CurCon]} = 0;
          }

          # switch into the right Container if we were told to run 
          # multi-threaded; each script or SQL statement may be executed 
          # using a different process
          if (!$SingleThreaded) {
            # as we are starting working on a new script or SQL statement, 
            # find next available process
            $CurProc = pickNextProc($ProcsUsed, $catcon_NumProcesses, 
                                    $CurProc + 1,
                                    $catcon_LogFilePathBase, \@catcon_ProcIds, 
                                    $catcon_RecoverFromChildDeath,
                                    @catcon_InternalConnectString, 
                                    $catcon_KillAllSessScript,
                                    $catcon_DoneCmd, $catcon_DebugOn);

            if ($CurProc < 0) {
              # some unexpected error was encountered
              log_msg("catconExec: unexpected error in pickNextProc\n");

              return 1;
            }

            additionalInitStmts(\@catcon_FileHandles, $CurProc, 
                                $Containers[$CurCon], 
                                $CurrentContainerQuery, 
                                $catcon_ProcIds[$CurProc],
                                $catcon_EchoOn, $catcon_DebugOn,
                                $catcon_UserScript, $catcon_FedRoot);
          }

          # if this is the first time we are using this process and 
          # the caller indicated that one or more statements need to be 
          # executed before using a process for the first time, issue these 
          # statements now
          if ($#NeedInitStmts >= 0 && $NeedInitStmts[$CurProc]) {
            firstProcUseStmts($ErrLoggingIdent, $$CustomErrLoggingIdent,
                              $catcon_ErrLogging, 
                              \@catcon_FileHandles, $CurProc, 
                              $catcon_PerProcInitStmts, \@NeedInitStmts, 
                              $catcon_UserScript, $catcon_FedRoot, 
                              $catcon_DebugOn);
          }

          print {$catcon_FileHandles[$CurProc]} qq#select '==== CATCON EXEC IN CONTAINERS ====' AS catconsection from dual\n/\n#;

          # element of @ScriptPathsToDisplay which corresponds to $FilePath
          my $FilePathToDisplay = $ScriptPathsToDisplay[++$ScriptPathDispIdx];

          # bracket script or statement with strings intended to make it 
          # easier to determine origin of output in the log file

          # @@@@ Added more info
          print {$catcon_FileHandles[$CurProc]} qq#select '==== $FilePathToDisplay' || ' Container:' || SYS_CONTEXT('USERENV','CON_NAME') || ' Id:' || SYS_CONTEXT('USERENV','CON_ID') || ' ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') || ' Proc:$CurProc ====' AS begin_running from sys.dual;\n/\n#;



          if ($catcon_ErrLogging) {
            # construct and issue SET ERRORLOGGING ON [TABLE...] statement
            # @@@@
            if ($bInitStmts)
            {
                err_logging_tbl_stmt($catcon_ErrLogging, \@catcon_FileHandles, 
                                     $CurProc, $catcon_UserScript, 
                                     $catcon_FedRoot, $catcon_DebugOn);
            }

            if ($ErrLoggingIdent) {
              # send SET ERRORLOGGING ON IDENTIFIER ... statement

              my $Stmt;

              if ($$CustomErrLoggingIdent) {
                # NOTE: if the caller of catconExec() has supplied a custom 
                #       Errorlogging Identifier, it has already been copied 
                #       into $ErrLoggingIdent which was then normalized, so 
                #       our use of $ErrLoggingIdent rather than 
                #       $CustomErrLoggingIdent below is intentional
                $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent, 0, 256)."'";
              } else {
                $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent.$FilePathToDisplay, 0, 256)."'";
              }

              if ($catcon_DebugOn || $catcon_Verbose) {
                log_msg("catconExec: sending $Stmt to process $CurProc\n");
              }

              printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                             $Stmt, "\n", $catcon_DebugOn);
            }
          }

          # statement to start/end spooling output of a SQL*Plus script
          my $SpoolStmt;

          # if the caller requested that we generate a spool file and we are 
          # about to execute a script, issue SPOOL ... REPLACE
          if ($catcon_SpoolOn && $SpoolFileNames[$ScriptPathDispIdx] ne "") {
            my $SpoolFileNameSuffix = 
              getSpoolFileNameSuffix($Containers[$CurCon]);

            $SpoolStmt = "SPOOL '" . $SpoolFileNames[$ScriptPathDispIdx] . $SpoolFileNameSuffix . "' REPLACE\n";
            
            if ($catcon_DebugOn) {
              log_msg("catconExec: sending $SpoolStmt to process $CurProc\n");
            }

            printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                           $SpoolStmt, "", $catcon_DebugOn);

            # remember that after we execute the script, we need to turn off 
            # spooling
            $SpoolStmt = "SPOOL OFF \n";
          }

          # if executing a query, will be set to the text of the query; 
          # otherwise, will be set to the name of the script with parameters, 
          # if any
          my $AppInfoAction;

          my $LastSlash; # position of last / or \ in the script name
      
          if ($FilePath !~ /^@/) {
            $AppInfoAction = $FilePath;
          } elsif (($LastSlash = rindex($FilePath, '/')) >= 0 ||
                   ($LastSlash = rindex($FilePath, '\\')) >= 0) {
            $AppInfoAction = substr($FilePath, $LastSlash + 1);
          } else {
            # FilePath contains neither backward nor forward slashes, so use 
            # it as is
            $AppInfoAction = $FilePath;
          }

          # $AppInfoAction may include parameters passed to the script. 
          # These parameters will be surrounded with single quotes, which will 
          # cause a problem since $AppInfoAction is used to construct a 
          # string parameter being used with 
          # ALTER SESSION SET APPLICATION ACTION.
          # To prevent this from happening, we will replace single quotes found
          #in $AppInfoAction with #
          $AppInfoAction =~ s/[']/#/g;

          # use ALTER SESSION SET APPLICATION MODULE/ACTION to identify 
          # process, Container and script or statement being run
          if (!$currCatUpgrd && !$catcon_CatShutdown{$Containers[$CurCon]}) {
            printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                           "ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'",
                           "\n/\n", $catcon_DebugOn);
          }
          if ($catcon_DebugOn) {
            log_msg("catconExec: issued ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n");
          }

          if (!$currCatUpgrd && !$catcon_CatShutdown{$Containers[$CurCon]}) {
            printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                           "ALTER SESSION SET APPLICATION ACTION = '$Containers[$CurCon]::$AppInfoAction'",
                           "\n/\n", $catcon_DebugOn);
          }

          if ($catcon_DebugOn) {
            log_msg("catconExec: issued ALTER SESSION SET APPLICATION ACTION = '$Containers[$CurCon]::$AppInfoAction'\n");
          }

          # execute next script or SQL statement

          if ($catcon_DebugOn || $catcon_Verbose) {
            log_script_execution($FilePathToDisplay, $Containers[$CurCon], 
                                 $CurProc);
          }

          # if catcon_DebugOn is set, printToSqlplus will normally issue 
          #   select "catconExec() ... as catcon_statement from dual"
          # which we need to avoid if we just finished running  catshutdown
          printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                         $FilePath, "\n", 
                         $catcon_DebugOn && !$prevCatShutdown);

          # if executing a statement, follow the statement with "/"
          if ($FilePath !~ /^@/) {
            print {$catcon_FileHandles[$CurProc]} "/\n";
          }

          # if we started spooling before running the script, turn it off after
          # it is done
          if ($SpoolStmt) {
            if ($catcon_DebugOn) {
              log_msg("catconExec: sending $SpoolStmt to process $CurProc\n");
            }
            
            printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                           $SpoolStmt, "", $catcon_DebugOn);
          }

          if (!$catcon_CatShutdown{$Containers[$CurCon]})
          {
            # @@@@ Added more info
            print {$catcon_FileHandles[$CurProc]} qq#select '==== $FilePathToDisplay' || ' Container:' || SYS_CONTEXT('USERENV','CON_NAME') || ' Id:' || SYS_CONTEXT('USERENV','CON_ID') || ' ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') || ' Proc:$CurProc ====' AS end_running from sys.dual;\n/\n#;
          }

          if ($FilePath =~ /^@/ && $catcon_UncommittedXactCheck) {
            # if asked to check whether a script has ended with uncommitted 
            # transaction, do so now
            printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
               qq#SELECT decode(COUNT(*), 0, 'OK', 'Script ended with uncommitted transaction') AS uncommitted_transaction_check FROM v\$transaction t, v\$session s, v\$mystat m WHERE t.ses_addr = s.saddr AND s.sid = m.sid AND ROWNUM = 1#, 
               "\n/\n", $catcon_DebugOn);
          }

          # unless we are running single-threaded, we a need a "done" file to 
          # be created after the current script or statement completes so that 
          # next_proc() would recognize that this process is available and 
          # consider it when picking the next process to run a script or SQL 
          # statement
          if (!$SingleThreaded) {
            # file which will indicate that process $CurProc finished its 
            # work and is ready for more
            #
            # Bug 18011217: append _catcon_$catcon_ProcIds[$CurProc] to 
            # $catcon_LogFilePathBase to avoid conflicts with other catcon 
            # processes running on the same host
            my $DoneFile = 
              done_file_name($catcon_LogFilePathBase, 
                             $catcon_ProcIds[$CurProc]);

            # if catcon_DebugOn is set, printToSqlplus will normally issue 
            #   "select 'catconExec() ... as catcon_statement from dual"
            # which we need to avoid if we are running catshutdown
            printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                           qq/$catcon_DoneCmd $DoneFile/,
                           "\n", 
                           $catcon_DebugOn && 
                             !$catcon_CatShutdown{$Containers[$CurCon]});

            if ($catcon_DebugOn) {
              log_msg <<msg;
catconExec: sent "$catcon_DoneCmd $DoneFile" to process 
    $CurProc (id = $catcon_ProcIds[$CurProc]) to indicate its availability 
    after completing $FilePath
msg
            }
          }

          $catcon_FileHandles[$CurProc]->flush;
        }

        # if we are running single-threaded, we a need a "done" file to be 
        # created after the last script or statement sent to the current 
        # Container completes so that next_proc() would recognize that this 
        # process is available and consider it when picking the next process 
        # to run a script or SQL statement
        if ($SingleThreaded) {
          # file which will indicate that process $CurProc finished its work 
          # and is ready for more
          #
          # Bug 18011217: append _catcon_$catcon_ProcIds[$CurProc] to 
          # $catcon_LogFilePathBase to avoid conflicts with other catcon 
          # processes running on the same host
          my $DoneFile = 
            done_file_name($catcon_LogFilePathBase, $catcon_ProcIds[$CurProc]);

          # if catcon_DebugOn is set, printToSqlplus will normally issue 
          #   select "catconExec() ... as catcon_statement from dual"
          # which we need to avoid if the last script we ran was catshutdown
          printToSqlplus("catconExec", $catcon_FileHandles[$CurProc],
                         qq/$catcon_DoneCmd $DoneFile/,
                         "\n", 
                         $catcon_DebugOn &&
                         index($ScriptPaths[$#ScriptPaths], "catshutdown") == 
                           -1);

          # flush the file so a subsequent test for file existence does 
          # not fail due to buffering
          $catcon_FileHandles[$CurProc]->flush;
      
          if ($catcon_DebugOn) {
            log_msg <<msg;
catconExec: sent "$catcon_DoneCmd $DoneFile" to process 
    $CurProc (id = $catcon_ProcIds[$CurProc]) to indicate its availability
msg
          }
        }
      }

      # if 
      #   - the user told us to issue per-process init and completion 
      #     statements and 
      #   - there are any such statements to send, 
      # they need to be passed to wait_for_completion
      my $EndStmts = $IssuePerProcStmts ? $catcon_PerProcEndStmts : undef;

      # wait_for_completion uses calls printToSqlplus to send "done" command to
      # processes and if catcon_DebugOn is set, printToSqlplus will normally 
      # issue 
      #   "select 'catconExec() ... as catcon_statement from dual"
      # which we need to avoid if the last script we ran was catshutdown
      if (wait_for_completion($ProcsUsed, $catcon_NumProcesses, 
            $catcon_LogFilePathBase, 
            @catcon_FileHandles, @catcon_ProcIds,
            $catcon_DoneCmd, @$EndStmts, 
            @catcon_Containers ? $CDBorFedRoot : undef,
            @catcon_InternalConnectString,
            $catcon_RecoverFromChildDeath,
            $catcon_KillAllSessScript, 
            index($ScriptPaths[$#ScriptPaths], "catshutdown") != -1,
            $catcon_DebugOn) == 1) {
        # unexpected error was encountered, return
        log_msg("catconExec: unexpected error in wait_for_completions\n");

        return 1;
      }
    }

    # success
    return 0;
  }

  #
  # catconRunSqlInEveryProcess - run specified SQL statement(s) in every 
  #                              process
  #
  # Parameters:
  #   - reference to a list of SQL statement(s) which should be run in every 
  #     process
  # Returns
  #   1 if some unexpected error was encountered; 0 otherwise
  #
  sub catconRunSqlInEveryProcess (\@) {
    my ($Stmts) = @_;

    my $ps;

    # there must be at least one statement to execute
    if (!$Stmts || !@$Stmts || $#$Stmts == -1) {
      log_msg("catconRunSqlInEveryProcess: At least one SQL statement must be supplied");
      return 1;
    }

    if ($catcon_DebugOn) {
      log_msg("running catconRunSqlInEveryProcess\nSQL statements:\n");
      foreach (@$Stmts) {
        log_msg("\t$_\n");
      }
    }

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconRunSqlInEveryProcess: catconInit has not been run");
      return 1;
    }

    # send each statement to each process
    foreach my $Stmt (@$Stmts) {
      for ($ps=0; $ps < $catcon_NumProcesses; $ps++) {
        printToSqlplus("catconRunSqlInEveryProcess", $catcon_FileHandles[$ps],
                       $Stmt, "\n", $catcon_DebugOn);
      }    
    }

    return 0;
  }

  #
  # catconShutdown - shutdown the database
  #
  # Parameters:
  #   - shutdown flavor (e.g. ABORT or IMMEDIATE)
  # 
  # Returns
  #   1 if some unexpected error was encountered; 0 otherwise
  #
  sub catconShutdown (;$) {

    my ($ShutdownFlavor) = @_;
      
    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconShutdown: catconInit has not been run");
      return 1;
    }

    # free up all resources
    catconWrapUp();

    # if someone wants to invoke any catcon subroutine, they will need 
    # to invoke catconInit first

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconShutdown: will shutdown database using SHUTDOWN $ShutdownFlavor\n");
    }

    # shutdown_db needs to be called after catconWrapUp() to make sure that 
    # all processes have exited.
    #
    # Bug 18011217: append _catcon_$catcon_ProcIds[0] to 
    # $catcon_LogFilePathBase to avoid conflicts with other catcon processes 
    # running on the same host
    shutdown_db(@catcon_InternalConnectString, $ShutdownFlavor,
                $catcon_DoneCmd, 
                done_file_name_prefix($catcon_LogFilePathBase, 
                                      $catcon_ProcIds[0]), 
                $catcon_DebugOn);

    return 0;
  }

  #
  # catconUpgEndSessions - Ends all sessions except one.
  #
  # Parameters:
  #   - None
  #
  sub catconUpgEndSessions () {
 
    # @@@@     
    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconUpgEndSessions: catconInit has not been run\n");
      return 1;
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconUpgEndSessions: Will end all sessions but one\n");
    }

    #
    # End Every Process but 1.
    #
    if ($catcon_NumProcesses > 1) {
 
        # @@@@
        my $ps;
        for ($ps = 1; $ps <= $catcon_NumProcesses - 1; $ps++) {
            if ($catcon_FileHandles[$ps]) {
                print {$catcon_FileHandles[$ps]} "PROMPT ========== PROCESS ENDED ==========\n";
            }
        }

        # End All Sql Processors but one
        end_processes(1, $catcon_NumProcesses - 1, @catcon_FileHandles, 
                  @catcon_ProcIds, $catcon_DebugOn, $catcon_LogFilePathBase,
                  -1, \%catcon_InstProcMap, \%catcon_ProcInstMap);

        # Set Number of processors to 1
        $catcon_NumProcesses = 1;
    }
    return 0;

  }


  #
  # catconUpgStartSessions - Starts all Sessions.
  #
  # Returns = 1 Error
  #           0 Success
  # Parameters:
  #   - $NumProcesses  = How many process to start
  #   - $ResetLogs     = Reset Logs TRUE(1) or FALSE(0)
  #   - $LogDir        = New Log Directory
  #   - $LogBase       = Log Base
  #   - $DebugOn       = catcon.pm Debugging
  #
  sub catconUpgStartSessions ($$$$$) {

    my ($NumProcesses,$ResetLogs,$LogDir,$LogBase,$DebugOn) = @_;
    my $FirstTime = 0;

    # Check if we are debugging 
    if ($DebugOn) {

        my $ts = TimeStamp();
        log_msg <<catconUpgStartSessions_DEBUG;
  running catconUpgStartSessions(NumProcesses   = $NumProcesses, 
                                 ResetLogs      = $ResetLogs,
                                 LogDir         = $LogDir, 
                                 LogBase        = $LogBase, 
                                 Debug          = $DebugOn)\t\t($ts)
catconUpgStartSessions_DEBUG
    }

      
    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconUpgStartSessions: catconInit has not been run\n");
      return 1;
    }

    # we need to save value of $catcon_LogFilePathBase here in case $ResetLogs 
    # is set since end_processes() uses $LogFilePathBase when cleaning up 
    # completion files generated by processes being "ended."  
    # Of course, if the old log directory has been deleted before this function
    # was called, we won't get much for our trouble, but at least we tried.
    my $saveLogFilePathBase = $catcon_LogFilePathBase;
    
    # Reset Log Directory
    if ($ResetLogs)
    {
        # Set First Time to True
        # And don't override the
        # number of processors.
        # We are only switching the
        # log files.
        $FirstTime = 1;
        $NumProcesses = $catcon_NumProcesses;

        # Close out error log file
        close ($CATCONOUT);

        # set Log File Path Base
        $catcon_LogFilePathBase = 
            set_log_file_base_path ($LogDir,$LogBase, $DebugOn);

        if (!$catcon_LogFilePathBase) {
            print STDERR
                "catconUpgStartSessions: Unexpected error returned ".
                "by set_log_file_base_path\n";
            return 1;
        }

        # Set Log directory
        $catcon_LogDir = $LogDir;
    }

    #
    # End current session
    #
    end_processes(0, $catcon_NumProcesses - 1, @catcon_FileHandles, 
                  @catcon_ProcIds, $catcon_DebugOn, $saveLogFilePathBase,
                  -1, \%catcon_InstProcMap, \%catcon_ProcInstMap);

    # delete the "kill all sessions script" since all processes it would try 
    # to kill are now gone
    sureunlink($catcon_KillAllSessScript, $catcon_DebugOn);

    if ($ResetLogs)
    {
        # we want to keep $catcon_KillAllSessScript 
        # in sync with $catcon_LogFilePathBase (because in some cases, the old 
        # log directory may no longer be around which will cause us to get 
        #     start_processes: failed to open (2) kill_all_sessions script
        # message when we try to append ALTER SYSTEM KILL SESSION statements 
        # to the kill_all_sessions script in start_processes().
        #
        # We waited until now to do it because of sureunlink() call above that 
        # tries to delete the kill_all_sessions script after end_processes()
        # returns
        $catcon_KillAllSessScript = 
            kill_session_script_name($catcon_LogFilePathBase, "ALL");
    }    
    
    #
    # Set number of Processors
    #
    $catcon_NumProcesses = $NumProcesses;

    #
    # Bug 22887047: we set SIG{CHLD} to 'IGNORE' in catconInit, and that's 
    #   where we want to leave it since we will deal with any dead child 
    #   processes when we come across them in the course of trying to pick a 
    #   next process to do perform some task (next_proc())
    #

    # start processes
    if (start_processes($catcon_NumProcesses, $catcon_LogFilePathBase, 
                        @catcon_FileHandles, @catcon_ProcIds, 
                        @catcon_Containers, $catcon_Root,
                        @catcon_UserConnectString, 
                        $catcon_EchoOn, $catcon_ErrLogging,
                        $catcon_DebugOn, $FirstTime,
                        $catcon_UserScript, $catcon_FedRoot,
                        $catcon_DoneCmd, $catcon_DisableLockdown,
                        -1, undef, $catcon_GenKillSessScript,
                        $catcon_KillAllSessScript, \%catcon_InstProcMap, 
                        \%catcon_ProcInstMap, \%catcon_InstConnStr)) {
      return 1;
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconUpgStartSessions: finished starting $catcon_NumProcesses processes\n");
    }

    return 0;

  }


  #
  # catconBounceProcesses - bounce all processes
  # 
  # Returns
  #   1 if some unexpected error was encountered; 0 otherwise
  #
  sub catconBounceProcesses () {

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconBounceProcesses: catconInit has not been run");
      return 1;
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      if ($catcon_DeadProc == -1) {
        log_msg("catconBounceProcesses: will bounce $catcon_NumProcesses ".
                "processes\n");
      } else {
        log_msg("catconBounceProcesses: will replace dead process ".
                "$catcon_DeadProc (id=$catcon_DeadProcId)\n");
      }
    }

    # end processes
    end_processes(0, $catcon_NumProcesses - 1, @catcon_FileHandles, 
                  @catcon_ProcIds, $catcon_DebugOn, $catcon_LogFilePathBase,
                  $catcon_DeadProc, \%catcon_InstProcMap, 
                  \%catcon_ProcInstMap);

    if ($catcon_DeadProc == -1) {
      # delete the "kill all sessions script" since all processes it would 
      # try to kill are now gone
      sureunlink($catcon_KillAllSessScript, $catcon_DebugOn);
      if ($catcon_DebugOn) {
        log_msg("catconBounceProcesses: $catcon_KillAllSessScript was ".
                "deleted because all processes listed in it have been ".
                "ended\n");
      }
    }

    #
    # Bug 22887047: we set SIG{CHLD} to 'IGNORE' in catconInit, and that's 
    #   where we want to leave it since we will deal with any dead child 
    #   processes when we come across them in the course of trying to pick a 
    #   next process to do perform some task (next_proc())
    #

    # start processes
    if (start_processes($catcon_NumProcesses, $catcon_LogFilePathBase, 
                        @catcon_FileHandles, @catcon_ProcIds, 
                        @catcon_Containers, $catcon_Root,
                        @catcon_UserConnectString, 
                        $catcon_EchoOn, $catcon_ErrLogging, $catcon_DebugOn, 0,
                        $catcon_UserScript, $catcon_FedRoot, 
                        $catcon_DoneCmd, $catcon_DisableLockdown,
                        $catcon_DeadProc, $catcon_DeadProcId,
                        $catcon_GenKillSessScript, 
                        $catcon_KillAllSessScript, \%catcon_InstProcMap, 
                        \%catcon_ProcInstMap, \%catcon_InstConnStr)) {
      return 1;
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      if ($catcon_DeadProc == -1) {
        log_msg("catconBounceProcesses: finished bouncing ".
                "$catcon_NumProcesses processes\n");
      } else {
        log_msg("catconBounceProcesses: finished replacing dead process ".
                "$catcon_DeadProc (id=$catcon_DeadProcId)\n");
      }
    }

    return 0;
  }

  #
  # bounce a process that died unexpectedly
  #
  # Parameters:
  # - index into catcon_ProcIds of the process that needs bouncing
  #
  sub catconBounceDeadProcess ($) {
    my ($deadProc) = @_;

    # remember that we need to bounce a dead process and save its id so we 
    # can produce a warning message in start_processes when we start a 
    # replacement process
    $catcon_DeadProc = $deadProc;
    $catcon_DeadProcId = $catcon_ProcIds[$deadProc];

    my $ret = catconBounceProcesses();

    # attempt to bounce the dead process has been made (successfully or 
    # otherwise)
    $catcon_DeadProc = -1;
    undef $catcon_DeadProcId;

    return ($ret) ? 1 : 0;
  }

  #
  # catconWrapUp - free any resources which may have been allocated by 
  #                various catcon.pl subroutines
  # 
  # Returns
  #   1 if some unexpected error was encountered; 0 otherwise
  #
  sub catconWrapUp () {

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconWrapUp: catconInit has not been run");
      return 1;
    }

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconWrapUp: (PID=$$) about to free up all resources\n");
    }


    # @@@@
    my $ps;
    for ($ps = 0; $ps <= $catcon_NumProcesses - 1; $ps++) {
        if ($catcon_FileHandles[$ps]) {
            print {$catcon_FileHandles[$ps]} "PROMPT ========== PROCESS ENDED ==========\n";
            }
    }


    # end processes
    end_processes(0, $catcon_NumProcesses - 1, @catcon_FileHandles, 
                  @catcon_ProcIds, $catcon_DebugOn, $catcon_LogFilePathBase,
                  -1, \%catcon_InstProcMap, \%catcon_ProcInstMap);

    # delete the "kill all sessions script" since all processes it would try 
    # to kill are now gone
    sureunlink($catcon_KillAllSessScript, $catcon_DebugOn);

    # if we reopened PDBs in modes specified by the caller in 
    # catconExec(), revert them to their original modes
    catconRevertPdbModes();

    # restore signal handlers which were reset in catconInit
    # NOTE: if, for example, $SIG{CHLD} existed but was not defined in 
    #       catconInit, I would have liked to undef $SIG{CHLD} here, but that 
    #       results in "Use of uninitialized value in scalar assignment" error,
    #       so I was forced to delete the $SIG{} element instead

    if ((exists $catcon_SaveSig{CHLD}) && (defined $catcon_SaveSig{CHLD})) {
      $SIG{CHLD}  = $catcon_SaveSig{CHLD};
    } else {
      delete $SIG{CHLD};
    }

    if ((exists $catcon_SaveSig{INT}) && (defined $catcon_SaveSig{INT})) {
      $SIG{INT}  = $catcon_SaveSig{INT};
    } else {
      delete $SIG{INT};
    }

    if ((exists $catcon_SaveSig{TERM}) && (defined $catcon_SaveSig{TERM})) {
      $SIG{TERM}  = $catcon_SaveSig{TERM};
    } else {
      delete $SIG{TERM};
    }

    if ((exists $catcon_SaveSig{QUIT}) && (defined $catcon_SaveSig{QUIT})) {
      $SIG{QUIT}  = $catcon_SaveSig{QUIT};
    } else {
      delete $SIG{QUIT};
    }

    # no catcon* subroutines should be invoked without first calling catconInit
    $catcon_InitDone = 0;

    if ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconWrapUp: done\n");
    }

    return 0;
  }

  #
  # catconIsCDB - return an indicator of whether the database to which we are 
  # connected is a CDB
  #
  sub catconIsCDB () {

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconIsCDB: catconInit has not been run");
      return undef;
    }

    if (@catcon_AllContainers) {
      return 1;
    } else {
      return 0;
    }
  }

  #
  # catconGetConNames - return an array of Container names if connected to a 
  #                     CDB
  #
  sub catconGetConNames () {

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconGetConNames: catconInit has not been run");
      return undef;
    }

    return @catcon_AllContainers;
  }

  #
  # catconQuery - run a query supplied by the caller 
  #
  # Parameters:
  #   - query to run (IN)
  #     NOTE: query needs to start with "select" with nothing preceding it.  
  #           Failure to do so will cause the function to return no results
  #   - name of PDB, if any, in which to run the query.  If not specified, 
  #     query will be run in CDB$ROOT.
  #
  # Returns:
  #   Output of the query.
  #
  sub catconQuery($$) {
    my ($query, $pdb) = @_;

    if ($catcon_DebugOn) {
      my $pdbNameString = $pdb ? $pdb : "unspecified";
      my $msg = <<catconQuery_DEBUG;
running catconQuery(
  query = $query, 
  pdb   = $pdbNameString)
catconQuery_DEBUG
      log_msg($msg);
    }

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconQuery: catconInit has not been run");
      return undef;
    }

    # each row returned by the query needs to start with the "marker" 
    # (C:A:T:C:O:N) which identifies rows returned by the query (as opposed 
    # to various rows produced by SQL*PLus), so we replace "select" which 
    # start the query with "select 'C:A:T:C:O:N' ||"
    #
    #@@@@
    # NOTE: PLEASE NOTIFY THE UPGRADE GROUP IF THIS TAG 'C:A:T:C:O:N' CHANGES.
    #       Upgrade is using the tag to support read only tablespaces.
    #       The tag is used in conjunction with a sql with clause.
    #       If we do not add the correct tag to the with clause
    #       the parsing of the return data will fail.
    #
    $query =~ s/^select/select 'C:A:T:C:O:N' ||/i;

    my @statements = (
      "connect ".$catcon_InternalConnectString[0]."\n",
      "set echo off\n",
      "set heading off\n",
    );

    # if the caller supplied a PDB name, add 
    #   ALTER SESSION SET CONTAINER = <PDB name>
    # but first we need to verify that the specified PDB exists
    if ($pdb) {

      # make sure only one PDB name was specified
      my @PdbNameArr = split(/  */, $pdb);

      if ($#PdbNameArr != 0) {
        log_msg("catconQuery: exactly one PDB name can be specified\n");
        return undef;
      }

      # make sure we are connected to a CDB
      if (!catconIsCDB()) {
        log_msg("catconQuery: PDB name was specified while connected to a non-CDB\n");
        return undef;
      }

      # make sure specified name does not refer to the Root
      if ((uc $PdbNameArr[0]) eq (uc $catcon_Root)) {
        log_msg("catconQuery: specified PDB name may not refer to the Root\n");
        return undef;
      }

      # make sure the specified PDB exists 
      if (!(@catcon_Containers = 
            validate_con_names($pdb, 0, @catcon_AllContainers, 
                \%catcon_IsConOpen, 
                0, # do not ignore non-existent or closed PDBs
                $catcon_IgnoreAllInaccessiblePDBs, # Ignore all PDBs
                $catcon_DebugOn))) {
        log_msg("catconQuery: ".$pdb." does not refer to an existing and ".
                "open PDB\n");
        return undef;
      }

      push @statements, "alter session set container=".$pdb."\n/\n";
    }

    # finally, add the query
    push @statements, "$query\n/\n";

    # Bug 18011217: append _catcon_$catcon_ProcIds[0] to  
    # $catcon_LogFilePathBase to avoid conflicts with other catcon processes 
    # running on the same host
    my ($Out_ref, $Spool_ref) = 
      exec_DB_script(@statements, "C:A:T:C:O:N", $catcon_DoneCmd, 
                     done_file_name_prefix($catcon_LogFilePathBase,
                                           $catcon_ProcIds[0]), 
                     $catcon_DebugOn);
    if ($catcon_DebugOn) {
      log_msg("catconQuery: returning ".($#$Out_ref+1)." rows {\n");

      foreach (@$Out_ref) {
        log_msg("catconQuery:\t$_\n");
      }

      log_msg("catconQuery: }\n");
    }

    return @$Out_ref;
  }

  #
  # catconUserPass - return password used when running user scripts or 
  #                  statements
  #
  # Parameters:
  #   - None
  #
  sub catconUserPass() {
      
    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      log_msg("catconUserPass: catconInit has not been run");
      return undef;
    }

    return $catcon_UserPass;
  }

  #
  # catconSetDbClosed - set a flag that tells catcon whether the DB is closed
  #
  # Parameters:
  #   - a flag indicating whether the DB is closed 
  #     (0 => OPEN; otherwise => CLOSED)
  #
  sub catconSetDbClosed($) {
    my ($isClosed) = @_;

    $catcon_DbClosed = $isClosed;
  }

  # return tag for the environment variable containing password for a user 
  # connect string
  sub catconGetUsrPasswdEnvTag () {
    return $USER_PASSWD_ENV_TAG;
  }

  # return tag for the environment variable containing password for the 
  # internal connect string
  sub catconGetIntPasswdEnvTag () {
    return $INTERNAL_PASSWD_ENV_TAG;
  }

  sub catcon_HandleSigINT () {
    log_msg("Signal INT was received.\n");

    # Bug 18488530: END block (which gets called when we call die) will take 
    # care of resetting PDB$SEED's mode

    # ignore signals that may interrupt our signal handling code
    $SIG{INT} = 'IGNORE';
    $SIG{TERM} = 'IGNORE';
    $SIG{QUIT} = 'IGNORE';

    die;  
  }

  sub catcon_HandleSigTERM () {
    log_msg("Signal TERM was received.\n");

    # Bug 18488530: END block (which gets called when we call die) will take 
    # care of resetting PDB$SEED's mode

    # ignore signals that may interrupt our signal handling code
    $SIG{INT} = 'IGNORE';
    $SIG{TERM} = 'IGNORE';
    $SIG{QUIT} = 'IGNORE';

    die;  
  }

  sub catcon_HandleSigQUIT () {
    log_msg("Signal QUIT was received.\n");

    # Bug 18488530: END block (which gets called when we call die) will take 
    # care of resetting PDB$SEED's mode

    # ignore signals that may interrupt our signal handling code
    $SIG{INT} = 'IGNORE';
    $SIG{TERM} = 'IGNORE';
    $SIG{QUIT} = 'IGNORE';

    die;  
  }

  # In addition to being called from catconWrapUp, this subroutine should be 
  # invoked as a part of handling SIGINT to ensure that PDBs get reverted to
  # modes in which they were open before catcon started manipulating it. 
  # assembled above
  #
  # A call to this function should be added to the END block to ensure that 
  # PDB modes get restored after die is invoked.
  sub catconRevertPdbModes() {
    # if we reopened PDB$SEED in a mode specified by the caller in 
    # catconExec(), revert it to the original mode
    #
    # NOTE: normally, we need to wait for all processes to be killed before 
    #       calling reset_pdb_modes() (because it closes PDBs before reopening 
    #       them in desired mode, and if any process is connected to such PDB
    #       while we are closing it, ALTER PDB CLOSE will hang), but if catcon 
    #       process itself is getting killed, there is no room for such 
    #       niceties.
    if (%catcon_RevertSeedPdbMode) {
      # Bug 18011217: append _catcon_$$_revertPdbModes_seed_pdb to 
      # $catcon_LogFilePathBase to avoid conflicts with other catcon 
      # processes running on the same host or with the subsequent call to 
      # reset_pdb_modes() or with calls from force_pdb_modes
      my $ret = 
        reset_pdb_modes(\@catcon_InternalConnectString,
                        \%catcon_RevertSeedPdbMode, $catcon_DoneCmd, 
                        done_file_name_prefix($catcon_LogFilePathBase, 
                                              $$."_revertPdbModes_seed_pdb"),
                        1, $catcon_DbmsVersion, 0, \%catcon_InstConnStr, 
                        $catcon_DebugOn);
    
      if ($ret == 1) {
        log_msg("catconRevertPdbModes: unexpected error trying to reopen ".
                "PDB\$SEED in original mode\n");
        # not returning error code since we are wrapping things up
      } elsif ($ret == 0) {
        if ($catcon_DebugOn || $catcon_Verbose) {
          log_msg("catconRevertPdbModes: reverted PDB\$SEED to original ".
                  "mode\n");
        }
      } elsif ($catcon_DebugOn || $catcon_Verbose) {
            log_msg <<msg;
catconRevertPdbModes: PDB\$SEED could NOT be reopened in original mode
    because it would conflict with the mode in which the CDB is open
    (this should never happen because that was the mode in which it was open 
     when catcon was called)
msg
      }

      undef %catcon_RevertSeedPdbMode;
    } elsif ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconRevertPdbModes: catcon_RevertSeedPdbMode was not set\n");
    }

    # ditto for user PDBs
    if (%catcon_RevertUserPdbModes) {
      my $ret = 
        reset_pdb_modes(\@catcon_InternalConnectString,
                        \%catcon_RevertUserPdbModes, $catcon_DoneCmd, 
                        done_file_name_prefix($catcon_LogFilePathBase, 
                                              $$."_revertPdbModes_user_pdbs"),
                        0, $catcon_DbmsVersion, 0, \%catcon_InstConnStr, 
                        $catcon_DebugOn);
    
      if ($ret == 1) {
        log_msg("catconRevertPdbModes: unexpected error trying to reopen ".
                "user PDBs in original mode\n");
        # not returning error code since we are wrapping things up
      } elsif ($ret == 0) {
        if ($catcon_DebugOn || $catcon_Verbose) {
          log_msg("catconRevertPdbModes: reverted user PDBs to original ".
                  "mode\n");
        }
      } elsif ($catcon_DebugOn || $catcon_Verbose) {
            log_msg <<msg;
catconRevertPdbModes: user PDBs could NOT be reopened in original modes
    because it would conflict with the mode in which the CDB is open
    (this should never happen because those were the modes in which they 
     were open when catcon was called)
msg
      }

      undef %catcon_RevertUserPdbModes;
    } elsif ($catcon_DebugOn || $catcon_Verbose) {
      log_msg("catconRevertPdbModes: catcon_RevertUserPdbModes was not set\n");
    }
  }

  #
  # catconSqlplus - run a SQL*Plus script supplied by the caller.
  #
  # Parameters:
  #   - a reference to a list of statements (i.e. a script) to execute; 
  #     the script needs to include CONNECT and/or SET CONTAINER 
  #     statements to get to the container in which statements need to be 
  #     executed
  #   - base for a name of a file which will be created to indicate that the 
  #     script has completed
  #
  # Returns:
  #   none
  #
  sub catconSqlplus(\@$) {
    my ($script, $DoneFilePathBase) = @_;

    my $isWindows; # are we running on Windows?

    # command which exec_DB_script will write to SQL*Plus so it would 
    # generate a "done" file (signalling to exec_DB_script that the script 
    # has completed)
    my $doneCmd;   

    # determine doneCmd appropriate to the OS on which we are running
    os_dependent_stuff($isWindows, $doneCmd);

    # NOTE: appending process id to $DoneFilePathBase to avoid conflicts with 
    #       other processes running on the same host
    my ($Out_ref, $Spool_ref) = 
      exec_DB_script(@$script, undef, $doneCmd, 
                     done_file_name_prefix($DoneFilePathBase, $PID),
                     0);
    return;
  }

  END {
    # ignore signals that may interrupt our signal handling code
    $SIG{INT} = 'IGNORE';
    $SIG{TERM} = 'IGNORE';
    $SIG{QUIT} = 'IGNORE';

    # Bug 21871308: if Ctrl-C was hit after we started one or more processes, 
    #   ensure that they all die gracefully, releasing all resources
    if ($catcon_KillAllSessScript && (-e $catcon_KillAllSessScript)) {
      kill_sqlplus_sessions(@catcon_InternalConnectString, 
                            $catcon_KillAllSessScript,
                            $catcon_DoneCmd, 
                            done_file_name_prefix($catcon_LogFilePathBase, $$),
                            1);
    }

    # Bug 18488530: make sure PDB modes gets reverted before catcon process 
    # dies
    catconRevertPdbModes();
  }
}

1;
