# Copyright (c) 2011, 2015, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      s_crsutils.pm - <one-line expansion of the name>
#
#    DESCRIPTION
#      OSD module for Windows
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    muhe        06/05/16 - Fix bug 22957000
#    luoli       12/16/15 - Fix bug 22349063
#    xyuan       11/25/15 - Fix bug 22253598
#    shullur     06/29/15 - For migrating CHM modules to new framework
#    jbhati      03/08/15 - Enabling Oracle Fence Service in Dev env
#    sbezawad    01/20/15 - Bug 20019354: Migrate OCR and OLR to new framework
#    luoli       11/25/14 - Fix bug 19955755 for windows
#    jmarcias    09/17/14 - Fix bug 19325613
#    muhe        09/05/14 - Fix bug 19147315
#    rdasari     07/28/14 - remove fence service for SIHA also
#    xyuan       04/24/14 - Fix bug 18635452
#    muhe        04/03/14 - Fix bug 16801852
#    muhe        03/12/14 - Fix bug 18373231
#    ssprasad    03/05/14 - Move afd conf related processing to oraafd.pm
#    ssprasad    02/05/14 - Add s_rm_afd_conf
#    siyarlag    10/18/13 - Bug\17628687: add s_is_Exadata
#    rdasari     10/09/13 - use OCRLOC instead of OCRCONFIG
#    shullur     08/29/13 - For moving CHM windows specific code
#    xyuan       04/26/13 - Fix bug 16636361
#    xyuan       04/10/13 - Add s_restoreocrloc and s_backupocrloc
#    xyuan       04/08/13 - XbranchMerge xyuan_bug-16579079 from
#                           st_has_12.1.0.1
#    xyuan       04/07/13 - XbranchMerge xyuan_bug-16495293 from
#                           st_has_12.1.0.1
#    ysharoni    04/05/13 - bug 16617648 add Mod rights to wallet lock files
#    xyuan       04/04/13 - Fix bug 16579079
#    xyuan       03/15/13 - Fix bug 16495293
#    sidshank    03/14/13 - XbranchMerge sidshank_bug-16382128_win from
#                           st_has_12.1.0.1
#    sidshank    03/04/13 - fix bug 16382128.
#    sidshank    02/19/13 - XbranchMerge sidshank_bug-16175611_win from
#    sidshank    02/19/13 - XbranchMerge sidshank_bug-16185238 from
#                           st_has_12.1.0.1
#    sidshank    01/25/13 - fix bug 16185238
#    sidshank    01/24/13 - fix bug 16175611.
#    gmaldona    10/31/12 - Create temporary file in OS TEMP directory
#    jmunozn     10/11/12 - Add s_get_qosctl_path function
#    sidshank    09/12/12 - fix bug 14511564.
#    sidshank    08/30/12 - fix bug 14530963.
#    shmubeen    08/08/12 - afd install functions
#    ysharoni    07/18/12 - bug 14266142 add node name to orcl owner in icacls
#    sidshank    05/23/12 - fix bug 14106919
#    sidshank    04/09/12 - Add Path module. Remove restore/redirect stdout
#                           subroutines.
#    ysharoni    03/27/12 - add s_gpnp_wallets_set_access_win
#    gmaldona    02/24/12 - I added a new function called s_run_as_user3
#    sidshank    02/07/12 - Remove OCFS related code
#    sidshank    01/12/12 - Adding dummy routine s_install_initd
#    sidshank    11/09/11 - Fix for tne bug 13352502
#    rvadraha    10/25/11 - Bug13247694, Fix ocfs upgrade
#    sidshank    08/22/11 - removing the workaround for 12739826
#    dpham       05/01/11 - New for 12c
# 
package s_crsutils;

use strict;
use Win32;
use Win32::NetAdmin qw(DOMAIN_ALIAS_RID_ADMINS GetAliasFromRID
                       LocalGroupIsMember GroupIsMember);
use Win32::TieRegistry (Delimiter => '/');
use Win32::Service;
use Win32API::File  qw(DeleteFile);
use File::Spec::Functions;
use File::Path;
use File::Temp qw/ tempfile /;
use Cwd;

# root script module
use crsutils;

# export vars and functions
use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);

@ISA = qw(Exporter);

my @exp_func = qw(s_check_SuperUser s_set_ownergroup s_reset_crshome
                  s_reset_crshome1 s_set_perms s_osd_setup
                  s_check_CRSConfig s_get_olr_file
                  s_reset_srvconfig
                  s_register_service s_unregister_service s_check_service
                  s_start_service s_run_as_user s_run_as_user2 s_init_scr
                  s_get_config_key s_isLink s_get_platform_family
                  s_getOldCrsHome s_redirect_souterr s_restore_souterr
                  s_stop_OldCrsStack s_RemoveInitResources s_CleanTempFiles
                  s_setParentDirOwner s_resetParentDirOwner s_checkOracleCM
                  s_ResetVotedisks s_createConfigEnvFile
                  s_isRAC_appropriate s_is92ConfigExists
                  s_configureCvuRpm s_removeCvuRpm s_remove_file s_getAbsLink
                  s_removeGPnPprofile 
                  s_is_HAIP_supported s_is_HAIP_NonFatal s_CheckNetworkConfig
                  s_houseCleaning s_add_upstart_conf s_NSCleanUp s_getGroupName
                  s_set_ownergroup_win
		  s_configureAllRemoteNodes s_removeSCR s_install_initd
                  s_restoreInitScripts s_restoreASMFiles 
                  s_gpnp_wallets_set_access_win
                  s_run_as_user3 s_copyRegKey s_stopService s_isServiceRunning
		  s_stopDeltOldASM s_upgrade_services s_deltService s_setPermsASMDisks  
		  s_rm_afdinit_init s_rm_afdinit_rclevel s_copy_afdinit_init
                  s_get_qosctl_path s_delete_Oracle_Services
                  s_is_Exadata s_getRemoteCrsHome s_migrateDBServiceSidsUpgrade
                  s_migrateDBServiceSidsDowngrade s_isServiceExists s_verifyClonedHome 
                 );

push @EXPORT, @exp_func;

####---------------------------------------------------------
#### Function for checking and returning Super User name
# ARGS : 1
# ARG1 : Program name
sub s_check_SuperUser
{
    trace ("Checking for super user privileges");

    my $superUser = $ENV{'USERNAME'};
    my $groupName = s_getGroupName();
    trace ("superUser=$superUser groupName=$groupName");

    # get group name for Administrators
    if (! $groupName) {
        return "";
    }

    # get user-name
    my $userName   = Win32::LoginName();
    $userName      =~ tr/a-z/A-Z/;
    my $errorMsg   = "User must be \"SYSTEM\", or $userName must be " .
                     "or a member of $groupName group to run root script";
    trace ("user=$userName");

    # get SYSTEM
    my $systemName = 'SYSTEM';
    if (! is_dev_env ()) {
       $systemName = s_getSystem();
       $systemName =~ tr/a-z/A-Z/;
    }

    if ($userName eq $systemName) {
       trace ("User has $superUser privileges");
       return $superUser;
    }

    # check if local user has privileges
    if (!(LocalGroupIsMember("", $groupName, $userName) ||
          GroupIsMember("", $groupName, $userName))) 
    {
       # local user does not have privileges
       # now check if ORACLE_OWNER has privileges
       
          $userName = $CFG->params('ORACLE_OWNER');
          trace ("verifying Admin privilege for user=$userName");

          if (!(LocalGroupIsMember("", $groupName, $userName) ||
             GroupIsMember("", $groupName, $userName))) 
          { 
             error ("$errorMsg");
             return "";
          }
    }

    trace ("User has $superUser privileges");

    return $superUser;
}

# This function gets only "Administrators" (non-qualified) group name.
sub s_getGroupName
{
   # get group name for Administrators
   my $groupName;

   if (! GetAliasFromRID("", DOMAIN_ALIAS_RID_ADMINS, $groupName)) {
      error ("GetAliasFromRID failed");
   }

   return $groupName;
}

####---------------------------------------------------------
#### Function for setting user and group on a specified path
# ARGS : 3
# ARG1 : Oracle owner
# ARG2 : Oracle group 
# ARG3 : file
sub s_set_ownergroup
{
    # Note: this function is a no-op on NT

    return SUCCESS;
}

# OBSOLETED - do not use
sub s_set_ownergroup_win
#----------------------------------------------------------------
# Function: Use cacls to change file permissions
# Args    : owner, group, file
#----------------------------------------------------------------
{
   my ($owner, $group, $file) = @_;

   if (! $owner) {
      error ("Null value passed for Oracle owner");
      return FAILED;
   }

   if (! $group) {
      error ("Null value passed for group name");
      return FAILED;
   }

   if (! $file) {
      error ("Null value passed for file or directory path");
      return FAILED;
   }

   if (! (-e $file)) {
      error ("The path \"" . $file . "\" does not exist");
      return FAILED;
   }

   # set permission
   my $cmd = "cmd /c cacls $file /E /G \"$group\":F \"$owner\":F > NUL";
   if ($CFG->DEBUG) { trace ("Invoking: $cmd"); }
   system ("$cmd");
   
   return SUCCESS;
}

#----------------------------------------------------------------
# Use ACLs to change permissions for directory lists - used for gpnp wallets.
#
# It uses icacls utility to traverse directories and set necessary ACLs.
# icacls suported from W2K8R2 and W7Enterprise.
# Older utilities cacls and xcacls have various issues and are obsoleted.
#        
# ARGS: 3
# ARG1 - orauser name (on windows - DOMAIN\USER)
# ARG2 - ref to list of private directories
# ARG3 - ref to list of directories with added read access
#----------------------------------------------------------------
sub s_gpnp_wallets_set_access_win
{
   my $orauser      = $_[0];
   my $pvt_dirs_ref = $_[1];
   my $pub_dirs_ref = $_[2];
   my @pvt_dirs = @{$pvt_dirs_ref};
   my @pub_dirs = @{$pub_dirs_ref};


#   # Skip ACL set in dev env.
#   if (is_dev_env ()) {
#      return SUCCESS;
#   }

   # Check if ORACLE_OWNER returned by OUI has a domain qulifier.
   # Since 12, OUI will return non-qualified-name for local users, and
   # domain-qualified-name for domain users.
   # icacls behavior for non-qualified names is following: 
   # if node-local-user exists, it is used, else if domain-user exist, it is
   # used instead. To avoid ambiguity, we will prepend node-local-users
   # passed without qualifier in script on every node.

   # see if owner is node-local (does not have a domain qualifier)
   # if so, prepend with local host name
   if (0 > index($orauser,'\\')) {
      my $host = tolower_host();
      $orauser = "$host\\$orauser";  
   }

   my $ispubdirs = FALSE;
   my $resstatus = SUCCESS;
   foreach my $dirs_ref (\@pvt_dirs, \@pub_dirs) { 
      my @dirs = @{$dirs_ref};
      foreach (@dirs) {
         # For every given dir and its content, set ACLs:
         # icacls preferred for Windows > W2K3 R2
         my @cmd = ('icacls.exe', $_,       # for given dir
                       '/inheritance:r',    # remove inherited rights
                       '/grant:r',          # grant full access
                       '*S-1-5-18:F',       # NT AUTHORITY\SYSTEM
                       '/grant:r',          # grant full access, obj/cont inh
                       '*S-1-5-18:(OI)(CI)(F)',     # NT AUTHORITY\SYSTEM
                       '/grant:r',          # grant full access
                       '*S-1-5-32-544:F',   # BUILTIN\Administrators
                       '/grant:r',          # grant full access
                       '*S-1-5-32-544:(OI)(CI)(F)', # BUILTIN\Administrators
                       '/grant:r',          # grant full access
                       "\"$orauser\":F",    # crsuser
                       '/grant:r',          # grant full access
                       "\"$orauser\":(OI)(CI)(F)",  # crsuser
                       '/remove:g',         # revoke granted
                       '*S-1-5-11',         # NTA\Authenticated Users
                       '/t',                # traverse dir content
                       '/c'                 # continue on error
                    );
         # for public directories, add read grant for authorized users
         if ($ispubdirs) {
           push( @cmd, '/grant:r',          # grant read/exec access
                       '*S-1-5-11:RX',      # NTA\Authenticated Users
                       '/grant:r',
                       '*S-1-5-11:(OI)(CI)(RX)', # NTA\Authenticated Users
                    );
         }
         # execute cmd
         my @out = system_cmd_capture(@cmd);
         my $rc  = shift @out;
         trace("out=@out");

	 if ($rc == 0) {
	    trace ("@cmd ... success");
	 }
	 else {
	    trace ("@cmd ... failed with rc=", $rc);
            $resstatus = FAILED;
         }
      }
      $ispubdirs = TRUE;  # next process public dirs;
   }
   # Make sure wallet locks in public directories can be modified by group
   # see bug 15853970, bug 16617648 
   foreach (@pub_dirs) {
     my $lcks = catfile( $_, '*.lck' );
     my @cmd = ( 'icacls.exe', $lcks,       # for lock files in pub dirs
                       '/grant',            # add grant modification access
                       '*S-1-5-11:M'        # NTA\Authenticated Users
               );
     # execute cmd
     my @out = system_cmd_capture(@cmd);
     my $rc  = shift @out;
     trace("out=@out");

     if ($rc == 0) {
        trace ("@cmd ... success");
     }
     else {
        trace ("@cmd ... failed with rc=", $rc);
        $resstatus = FAILED;
     }
   } 
   if (! $resstatus) {
      trace("#################################################"); 
      trace("Setting restricred access to gpnp wallets encountered problems."); 
      trace("Affected directories: ".join(' ',@pvt_dirs)."  "
                                    .join(' ',@pub_dirs));
      trace("Make sure filesystem supports icacls and ACLs.");
      trace("#################################################"); 

      error("There was a problem setting restricted access to GPnP wallets. "
           ."System security compromised.");
   }
   return $resstatus;
}

####---------------------------------------------------------
#### Function for setting permissions on a specified path
# ARGS : 2
# ARG1 : permissions
# ARG3 : file/dir
sub s_set_perms
{
    # Note: this function is a no-op on NT

    return SUCCESS;
}

####-----------------------------------------------------------------------
#### Function for installing the Oracle ARP driver
# ARGS: 0
sub s_create_arp_service
{
  if (is_dev_env ())
  {
    return SUCCESS;
  }

  my ($string, $major, $minor, $build, $id, $spmajor, $spminor, $suitemask, $producttype) =
    Win32::GetOSVersion();

  # OS                      ID    MAJOR   MINOR
  # Windows Vista            2      6       0
  # Windows Server 2008      2      6       0
  # Windows 7                2      6       1
  # Windows Server 2008 R2   2      6       1

  # PRODUCTTYPE
  # 1 - Workstation (NT 4, 2000 Pro, XP Home, XP Pro, Vista)
  # 2 - Domaincontroller
  # 3 - Server (2000 Server, Server 2003, Server 2008)

  if ((6 == $major) && (0 == $minor || 1 == $minor) && (2 == $id)
       && (3 == $producttype))
  {
    trace("Installing the Oracle ARP driver");

    my $netcfg = catfile($ENV{SYSTEMROOT}, 'system32', 'netcfg.exe');
    if (! -e $netcfg)
    {
      trace("Invalid executable '$netcfg'");
      return FAILED;
    }

    my $infPath = catfile($CFG->ORA_CRS_HOME, 'bin', 'oraarpdrv.inf');
    $infPath =~ s{\\}{\\\\\\\\}g;
    trace("The location of INF is '$infPath'");

    my @cmd = ($netcfg, "-l", $infPath, "-c", "p", "-i", "orcl_ndisprot");
    my @out = system_cmd_capture(@cmd);
    my $rc = shift @out;
    if (0 == $rc)
    {
      trace("@cmd ... succeeded");
    }
    else
    {
      trace("@cmd ... failed with rc=", $rc >> 8);
      return FAILED;
    }

    # Start the service oraarpdrv
    if (SUCCESS != s_startService("oraarpdrv"))
    {
      trace("Failed to start the Oracle ARP service");
      return FAILED;
    }
  }

  trace("Succeeded in installing the Oracle ARP driver");
  return SUCCESS;
}

####-----------------------------------------------------------------------
#### Function for creating and starting Oracle Fence Service
# ARGS: 0
sub s_create_start_FenceServ
{
   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my ($cmd, $status);

   # Create Oracle Fence Service
   trace ("Creating Oracle Fence Service...");
   my $crssetup = catfile ($ORACLE_HOME,
                              "bin", "crssetup.exe");
   $cmd = "$crssetup installFence";
   if ($CFG->DEBUG) { trace ("Invoking: $cmd"); }

   $status = system ("$cmd");
   if ($status == 0) {
     trace ("Create Oracle Fence Service successfully");
   }
   else {
     error ("Create Oracle Fence Service failed");
     return FAILED;
   }

   # Start "OraFenceService"
   trace("Starting Oracle Fence Service ...");
   if (SUCCESS != s_startService ("OraFenceService")) {
      return FAILED;
   } 
       
   return SUCCESS;
}

####-----------------------------------------------------------------------
#### Function for performing NT-specific setup
# ARGS: 0
sub s_osd_setup
{
   if (! $CFG->UPGRADE)
   {
     # Create & start Oracle Fence Service
     trace("Create and start Oracle Fence Service");
     s_create_start_FenceServ();
   }

   # if in ADE env, skip these steps and return success
   if (is_dev_env ()) {
      return SUCCESS;
   }

   #set perms on  ASM Links if any.
   s_setPermsASMDisks();    

   # Install Oracle ARP Driver
   #if (! s_create_arp_service())
   #{
   #  return FAILED;
   #}
   return SUCCESS;
}

sub s_migrateDBServiceSidsUpgrade
{
  my $CLUUTIL = catfile($CFG->ORA_CRS_HOME, 'bin', 'cluutil');
  my @program = ($CLUUTIL, '-exec', '-upgradeDBServices');
  my @capout = ();

  my $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@capout, @program);
  
  if (0 != $rc)
   {
     trace("Failed to migrate the database service SIDs for Upgrade. Error code: $rc Output: @capout ");	
     return FAILED;
   }
   else
   {
     trace("Migrated the database service SIDs for Upgrade.");
     return SUCCESS;
   }
}

sub s_migrateDBServiceSidsDowngrade
{
  my $CLUUTIL = catfile($CFG->ORA_CRS_HOME, 'bin', 'cluutil');
  my @program = ($CLUUTIL, '-exec', '-downgradeDBServices');
  my @capout = ();

  my $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@capout, @program);
  
  if (0 != $rc)
   {
     trace("Failed to migrate the database service SIDs for Downgrade. Error code: $rc Output: @capout ");	
     return FAILED;
   }
   else
   {
     trace("Migrated the database service SIDs for Downgrade.");
     return SUCCESS;
   }
}


####-----------------------------------------------------------------------
#### Function for removing the Oracle ARP driver
# ARGS: 0
sub s_remove_arp_service
{
  if (is_dev_env ())
  {
    return SUCCESS;
  }

  my ($string, $major, $minor, $build, $id, $spmajor, $spminor, $suitemask, $producttype) =
    Win32::GetOSVersion();

  # OS                      ID    MAJOR   MINOR
  # Windows Vista            2      6       0
  # Windows Server 2008      2      6       0
  # Windows 7                2      6       1
  # Windows Server 2008 R2   2      6       1

  # PRODUCTTYPE
  # 1 - Workstation (NT 4, 2000 Pro, XP Home, XP Pro, Vista)
  # 2 - Domaincontroller
  # 3 - Server (2000 Server, Server 2003, Server 2008)

  if ((6 == $major) && (0 == $minor || 1 == $minor) && (2 == $id)
       && (3 == $producttype))
  {
    trace("Removing the Oracle ARP driver");

    if (!s_stopService("oraarpdrv"))
    {
      trace("Unable to stop the Oracle ARP service");
      return FAILED;
    }

    my $netcfg = catfile($ENV{SYSTEMROOT}, 'system32', 'netcfg.exe');
    if (! -e $netcfg)
    {
      error("Invalid executable '$netcfg'");
      return FAILED;
    }

    my @cmd = ($netcfg, "-u", "orcl_ndisprot");
    my @out = system_cmd_capture(@cmd);
    my $rc = shift @out;
    if (0 == $rc)
    {
      trace("@cmd ... succeeded");
    }
    else
    {
      trace("@cmd ... failed with rc=", $rc >> 8);
      return FAILED;
    }
  }

  trace("Succeeded in removing the Oracle ARP driver");
  return SUCCESS;
}

sub s_isServiceExists
#-------------------------------------------------------------------------------
# Function: Check if Windows service exists
# Args    : service name
#-------------------------------------------------------------------------------
{
   my $svcName = $_[0];
   my %srvc_status;
   
   Win32::Service::GetStatus("", $svcName, \%srvc_status);
  
   if ($srvc_status{"CurrentState"} =~ /[1-7]/) {
      return TRUE;
   }
   else {   
      return FALSE;
   }

}

sub s_isServiceRunning
#-------------------------------------------------------------------------------
# Function: Check Windows service is running or start_pending
# Args    : service name
#-------------------------------------------------------------------------------
{
   my $svcName = $_[0];
   my %srvc_status;
   
   # check if service is running
   Win32::Service::GetStatus("", $svcName, \%srvc_status);
  
   # 4 means service is running
   if ($srvc_status{"CurrentState"} == 4) {
      if ($CFG->DEBUG) { trace ("$svcName is running..."); }
      return TRUE;
   }
   else {   
      return FALSE;
   }
}

sub s_startService
#----------------------------------------------------------------
# Function: Start Windows service
# Args    : service name
#----------------------------------------------------------------
{
   my $svcName = $_[0];
   
   if (s_isServiceRunning ($svcName)) { 
      return SUCCESS;
   }  

   trace ("Starting $svcName...");
   if (! Win32::Service::StartService ("", $svcName)) {
      error ("Start of $svcName failed");
      return FAILED;
   }

   # wait for service to start
   my $retries = 5;
   my $srv_running = FALSE;
   
   while ($retries && (! $srv_running)) {
      if (s_isServiceRunning ($svcName)) { 
         $srv_running = TRUE;
      }
      else {
         trace ("Waiting for $svcName to start");
         sleep (60);
         $retries--;
      }
   }   
       
   if (! $srv_running) {
      error ("Error $svcName failed to start");
      return FAILED;
   }

   return SUCCESS;
}

sub s_deltService
#-------------------------------------------------------------------------------
# Function: Delete Windows service
# Args    : 1 - service name
#-------------------------------------------------------------------------------
{
   my $svcName = $_[0];

   if (s_isServiceExists($svcName)) {
      my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
      if (s_stopService($svcName)) {
         my $oradim = catfile ($ORACLE_HOME, 
                               'bin', 'oradim.exe');
         my @cmd    = ($oradim, "-ex", "s", "delete", "\"$svcName\"");
         my $rc     = system(@cmd);

         if ($rc == 0) {
            trace ("@cmd ... success");
            return SUCCESS;
         }
         else {
            trace ("@cmd ... failed with rc=", $rc >> 8);
            return FAILED;
         }
      }
   }

   return SUCCESS;
}

###------------------------------------------------------------------------
### Fucntion for deleting the specified Oracle service only if it's running
### from the specified crs home.
# ARGS: 2
# ARG1: Service name (it's not the full service name, it's a regular 
#       expression to be matched with the full name.)
# ARG2: CRS home to be compared

sub s_delete_Oracle_Services
{
   my $svc2delt = $_[0];
   my $crsHome2compare = $_[1];
   my $success = TRUE;
   my $svc_name;
   my %srvc_list;

   Win32::Service::GetServices("",\%srvc_list);

   foreach $svc_name (keys %srvc_list)
   {
     if ($svc_name =~ /^$svc2delt/i)
     {
        trace("Oracle Service identified is $svc_name");
	my @cmd1 = ("sc","qc",$svc_name);
	my @capout1;
  
        my $rc = s_run_as_user2(\@cmd1,$CFG->params('ORACLE_OWNER'), \@capout1);   

        if ($rc != 0)
        {
           $success = FALSE;
        }

        trace("executed @cmd1 with results... @capout1");

        if ($success)
        {
           foreach(@capout1)
           {
             chomp($_);
                 
             if ($_ =~ /BINARY_PATH_NAME/i)
             {
                my @res = split (/\s+/,trim($_));
                trim($res[2]);

		#Correct the path seprators to one standard before comparing them
                $crsHome2compare =~ s/\\\\/\\/g;
                $crsHome2compare =~ s/\//\\/g;
                $res[2] =~ s/\\\\/\\/g;
                $res[2] =~ s/\//\\/g;

                trace("binary path of service is $res[2] and crs home to be compared is $crsHome2compare");
                #Check if the specified Oracle Service binary is from the specified crs home location.
                if ($res[2] =~ /\Q$crsHome2compare/i) {
                   trace("attempting to delete $svc_name");
                   s_deltService($svc_name);
                }
             }
           }
        }
      }
    }
}

####-----------------------------------------------------------------------
#### Function for checking if CRS is already configured
# ARGS: 2
# ARG1: hostname
# ARG2: crs user
sub s_check_CRSConfig
{

    # ignore all args on NT

    # Check if OCR registry entry exists.
    # XXX: do we need any additional checks?
    my $OCRLOC = $CFG->params('OCRLOC');

    if ($Registry->{"LMachine/$OCRLOC/"}) {
        trace ("HKLM/$OCRLOC/ is already configured\n");
        return TRUE;
    }
    else {
        trace ("HKLM/$OCRLOC/ is NOT configured\n");
        return FALSE;
    }
}

####---------------------------------------------------------
#### Function for invalidating srvconfig_loc in srvconfig.loc file
sub s_reset_srvconfig
{

    # XXX: currently a no-op on NT; do we need to do anything here?  Like, say,
    # remove some registry (sub)keys??
    return SUCCESS;
}

####---------------------------------------------------------
#### Function for resetting crshome user and permissions
sub s_reset_crshome
{

    # currently a no-op on NT;
    return SUCCESS;
}

####---------------------------------------------------------
#### Function for registering daemon/service with init
# ARGS: 2
# ARG1: daemon to be registered
# ARG2: crs home where the service should be installed
sub s_register_service
{

    my $srv = $_[0];
    my $ORACLE_HOME = $_[1];
    if (! defined $ORACLE_HOME)
    {
       $ORACLE_HOME = $CFG->params('ORACLE_HOME');
    }
    if ( $CFG->DOWNGRADE ) {
       $ORACLE_HOME = $CFG->OLD_CRS_HOME;
    }
    trace ("srv=$srv ho=$ORACLE_HOME");
    my $SRVBIN = catfile ($ORACLE_HOME, "bin", $srv);
    my $status = system ("$SRVBIN install");

    # XXX: ignore error in Windows dev env as mktwork would have already
    # registered this service
    if ((0 != $status) && !is_dev_env ()) {
        return FAILED;
    }

    return SUCCESS;

}

####---------------------------------------------------------
#### Function for unregistering daemon/service
# ARGS: 1
# ARG1: daemon to be unregistered
sub s_unregister_service
{
    my $srv = $_[0];
    my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
    my $SRVBIN = catfile ($ORACLE_HOME, "bin", $srv);
    my $status = system ("$SRVBIN remove");
    if (0 != $status) {
        return FAILED;
    }

    return SUCCESS;
}

####---------------------------------------------------------
#### Function for starting daemon/service
# ARGS: 3
# ARG1: daemon to be started
# ARG2: user under whom daemon/service needs to be started
# ARG3: crs home where the service should be started
sub s_start_service
{
    my $srv  = $_[0];
    my $user = $_[1]; # this arg is ignored on NT
    my $crsHome = $_[2];

    if (! defined $crsHome)
    {
       $crsHome = $CFG->params('ORACLE_HOME');
    }
    my $SRVBIN = catfile ($crsHome, "bin", $srv);
    my $status = system ("$SRVBIN start");

    if (0 == $status) {
        trace ("$srv is starting");
        print  "$srv is starting\n";
    } else {
        trace("failed path = $SRVBIN");
        error ("$srv failed to start");
        return FAILED;
    }

    return SUCCESS;
}

####---------------------------------------------------------
#### Function for stopping daemon/service
# ARGS: 2
# ARG1: daemon to be stopped
# ARG2: user under whom daemon/service needs to be stopped
sub s_stop_service
{
    my $srv = $_[0];
    my $user = $_[1]; # this arg is ignored on NT
    my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
    my $SRVBIN = catfile ($ORACLE_HOME, "bin", $srv);
    my $status = system ("$SRVBIN stop");
    if (0 != $status) {
        return FAILED;
    }

    return SUCCESS;
}

####---------------------------------------------------------
#### Function for checking daemon (OSD actions)
# ARGS: 2
# ARG1: daemon to be checked
# ARG2: is daemon running?
sub s_check_service
{
   # no-op on NT; generic actions in check_service() are sufficient
   return SUCCESS;
}

####---------------------------------------------------------
#### Function for initializing SCR settings
# ARGS: 0
sub s_init_scr
{
   # this function is a no-op on NT
   return SUCCESS;
}

####---------------------------------------------------------
#### Function for running a command as given user
# ARGS: 2
# ARG1: cmd to be executed
# ARG2: user name
sub s_run_as_user
{
    my $cmd  = $_[0];
    # ARG2 is ignored on NT; command is always run as current user

    trace ("s_run_as_user: Running $cmd");
    return system ($cmd);
}

####---------------------------------------------------------
#### Function for running a command as given user, returning back 
#### stdout/stderra output
# ARGS: 3
# ARG1: ref to cmdlist argv list to be executed
# ARG2: user name, can be undef
# ARG3: ref to resulting array of stderr/out, can be undef
sub s_run_as_user2
{
    my $cmdlistref = $_[0];
    my @cmdlist = @{$cmdlistref};
    my $usr = $_[1]; # ARG2 is ignored on NT; 
                     # command is always run as current user
    my $capoutref = $_[2];
    my $rc = -1;
    my $cmd = join( ' ', @cmdlist );

    # capture stdout/stderr, if requested
    if (defined($capoutref))
    {
      @{$capoutref} = ();

      trace ("s_run_as_user2: Running $cmd");

      # system() with stdout/stderr capture. 
      # Note that this is a portable notation in perl
      # see http://perldoc.perl.org/perlfaq8.html
      open (CMDOUT, "$cmd 2>&1 |" );
      @{$capoutref} = <CMDOUT>;
      close (CMDOUT); # to get $?
      $rc = $?;

      if ($rc == 0) {
          trace ("$cmdlist[0] successfully executed\n");
      }
      elsif ($rc == -1) {
          trace ("$cmdlist[0] failed to execute\n");
      }
      elsif ($rc & 127) {
          trace ("$cmdlist[0]  died with signal %d, %s coredump\n",
              ($rc & 127),  ($rc & 128) ? 'with' : 'without');
      }
      else {
          $rc = $rc >> 8;
          trace ("$cmdlist[0] exited with $rc\n");
      }
    }
    else  # regular system() call
    {
      $rc = s_run_as_user( $cmd, $usr );
    }

    return $rc;
}

####---------------------------------------------------------
#### Function for running a command as given user and inject
#### one value into stdin
# ARGS: 3
# ARG1: user name
# ARG2: cmd to be executed
# ARG3: value to be injected into stdin
sub s_run_as_user3
{
   # read parameters
   # ARG1 is ignored on NT; command is always run as current user
   my $user  = $_[0];
   my $cmd   = $_[1];
   my $param = $_[2];
   
   # create the temporary file and leave it closed.
   my (undef, $cmdout) = tempfile("oracleXXXXX", OPEN => 0, TMPDIR => 1);
   
   # create final command
   my $cmd2 = join (" ", "|", @{$cmd}, ">>$cmdout");
   trace ("s_run_as_user3  Invoking \"$cmd2\"");
   
   # execute the command
   open(COMMAND, $cmd2);
   
   # inject the parameter
   print COMMAND $param;
   
   # close the stream
   close COMMAND;
   my $rc = $?;
   
   # read output and delete temp file
   open(COMMAND_OUT, "$cmdout");
   my @out = (<COMMAND_OUT>);
   close COMMAND_OUT;
   
   # remove the file when $cmdout is out of scope
   s_remove_file("$cmdout");
   
   # join result of command and output
   return ($rc,@out);
}

####---------------------------------------------------------
#### Function for getting value corresponding to a key in ocr.loc or olr.loc
# ARGS: 2
# ARG1: ocr/olr
# ARG2: key
sub s_get_config_key
{
   my $src   = $_[0];
   my $key   = $_[1];
   my $OCRLOC = $CFG->params('OCRLOC');
   my $OLRLOC = $CFG->params('OLRLOC');
   my $SRVLOC = $CFG->params('SRVLOC');
   my $value = "";
   $src      =~ tr/a-z/A-Z/;
   my $reg;

   if ($src eq 'OCR') {
      $reg = $OCRLOC;
   }
   elsif ($src eq 'OLR') {
      $reg = $OLRLOC;
   }
   elsif ($src eq 'SRV') {
      $reg = $SRVLOC;
   }

   $value = $Registry->{"LMachine/$reg//$key"};
   trace("The value for registry key [LMachine/$reg//$key] is [$value]"); 

   return $value;
}

####---------------------------------------------------------
#### Function for getting platform family
# ARGS: 0
sub s_get_platform_family
{
    return "windows";
}

####---------------------------------------------------------
#### Function for checking if a path is a link, and if so, return the target
#### path
#### Note: this function is applicable only to Oracle dev env, where a symlink
#### driver is used.  This will not be applicable in production env, and
#### s_isLink() will always return "" (FALSE)
# ARGS: 1
# ARG1: file/dir path
sub s_isLink
{
    my $path = $_[0];
    my $target = "";

    if (!is_dev_env ()) {
        return $target;
    }

    # run qln and get its output into a string
    open (LNKDRV, "qln $path |") or return "";
    my $op = join ("", <LNKDRV>);
    close (LNKDRV);

    # if qln returns a target path for $path, populate $target
    if ($op && ($op =~ m/->/)) {
        my $key;
        my $ptr;
        ($key, $ptr, $target,) = split (/ /, $op);
    }

    return $target;
}

####--------------------------------
#### Function for redirecting output
# ARGS: 1
# ARG1: file to redirect to
sub s_redirect_souterr
{
    # redirect STDOUT/STDERR to a file
    open(SAVEOUT, ">&STDOUT");
    open(SAVEERR, ">&STDERR");

    open(STDOUT, ">$_[0]") or die "Can't redirect stdout";
    open(STDERR, ">&STDOUT") or die "Can't dup stdout";

    select(STDOUT); $| = 1;  # unbuffer
    select(STDERR); $| = 1;  # unbuffer
}


####---------------------------------------------------------
#### Function for restoring output
# ARGS: 0
sub s_restore_souterr
{
    # restore STDOUT/STDERR
    close(STDOUT);
    close(STDERR);

    open(STDOUT, ">&SAVEOUT");
    open(STDERR, ">&SAVEERR");
}

sub s_getOldCrsHome
#-------------------------------------------------------------------------------
# Function: Get old crshome 
# Args    : none
# Return  : old crshome
#-------------------------------------------------------------------------------
{
   my $oldCRSHome = $CFG->params('OLD_CRS_HOME');

   return $oldCRSHome;
}

####---------------------------------------------------------
##### Function for getting the CRS Home of remote node
## ARGS : [0] remote node name
sub s_getRemoteCrsHome
{
  my $remoteNode = $_[0];
  my $remoteCrsHome;
  my $key = "crs_home";
  
  trace("get the CRS home from node $remoteNode");
  my $OLRLOC = $CFG->params('OLRLOC');
  trace("The CFG parameter of OLRLOC is $OLRLOC.");
  my $remoteCrsHome = $Registry->Connect("\\\\$remoteNode", 
                    "LMachine/$OLRLOC")->{$key};
  if ($remoteCrsHome)
  {
    trace("The CRS home of node: $remoteNode is $remoteCrsHome.");
    return $remoteCrsHome;
  }
  else 
  {
    # failed to connect to remote registry or registry empty
    trace("Failed to get $key from remote registry.");
    die(dieformat(536, $remoteNode));
  }
}

####---------------------------------------------------------
#### Function for stopping the services from OldCrsHome
# ARGS:  1

sub s_stop_OldCrsStack
{
  my $oldCrsHome = $_[0];
  my $crsctl     = catfile($oldCrsHome, "bin", "crsctl");
  my @cmd	 = ($crsctl, 'stop', 'crs');
  
  my $rc = system(@cmd);

  return $rc;
}

sub s_checkOracleCM 
#----------------------------------------------------------------
# Function: Check for OracleCMService 
# Args    : none
# Return  : TRUE - if found
#----------------------------------------------------------------
{
   my $svcName = "OracleCMService";
   my %status;

   Win32::Service::GetStatus("",$svcName,\%status);
   my $service_status = $status{CurrentState};

   if ($service_status == 4) {
      # 4 means service is running
      return TRUE;
   }
   else {
      return FALSE;
   }
}

sub s_configureAllRemoteNodes
#---------------------------------------------------------------------
# Function: Automatically execute rootcrs.pl on all remote nodes
#           by calling 'crssetup install'
# Args    : 0 
#---------------------------------------------------------------------
{
   trace ("call 'crssetup install' to configure all remote nodes");
   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my $crssetup = catfile ($ORACLE_HOME, "bin", "crssetup.exe");
   my @remote_param = ('-remotenode'); 
   if ($CFG->UPGRADE) {
      push @remote_param, '-upgrade';
   }

   my @cmd = ("$crssetup", "install", "-remoteParams", "\"@remote_param\""); 
   trace ("cmd=@cmd");
   system(@cmd);
   my $rc = $? >> 8;
   trace("rc from crssetup=$rc ");

   if ($rc == 0) {
      return SUCCESS;
   } 
   else {
      return FAILED;
   }
}

sub s_remove_file
#-------------------------------------------------------------------------------
# Function: Remove file on Windows
# Args    : File
#-------------------------------------------------------------------------------
{
   my $file = $_[0];
   my $status = SUCCESS;


   if (-e $file) {
      trace("Removing file: $file");
      if (0 == DeleteFile($file)) {
        trace("Failed to remove file: $file");
        $status = FAILED;
      } else {
        trace("Successfully removed file: $file");
      }
   }
   return $status;
}


sub s_ResetVotedisks
#-------------------------------------------------------------------------------
# Function: Reset voting disks
# Args    : [0] list of voting disks
#-------------------------------------------------------------------------------
{
   my @votedisk_list = @_;
   my $vdisk;

   trace ("Reset voting disks:@votedisk_list");
   trace ("CRS_STORAGE_OPTION:" . 
           $CFG->params('CRS_STORAGE_OPTION'));

   if ($CFG->params('CRS_STORAGE_OPTION') != 1) {
      foreach $vdisk (@votedisk_list) {
         trace("Removing voting disk: $vdisk");
         s_remove_file($vdisk);
      }
   }
}

sub s_CleanTempFiles
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_createConfigEnvFile
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_isRAC_appropriate
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_removeCvuRpm
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_is92ConfigExists
#-------------------------------------------------------------------------------
# Function: Check if Oracle configuration exists in 9.2
# Args    : none
# Returns : TRUE  if     exists
#           FALSE if not exists
#-------------------------------------------------------------------------------
{
   trace ("Checking repository used for 9i installations");

   # Check if osd9i key exists
   if ($Registry->{"LMachine/Software/oracle/osd9i/"}) {
      trace ("Oracle 92 configuration and SKGXN library exists");
      return TRUE;
   }

   trace ("Oracle 92 configuration and SKGXN library not exists");
   return FALSE;
}

sub s_RemoveInitResources
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_houseCleaning
{
   s_remove_arp_service();

   s_clsecho_uninstall();

   s_removeFenceServ();

   s_deltService("Oracle Object Service");

   s_deltService("OracleOHService");

   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my ($key, $key_param);
   if (! $CFG->SIHA)
   {
     s_delete_Oracle_Services("OracleAPXService\\+APX", $ORACLE_HOME);
   }
   s_delete_Oracle_Services("OracleASMService\\+ASM", $ORACLE_HOME);
}

sub s_deltRegKey
#-------------------------------------------------------------------------------
# Function: Delete registry key and its subkeys
# Args    : 1 (key) 
#-------------------------------------------------------------------------------
{
   my $key      = $_[0];
   my $checkkey = $Registry->Open("$key") || return SUCCESS;

   my @subkeys = $Registry->{"$key"}->SubKeyNames;
   my ($delt, $subkey);

   # delete all subkeys 
   # Fixme: this function should be recursively delete
   foreach $subkey (@subkeys) {
      print("delete subkey=$subkey\n");
      trace("delete subkey=$subkey");
      $delt = delete $Registry->{"$key/$subkey/"};
   }

   # delete key 
   $delt = delete $Registry->{"$key/"};
}

sub s_getAbsLink
{
   # this function is a no-op on NT
   return SUCCESS;
}

sub s_removeSCR
{
   my $key   = "SCR";
   my $value = delete $Registry->{"LMachine/Software/oracle/$key/"};
}

sub s_clsecho_uninstall
#-------------------------------------------------------------------------------
# Function: Call "clsecho -uninstall"
# Args    : 1 
#-------------------------------------------------------------------------------
{
   if ($CFG->platform_family eq "windows")
   {
     my $ORA_CRS_HOME = $CFG->ORA_CRS_HOME;
     my $clsecho    = catfile ($ORA_CRS_HOME, "bin", "clsecho.exe");

     my @cmd = ($clsecho, '-uninstall');
     system_cmd(@cmd);
   }

    return SUCCESS;
}

sub s_removeFenceServ
#-------------------------------------------------------------------------------
# Function: remove oracle fence service
# Args    : 0 
#-------------------------------------------------------------------------------
{
   trace("Stopping Oracle Fence Service ...");
   s_stopService("OraFenceService");

   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my $crssetup = catfile($ORACLE_HOME, 'bin', 'crssetup.exe');
   my $cmd	= "$crssetup deinstallfence";
   trace ("Remove Oracle Fence Service... $cmd");
   print ("Remove Oracle Fence Service... $cmd\n");

   my $status = system($cmd);
   if ($status == 0) {
      trace ("Remove Oracle Fence Service successfully");
   }
   else {
      error ("Remove Oracle Fence Service failed");
      return FAILED;
   }

   return SUCCESS;
}

sub s_getAuthorizedOwner
#-------------------------------------------------------------------------------
# Function: Get authorized owner ("NT AUTHORITY\SYSTEM")
# Args    : none
#-------------------------------------------------------------------------------
{
   # ohasd.exe should be owned by 'NT AUTHORITY\SYSTEM'
   # therefore get its file permission

   my ($owner, $dummy);

   if (is_dev_env ()) {
      $owner = "NT AUTHORITY\\SYSTEM";
      return $owner;
   }

   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my $ohasd_file = catfile($ORACLE_HOME, 'bin', 'ohasd.exe');

   my @out = system_cmd_capture('cacls.exe', "$ohasd_file");
   my $rc  = shift @out;
   trace ("output from cacls=@out");

   if ($rc == 0) {
      my @grep_out = grep(/\SYSTEM:/, @out); # grep for SYSTEM
      if (scalar(@grep_out) > 0) {
         ($owner, $dummy) = split(/:/, trim($grep_out[0]));
      }
   }

   trace ("owner from cacls=$owner");

   if (! $owner) {
      die ("Unable to get authorized owner");
   }

   return $owner;
}

sub s_copyRegKey
{
   my $from_key = $_[0];
   my $to_key   = $_[1];


   trace ("copy registry key from=$from_key to=$to_key");
   $Registry->{"LMachine/$to_key/"} = $Registry->{"LMachine/$from_key/"};
}

sub s_stopService
#-------------------------------------------------------------------------------
# Function: Stop Windows service
# Args    : 1 - service name
#-------------------------------------------------------------------------------
{
   my $svcName     = $_[0];
   my $svc_stopped = FALSE;
   my $retries     = 5;
   my %svc_status;
   
   if (s_isServiceRunning($svcName)) { 
      # stop service
      Win32::Service::StopService("", $svcName);
 
      while ($retries && (! $svc_stopped)) {
   	 # get service status
         Win32::Service::GetStatus("", $svcName, \%svc_status);
  
         # 1 means service stopped
         if ($svc_status{"CurrentState"} == 1) {
            $svc_stopped = TRUE;
         }
         else {
            trace ("Waiting for $svcName to stop");
            sleep (60);
            $retries--;
         }
      }
   }
   else {
      $svc_stopped = TRUE;
   }
   
   if ($svc_stopped) {
      trace ("stop of $svcName ... success");
   }
   else {
      trace ("stop of $svcName ... failed");
   }

   return $svc_stopped;
}

sub s_setPermsASMDisks 
{
 
   trace("executing asmtool to retrieve ASM links");
   my $asmtool;
   my $oradim;
   my $pathprefix = "\\\\\.\\";
   my $devpath ;

   my $success     = TRUE;

   if(is_dev_env())
   {
      $asmtool = catfile($CFG->ORA_CRS_HOME,"rdbms","bin", "asmtool.exe");
      $oradim  = catfile($CFG->ORA_CRS_HOME,"rdbms","bin", "oradim.exe");
   }
   else
   {
      $asmtool = catfile($CFG->ORA_CRS_HOME,"bin", "asmtool.exe");
      $oradim  = catfile($CFG->ORA_CRS_HOME,"bin", "oradim.exe");
   }

   my @cmd1 = ($asmtool, "-list");
   my @capout1;
   my @cmd2;
   my @capout2;

   my $rc = s_run_as_user2(\@cmd1,$CFG->params('ORACLE_OWNER'), \@capout1);

   if ($rc != 0)
   {
      $success = FALSE;
   }

   trace("executed @cmd1 with results... @capout1");

   if ($success)
   {
      foreach(@capout1)
      {
         chomp($_);
         my @res=split(/\s+/);
         trace("Device name is $res[0]\n");
         if ($res[0] =~ /^ORCL/i)
         {
            $devpath = "$pathprefix" . "$res[0]";
            @cmd2 = ($oradim,"-ACL","-setperm","RawDevice","-USER","raw","-OBJTYPE","RawDevice","-OBJPATH",$devpath,"-RECURSE","false");
            
            $rc = s_run_as_user2(\@cmd2,$CFG->params('ORACLE_OWNER'), \@capout2);

            trace("@cmd2 ... result: @capout2");
         }
      }
   }
}

sub s_stopDeltOldASM
{
   my $asm_service = "OracleASMService\+ASM";
   my $success     = TRUE;

   if (s_isServiceExists($asm_service)) {

      my @cmd = ('net', 'stop', $asm_service, '/Y');
      my @out = system_cmd_capture(@cmd);
      trace("out=@out");
      my $rc  = shift @out;

      if ($rc == 0) {
	 trace ("@cmd ... success");
      }
      else {
        trace ("@cmd ... failed with rc=", $rc >> 8);
        $success = FALSE;
      }

      if ($success) {
	 # delete ASM dependencies
	 @cmd = ('sc', 'config', $asm_service, 'depend=', "\"\"");
         @out = system_cmd_capture(@cmd);
         trace("out=@out");
         $rc  = shift @out;

	 if ($rc == 0) {
	    trace ("@cmd ... success");
	 }
	 else {
	    trace ("@cmd ... failed with rc=", $rc >> 8);
            $success = FALSE;
	 }
      }
   }

   if (! $success) {
      die ("Unable to stop $asm_service and its dependencies");
   }


   return $success;
}

sub s_get_olr_file
{
   my $key = $_[0];

   return s_get_config_key("OLR", $key);
}

sub s_getDomainName
{

   return Win32::DomainName();
}

sub s_deltOldServ
{
   my $rc = s_deltService("OracleCRService");

   if ($rc) {
      s_deltService("OracleCSService");
   }

   if ($rc) {
      s_deltService("OracleEVMService");
   }

   return $rc;
}

sub s_resetParentDirOwner
{
   # this function is a no-op on NT
   return SUCCESS;
}

sub s_is_HAIP_supported
{
  return FALSE;
}

sub s_CheckNetworkConfig
{
   # this function is a no-op on NT
  return;
}

sub s_upgrade_services
{
   s_remove_arp_service();
   s_removeFenceServ();
   s_deltService("Oracle Object Service");

   # Create & start Oracle Fence Service 
   trace("Create and start Oracle Fence Service");
   s_create_start_FenceServ();
}

sub s_getSystem
{
   my $param_file = $CFG->paramfile;

   open (PARAM, $param_file);
   my @list = grep(/ORACLE_HOME=/, <PARAM>);
   close (PARAM);

   my ($dummy, $crshome) = split (/=/, $list[0]);
   chomp $crshome;
   my $crssetup = catfile ($crshome, 'bin', 'crssetup.exe');
   my @cmd = ($crssetup, 'getsystem');
   trace("@cmd");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;

   if (0 != $rc) {
      die ("Get SYSTEM failed with rc=", $rc >> 8);
      return FAILED;
   }

   return $out[0];
}

# HAIP is not used on Windows sto stubbed out
sub s_is_HAIP_NonFatal
{
  return FALSE;
}

sub s_NSCleanUp
{
  return FALSE;
}

sub s_removeGPnPprofile
{
  return;
}

sub s_is_Exadata
{
  return FALSE;
}

sub s_install_initd
{
    # this function is a no-op on NT
    return SUCCESS;
}

sub s_restoreInitScripts
{
  # This sub is a no-op on Windows
  return SUCCESS;
}

sub s_restoreASMFiles
{
  # This sub is a no-op on Windows
  return SUCCESS;
}

sub s_copy_afdinit_init
{
  #no-op
  return SUCCESS;
}

sub s_rm_afdinit_init
{
  #no-op	
  return SUCCESS;
}	

sub s_rm_afdinit_rclevel
{
  #no-op	
  return SUCCESS;
}

#-------------------------------------------------------------------------------
# Function:  Returns the path to the qosctl script
# Args    :  none
# Returns :  Path to the qosctl script
#-------------------------------------------------------------------------------
sub s_get_qosctl_path {
  my $execPath = catfile( $CFG->ORA_CRS_HOME, 'bin', 'qosctl.bat' );
  if ( !( -x "${execPath}" ) ) {
    trace("The file ${execPath} either does not exist or is not executable");
  }
  return $execPath;
}

sub s_verifyClonedHome
{
    my $oldOHome = $_[0];
    my $newOHome = $_[1];
    
    my $oldUserLoc = "LMachine/SOFTWARE/ORACLE/KEY_$oldOHome/ORACLE_SVCUSER";
    my $newUserLoc = "LMachine/SOFTWARE/ORACLE/KEY_$newOHome/ORACLE_SVCUSER";

    my $oldOHomeUser = $Registry->{$oldUserLoc}; # Get the old oracle home user
    my $newOHomeUser = $Registry->{$newUserLoc}; # Get the new oracle home user

    if (lc $oldOHomeUser ne lc $newOHomeUser)
    {
        trace("The target Oracle Home has a different user than the original home,"
            ." unsupported use case.");
        die(dieformat(47, $oldOHomeUser, $newOHomeUser));
    }
}

1;
